/**
 * ipmi_sdr.h
 *
 * IPMI spec defined sensor data records
 *
 * (c) 2004 Peppercon AG, 12/15/2004, tbr@peppecon.de
 */

#ifndef __IPMI_SDR_H__
#define __IPMI_SDR_H__

#include <pp/bmc/ipmi_bits.h>
#include <pp/bmc/utils.h>

#define IPMI_SDR_ACCESS_NOT_SUPP    0
#define IPMI_SDR_ACCESS_READ_ONLY   1
#define IPMI_SDR_ACCESS_READ_WRITE  2
#define IPMI_SDR_ACCESS_FIXED       3

/**
 * SDR header, used for all possible SDRs
 */
struct ipmi_sdr_header_s {
    unsigned short	id_le16;         /* record ID */
    unsigned char	version;	/* SDR version (51h) */
    unsigned char	type;		/* record type */
    unsigned char	length;		/* remaining record bytes */
    unsigned char       data[0];
} __attribute__ ((packed));
typedef struct ipmi_sdr_header_s ipmi_sdr_header_t;

/**
 * SDR key for sensors, used in all sensor SDRs
 */
struct ipmi_sdr_key_sensor_s {
    unsigned char	owner_id;
    BITFIELD3(unsigned char,
	      lun      : 2,	         /* sensor owner lun */
	      _resv    : 2,
	      channel  : 4 );	         /* channel number */
    unsigned char sensor_num;            /* unique sensor number */
} __attribute__ ((packed));
typedef struct ipmi_sdr_key_sensor_s ipmi_sdr_key_sensor_t;

/**
 * SDR entity identifier
 */
struct ipmi_sdr_entity_s {
    unsigned char	id;              /* physical entity id */
    BITFIELD2(unsigned char,
	      instance   : 7,            /* instance number */
	      is_logical : 1 );          /* physical/logical */
} __attribute__ ((packed));
typedef struct ipmi_sdr_entity_s ipmi_sdr_entity_t;

/**
 * SDR sensor id string
 */

#define IPMI_SENSOR_ID_STRING_LEN 16

struct ipmi_sdr_id_s {
    BITFIELD3(unsigned char, 
	      length: 5,     /* sensor ID string length */
	      _resv:  1,     
	      code:   2 );   /* sensor ID string encoding */
    unsigned char string[IPMI_SENSOR_ID_STRING_LEN];
} __attribute__ ((packed));
typedef struct ipmi_sdr_id_s ipmi_sdr_id_t;

/**
 * SDR basic sensor description
 */
struct ipmi_sdr_sensor_s {
    struct {
	BITFIELD8(unsigned char,
		  sensor_scan  : 1,
		  event_gen    : 1,
		  type         : 1,
		  hysteresis   : 1,
		  thresholds   : 1,
		  events       : 1,
		  scanning     : 1,
		  __reserved   : 1 );
    } __attribute__ ((packed)) init;
	
    struct {
	BITFIELD5(unsigned char,
		  event_msg    : 2,
		  threshold    : 2, // IPMI_SDR_ACCESS_xxxx
		  hysteresis   : 2, // IPMI_SDR_ACCESS_xxxx
		  rearm        : 1,
		  ignore       : 1 );
    } __attribute__ ((packed)) caps;
    
    unsigned char type;
    unsigned char event_type;               /* event/reading type code */
} __attribute__ ((packed));
typedef struct ipmi_sdr_sensor_s ipmi_sdr_sensor_t;

/**
 * SDR sensor threshold settings
 */
struct ipmi_sdr_threshold_s {
    struct {
	unsigned char non_recover;
	unsigned char critical;
	unsigned char non_critical;
    } __attribute__ ((packed)) upper;
    
    struct {
	unsigned char non_recover;
	unsigned char critical;
	unsigned char non_critical;
    } __attribute__ ((packed)) lower;
    
    struct {
	unsigned char positive;
	unsigned char negative;
    } __attribute__ ((packed)) hysteresis;
} __attribute__ ((packed));
typedef struct ipmi_sdr_threshold_s ipmi_sdr_threshold_t;

/**
 * SDR sensor event mask, for threshold based sensors and discrete sensors
 */
union ipmi_sdr_evt_mask_e {
    struct {
	unsigned short	assert_event_le16;   /* assertion event mask */
	unsigned short	deassert_event_le16; /* de-assertion event mask */
	unsigned short	read_le16;	     /* discrete reaading mask */
    } __attribute__ ((packed)) discrete;
	
    struct {
        unsigned short  assert_event_le16;   /* assertion event mask */
        unsigned short  deassert_event_le16; /* de-assertion event mask */
	unsigned char	set;	/* settable threshold mask */
	unsigned char	read;	/* readable threshold mask */
    } __attribute__ ((packed)) threshold;
}  __attribute__ ((packed));
typedef union ipmi_sdr_evt_mask_e ipmi_sdr_evt_mask_t;

/**
 * SDR Full Sensor
 */
struct ipmi_sdr_full_sensor_s {
    ipmi_sdr_header_t     header;
    ipmi_sdr_key_sensor_t key;
    ipmi_sdr_entity_t     entity;
    ipmi_sdr_sensor_t     sensor;
    ipmi_sdr_evt_mask_t   mask;                   /* masks for disc/thresh */
    
    struct {
	BITFIELD4(unsigned char,
		  pct           : 1,
		  modifier      : 2,
		  rate          : 3,
		  analog        : 2 );
	struct {
	    unsigned char	base;
	    unsigned char	modifier;
	} __attribute__ ((packed)) type;
    } __attribute__ ((packed)) unit;

    unsigned char  linearization; /* 70h=non linear, 71h-7Fh=non linear, OEM */
    unsigned short mtol;	  /* M & tolerance */
    unsigned short bacc;          /* B & accuracy */
    BITFIELD3(unsigned char,
	      direction : 2,      /* sensor direction       */
	      accexp    : 2,      /* accuracy exp, unsigned */
	      acc       : 4 );    /* accuracy, MS 4 bit     */
    BITFIELD2(char,
	      bexp      : 4,      /* b exponent, 2'compl, signed */
	      rexp      : 4 );    /* result exponent, 2'compl, signed */
    
    struct {
	BITFIELD4(unsigned char,
		  nominal_read  : 1,	/* nominal reading field specified */
		  normal_max    : 1,	/* normal max field specified */
		  normal_min    : 1,	/* normal min field specified */
		  __reserved    : 5 );
    } __attribute__ ((packed)) analog_flag;

    unsigned char nominal_read;         /* nominal reading, raw value */
    unsigned char normal_max;           /* normal maximum, raw value */
    unsigned char normal_min;           /* normal minimum, raw value */
    unsigned char sensor_max;           /* sensor maximum, raw value */
    unsigned char sensor_min;           /* sensor minimum, raw value */

    ipmi_sdr_threshold_t threshold;     /* thresholds and hysteresis */
    
    unsigned char __reserved[2];
    unsigned char oem;		/* reserved for OEM use */

    ipmi_sdr_id_t id;           /* sensor ID string */
    
} __attribute__ ((packed));
typedef struct ipmi_sdr_full_sensor_s ipmi_sdr_full_sensor_t;

/**
 * SDR Compact Sensor
 */
struct ipmi_sdr_compact_sensor_s {
    ipmi_sdr_header_t     header;
    ipmi_sdr_key_sensor_t key;
    ipmi_sdr_entity_t     entity;
    ipmi_sdr_sensor_t     sensor;
    ipmi_sdr_evt_mask_t   mask;

    struct {
	BITFIELD4(unsigned char,
		  pct           : 1,
		  modifier      : 2,
		  rate          : 3,
		  analog        : 2 );
	struct {
	    unsigned char	base;
	    unsigned char	modifier;
	} __attribute__ ((packed)) type;
    } __attribute__ ((packed)) unit;

    struct {
	BITFIELD3(unsigned char,
		  count       : 4,
		  mod_type    : 2,
		  direction   : 2 );
	BITFIELD2(unsigned char,
		  mod_offset  : 7,
		  entity_inst : 1 );
    } __attribute__ ((packed)) share;

    struct {
	struct {
	    unsigned char	positive;
	    unsigned char	negative;
	} __attribute__ ((packed)) hysteresis;
    } __attribute__ ((packed)) threshold;

    unsigned char __reserved[3];
    unsigned char oem;		/* reserved for OEM use */
    ipmi_sdr_id_t id;           /* sensor ID string */
} __attribute__ ((packed));
typedef struct ipmi_sdr_compact_sensor_s ipmi_sdr_compact_sensor_t;

/**
 * SDR event only Sensor
 */
struct ipmi_sdr_eventonly_sensor_s {
    ipmi_sdr_header_t     header;
    ipmi_sdr_key_sensor_t key;
    ipmi_sdr_entity_t     entity;
    
    unsigned char sensor_type;  /* sensor type */
    unsigned char event_type;   /* event/reading type code */
    
    struct {
	BITFIELD3(unsigned char,
		  count       : 4,
		  mod_type    : 2,
		  sens_dir    : 2 );
	BITFIELD2(unsigned char,
		  mod_offset  : 7,
		  entity_inst : 1 );
    } __attribute__ ((packed)) share;

    unsigned char __reserved;
    unsigned char oem;	        /* reserved for OEM use */
    ipmi_sdr_id_t id;           /* sensor ID string */
} __attribute__ ((packed));
typedef struct ipmi_sdr_eventonly_sensor_s ipmi_sdr_eventonly_sensor_t;


/**
 * SDR FRU device locator
 */
struct ipmi_sdr_fru_locator_s {
    ipmi_sdr_header_t     header;
    struct {
	BITFIELD2(unsigned char,
		  __reserved1     : 1, 
		  dev_access_addr : 7 );
	unsigned char fru_device_id;
	BITFIELD4(unsigned char,
		  private_bus	: 3,
		  access_lun	: 2,
		  __reserved2	: 2,
		  logical_dev	: 1 );
	BITFIELD2(unsigned char,
		  __reserved3	: 4,
		  channel_num	: 4 );
    } __attribute__ ((packed)) key;
    
    unsigned char __reserved4;
    unsigned char device_type;
    unsigned char device_type_modifier;
    struct {
	unsigned char id;
	unsigned char instance;
    } __attribute__ ((packed)) entity;
    unsigned char oem;
    ipmi_sdr_id_t id;           /* sensor ID string */
} __attribute__ ((packed));
typedef struct ipmi_sdr_fru_locator_s ipmi_sdr_fru_locator_t;

/**
 * SDR MC locator
 */
struct ipmi_sdr_mc_locator_s {
    ipmi_sdr_header_t     header;
    BITFIELD2(unsigned char,
	      __reserved1	: 1,
	      dev_slave_addr	: 7 );
    BITFIELD2(unsigned char,
	      channel_num 	: 4,
	      __reserved2	: 4 );
    BITFIELD3(unsigned char,
	      global_init	: 4,
	      __reserved3	: 1,
	      pwr_state_notif	: 3 );
    unsigned char dev_support;
    unsigned char __reserved4[3];
    struct {
	unsigned char id;
	unsigned char instance;
    } __attribute__ ((packed)) entity;
    unsigned char oem;
    ipmi_sdr_id_t id;           /* sensor ID string */
} __attribute__ ((packed));
typedef struct ipmi_sdr_mc_locator_s ipmi_sdr_mc_locator_t;

/*
 * structure to handle sdr init values from topo file
 */
typedef struct {
    unsigned char  num;			/* sensor number */ 
    unsigned char  eid;			/* entity id */
    unsigned char  ein;			/* entity instance */
    unsigned char  cap;			/* sensor capabilities */
    unsigned char  sty;			/* sensor type */
    unsigned char  erc;			/* event/reading type code */
    union {
	struct {
	    unsigned short ltm;		/* lower threshold/assertion event mask */
	    unsigned short utm;		/* upper threshold/deassertion event mask */
	    unsigned char  stm;		/* setable threshold mask */
	    unsigned char  rtm;		/* readable threshold mask */
	} threshold;
	struct {
	    unsigned short eam;		/* assertion event mask */
	    unsigned short edm;		/* de-assertion event mask */
	    unsigned short rdm;		/* discrete reaading mask */
	} discrete;
    } mask;    
    unsigned char  adf;			/* analog data format */
    unsigned char  run;			/* rate unit */
    unsigned char  mnc;			/* modifier unit config */
    unsigned char  pct;			/* percentage */
    unsigned char  bun;			/* base unit */
    unsigned char  mun;			/* modifier unit */
    union {
	struct {
	    unsigned char  lin;		/* linearization */
	    signed short   m;		/* M */
	    unsigned char  tol;		/* tolerance */
	    signed short   b;		/* B */
	    unsigned short acc;		/* accuracy */
	    unsigned char  aex;		/* accuracy exponent */
	    unsigned char  sed;		/* sensor direction */
	    signed char    rex;		/* R (result) exponent */
	    signed char    bex;		/* B exponent */
	    unsigned char  acf;		/* analog characteristic flags */
	    unsigned char  nrd;		/* nominal_read */
	    unsigned char  nma;		/* normal_max */
	    unsigned char  nmi;		/* normal_min */
	    unsigned char  sma;		/* sensor max reading */
	    unsigned char  smi;		/* sensor min reading */
	    unsigned char  nru;		/* non_recover upper thresh */
	    unsigned char  cru;		/* critical upper thresh */
	    unsigned char  ncu;		/* non_critical upper thresh */
	    unsigned char  nrl;		/* non_recover lower thresh */
	    unsigned char  crl;		/* critical lower thresh */
	    unsigned char  ncl;		/* non_critical lower thresh */
	} full;
	struct {
	    unsigned char  sed;		/* sensor direction */
	    unsigned char  ist;		/* id string inst. mod. type */
	    unsigned char  shc;		/* share count */
	    unsigned char  eis;		/* entity instance sharing */
	    unsigned char  iso;		/* id string inst. mod. offset */
	} compact;
    } type;
    unsigned char pos;			/* positive hysteresis */
    unsigned char neg;			/* negative hysteresis */
    unsigned char oem;			/* for oem use */
    char*         ids;			/* ID string/name */
} pp_tp_sdr_init_values_t;

/*
 * Utility functions
 */
ipmi_sdr_full_sensor_t* ipmi_sdr_full_create(pp_tp_sdr_init_values_t *sv);
ipmi_sdr_compact_sensor_t* ipmi_sdr_compact_create(pp_tp_sdr_init_values_t *sv);

ipmi_sdr_full_sensor_t* ipmi_sdr_full_part_create(unsigned char sens_type,
					   unsigned char reading_type,
					   unsigned char base_unit,
					   unsigned char analog_format,
					   unsigned short mtol,
					   unsigned short bacc,
					   char bexp,
					   char rexp,
					   unsigned char sens_max,
					   unsigned char sens_min);

ipmi_sdr_compact_sensor_t* ipmi_sdr_compact_part_create(unsigned char sens_type,
						 unsigned char reading_type);
ipmi_sdr_header_t* ipmi_sdr_copy(ipmi_sdr_header_t* this);
int ipmi_sdr_fill_sdr(ipmi_sdr_header_t* this, pp_tp_sdr_init_values_t *sv);
ipmi_sdr_id_t* ipmi_sdr_get_id_ptr(ipmi_sdr_header_t* sdr);
ipmi_sdr_key_sensor_t* ipmi_sdr_get_key_sensor_ptr(ipmi_sdr_header_t* sdr);
ipmi_sdr_entity_t* ipmi_sdr_get_entity_ptr(ipmi_sdr_header_t* sdr);
unsigned char* ipmi_sdr_get_sensor_type_ptr(ipmi_sdr_header_t* sdr);
unsigned char* ipmi_sdr_get_event_type_ptr(ipmi_sdr_header_t* sdr);
unsigned char ipmi_sdr_get_analog_data_format(ipmi_sdr_header_t*sdr);
ipmi_sdr_sensor_t* ipmi_sdr_get_sensor_ptr(ipmi_sdr_header_t* sdr);
ipmi_sdr_evt_mask_t* ipmi_sdr_get_event_mask_ptr(ipmi_sdr_header_t* sdr);
ipmi_sdr_threshold_t* ipmi_sdr_get_threshold_ptr(ipmi_sdr_header_t* sdr);

int ipmi_sdr_is_discrete(ipmi_sdr_header_t* sdr);
int ipmi_sdr_event_type_is_discrete(unsigned char evt_type);

unsigned char ipmi_sdr_min_by_form(unsigned char analog_format);
unsigned char ipmi_sdr_max_by_form(unsigned char analog_format);

int32_t pp_bmc_utos(uint32_t val, int bits);

#define PP_BMC_SDR_M_TOL_2_MTOL(m, tol)	                                \
    cpu_to_be16_const((((u_short)(m) & 0x00ff) << 8)			\
		      | (((u_short)(m) & 0x8000) >> 8)			\
		      | (((u_short)(m) & 0x0100) >> 2)			\
		      | ((tol) & 0x3f))

#define PP_BMC_SDR_MTOL_2_M(mtol)	                                \
    ((short)(pp_bmc_utos((((be16_to_cpu_const(mtol) & 0xff00) >> 8)	\
			  | ((be16_to_cpu_const(mtol) & 0xc0) << 2)), 10)))

#define PP_BMC_SDR_MTOL_2_TOL(mtol)	   \
    ((u_short)be16_to_cpu_const(mtol) & 0x3f)

#define PP_BMC_SDR_B_ACCURACY_2_BACC(b, acc)	                                \
    cpu_to_be16_const((((u_short)(b) & 0x00ff) << 8)			\
		      | (((u_short)(b) & 0x8000) >> 8)			\
		      | (((u_short)(b) & 0x0100) >> 2)			\
		      | ((acc) & 0x3f))

#define PP_BMC_SDR_BACC_2_B(bacc)	                                \
    ((short)(pp_bmc_utos((((be16_to_cpu_const(bacc) & 0xff00) >> 8)	\
			  | ((be16_to_cpu_const(bacc) & 0xc0) << 2)), 10)))

#define PP_BMC_SDR_BACC_2_ACCURACY(bacc)	   \
    ((u_short)be16_to_cpu_const(bacc) & 0x3f)

#define PP_BMC_SDR_ACCMS_ACCEXP_DIR(msacc, accexp, dir) \
    ((((u_char)(msacc)  & 0xf0) << 4)  |		\
     (((u_char)(accexp) & 0x03) << 2)  |		\
     (((u_char)(dir)    & 0x03) << 0)

#endif
