/**
 * user_manager.h
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 *
 * Basic user management. Allows access to the lara UM.
 * Both empty usernames and role only logins (for IPMIv20
 * null users) are possible with this interface. Configured
 * users will be returned even if their ipmi privilege
 * level is stil unspecified.
 * 
 * Practically anonymous users are not implemented
 * yet. Setting null usernames will yield an error.
 * 
 * Note: anonymous users are not necessarily mapped to
 * IPMI user ID 1.
 */

#ifndef __USER_MANAGER_H__
#define __USER_MANAGER_H__

/**
 * Initialize the user manager
 */
int pp_bmc_user_manager_init(void);

/**
 * Shuts down the user manager
 */
void pp_bmc_user_manager_cleanup(void);

/**
 * Returns IPMI UID of the specified user name
 * (1-based, PP_ERR on error)
 */
int pp_bmc_user_uid_by_name(const char* name);

/**
 * Returns IPMI UID of the specified user name
 * with the specified privilege level. Used for
 * IPMIv2.0 role only logins only.
 * (1-based, PP_ERR on error)
 */
int pp_bmc_user_uid_by_name_privlevel(const char* name, int channel, unsigned char privilege_level);

/**
 * Counts number of IPMI-enabled users
 * (expensive)
 */
int pp_bmc_user_count_enabled(int channel);

/**
 * Returns max used UID
 * (cheap)
 */
int pp_bmc_user_get_max_uid(void);

/**
 * Count and return users with different login types. Anonymous users
 * have an empty username and empty password (can only be 0 or 1).
 * Null users have a null username, but non-null password and regular
 * users have a password.
 */
int pp_bmc_user_get_login_types(int* anonymous_user_count, int* null_user_count, int* regular_user_count);

/**
 * Returns the user's name. UID will be checked.
 * (suggested buflen: 17)
 */
int pp_bmc_user_get_name(int uid, char* name, unsigned int buflen);

/**
 * Changes the user's name. Please guarantee that
 * 'name' is a 0 terminated string. UID will be checked.
 */
int pp_bmc_user_set_name(int uid, const char* name);

/**
 * Check if user enabled for IPMI
 */
int pp_bmc_user_get_enable(int uid);

/**
 * Enable/disable this user for IPMI
 */
int pp_bmc_user_set_enable(int uid, int enable);

/**
 * Returns the user's password. uid will be checked.
 * (suggested buflen: 65)
 */
int pp_bmc_user_get_password(int uid, char* password, unsigned int buflen);

/**
 * Changes the user's password. Please guarantee that
 * 'name' is a 0 terminated string. uid will be checked.
 */
int pp_bmc_user_set_password(int uid, const char* password);

/**
 * Returns the user's internal user ID
 */
int pp_bmc_user_get_pp_uid(int uid);

/**
 * Returns the user's IPMI privilege level
 * (IPMI_PRIV_xxx, PP_ERR on error)
 */
int pp_bmc_user_get_priv_level(int uid, int channel);

/**
 * Changes the user's IPMI privilege level
 */
int pp_bmc_user_set_priv_level(int uid, int channel, int priv_level);

/**
 * Returns 1 if the user is restricted to callback, 0 otherwise.
 */
int pp_bmc_user_callback_only(int uid, int channel);

/**
 * Returns 1 if user link authentication is enabled, 0 otherwise.
 */
int pp_bmc_user_link_auth(int uid, int channel);

/**
 * Returns 1 if the IPMI messaging is enabled for this user/channel, 0 otherwise.
 */
int pp_bmc_user_msg_enabled(int uid, int channel);

/**
 * Changes the user's IPMI privilege level and access flags
 */
int pp_bmc_user_set_priv_level_and_flags(int uid, int channel,
	int priv_level, int callback_only, int link_auth, int msg_enabled);

/**
 * Returns 1 if the user is allowed to use SOL, 0 otherwise.
 */
int pp_bmc_user_may_use_sol(int uid);

/**
 * Sets whether the use is allowed to use SOL or not.
 * @returns PP_SUC/PP_ERR
 */
int pp_bmc_user_set_may_use_sol(int uid, int may_use_sol);


#endif // !__USER_MANAGER_H__
