/**
 * session_manager.c
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 * 
 * The session manager for the ipmi core.
 * 
 * Attention: if the session is deleted the session timeout must be
 * deleted as well.
 */

#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/utils.h>
#include <pp/bmc/session_manager.h>
#include <pp/bmc/user_manager.h>
#include <pp/bmc/lan_cipher_suites.h>

#include <pp/selector.h>
#include <pp/vector.h>
#include <pp/base.h>
#include <pp/cfg.h>

#include <string.h>
#include <stdlib.h>

static unsigned int sessIdCnt;
static unsigned char sessHndlCnt;
static struct list_head session_root;

static int session_inactivity_timeout(int timeout_hndl UNUSED, void* ctx) {
    imsg_session_t* session;
    unsigned int session_id;
    
    session = (imsg_session_t*)ctx;

    session_id = session->session_id;
    
    session->inactivity_timeout = -1;
    if (pp_bmc_close_session(session_id) != PP_SUC) {
        pp_bmc_log_debug("[SEM] closing inactive session %u failed", session_id);
    } else {
        pp_bmc_log_debug("[SEM] closing inactive session %u", session_id);
    }
    
    return PP_SUC;
}

imsg_session_t* pp_bmc_create_session() {
    imsg_session_t* session;
    struct list_head* act;
    int list_size;
    
    session = malloc(sizeof(imsg_session_t));
    memset(session, 0, sizeof(imsg_session_t));
    
    list_size = 0;
    act = session_root.next;
    while (act != &session_root) {
        list_size++;
        act = act->next;
    }
    if (list_size > 0x3E) {  // 0x3E because 0 is no valid session handle
        // the logical session structure is full, we cannot open a new session
        pp_bmc_log_warn("[SESS] logical session structure full - create session rejected ");
        free(session);
        return NULL;
    }
    
    // find a free handle for the session, we have less than 0x3E entries so we will find one
    do {
        sessHndlCnt++;
        if (sessHndlCnt > 0x3F) sessHndlCnt = 1;
        
        // check if this handle is in use
        act = session_root.next;
        while (act != &session_root) {
            // test this session
            if (list_entry(act, imsg_session_t, link)->session_handle == sessHndlCnt)
                break;
            act = act->next;
        }
        // this loop will exit with (act == session_root) only if all session were successfully tested
    }  while (act != &session_root);
    session->session_handle = sessHndlCnt;
    
    do {
        // create a new session_id
        sessIdCnt++;
        if (sessIdCnt == 0) sessIdCnt++;  // session 0 is not allowed
        
        // check if this sessionID is used
        act = session_root.next;
        while (act != &session_root) {
            // test this session
            if (list_entry(act, imsg_session_t, link)->session_id == sessIdCnt)
                break;
            act = act->next;
        }
        // this loop will exit with (act == session_root) only if all session were successfully tested
    }  while (act != &session_root);
    
    // we found a valid session_id
    session->session_id = sessIdCnt;

    session->chan = 0x0e; // init with PRESENT channel, this is invalid later on

    session->addr = NULL;
    
    session->ref_count = 1;
    list_add(&(session->link), &session_root);
    
    session->inactivity_timeout = pp_select_add_to(60000, 0, session_inactivity_timeout, session);
    
    return session;
}

imsg_session_t* pp_bmc_create_session15(unsigned char* username,
                                        unsigned char channel)
{
    imsg_session_t* session;
    int i;
    int uid = 0;
    int priv_lvl = IPMI_PRIV_UNSPEC;
    char name[17];
    char passwd[65];

    // make username NULL-terminated
    memcpy(name, username, 16);
    name[17] = '\0';

    uid = pp_bmc_user_uid_by_name(name);
    if (uid < 1) {
        pp_bmc_log_debug("[SEM] unknown IPMI user %s", name);
        return NULL;
    }

    if (!pp_bmc_user_get_enable(uid)) {
        pp_bmc_log_warn("[SEM] user %s is not enabled", name);
        return NULL;
    }

    priv_lvl = pp_bmc_user_get_priv_level(uid, channel);
    if (priv_lvl <= 0) {
        pp_bmc_log_warn("[SEM] user %s has invalid ipmi privilege level %d for channel %d", name, priv_lvl, channel);
        return NULL;
    }

    if (PP_ERR == pp_bmc_user_get_password(uid, passwd, sizeof(passwd))) {
        pp_bmc_log_warn("[SEM] cant read password of user %s", name);
        return NULL;
    }

    // ipmi privilege levels are assigned independent of channels

    // create new session
    session = pp_bmc_create_session();
    if (session == NULL) {
        return NULL;
    }

    /* copy password and cut to 16 chars */
    session->password = malloc(16);
    memset(session->password, 0, 16);
    strncpy(session->password, passwd, 16);
    session->password_len = 16;

    session->challenge = malloc(16);
    srand(time(NULL));
    for (i = 0; i < 16; i++)
        session->challenge[i] = rand() % 0xff;

    /* privilege levels won't be known until activation */
    session->max_priv_level = IPMI_PRIV_UNSPEC;
    session->cur_priv_level = IPMI_PRIV_UNSPEC;
    
    session->chan = channel;
    session->uid = uid & 0x1f;

    pp_bmc_log_debug("[SEM] created new 1.5 session %u", session->session_id);
    session->ref_count++;
    return session;
}

imsg_session_t* pp_bmc_create_session20(unsigned char req_priv_lvl,
                                        unsigned char auth_cipher,
                                        unsigned char integrity_cipher,
                                        unsigned char conf_cipher)
{
    imsg_session_t* session;
    int i;
    pp_cipher_suite_record_t* cur_cipher_suite;
    int priv_level;
    char* priv_level_str;
    
    int match_auth = -1;
    int match_int = -1;
    int match_conf = -1;
    int match_privlvl = -1;
    
    /*
     * iterate through implemented cipher suites, check if the algorithms
     * match the request and find either the first or the best match.
     */

    match_privlvl = IPMI_PRIV_UNSPEC;
    for (i=0; i<PP_BMC_LAN_CIPHER_SUITE_COUNT; i++) {
        cur_cipher_suite = &(pp_bmc_lan_enabled_cipher_suites[i]);
        
        if ( ((auth_cipher != IPMI20_AUTH_ANY) && (cur_cipher_suite->auth_algorithm != auth_cipher)) ||
             ((conf_cipher != IPMI20_CONF_ANY) && (cur_cipher_suite->conf_algorithm != conf_cipher)) ||
             ((integrity_cipher != IPMI20_INTEGRITY_ANY) && (cur_cipher_suite->int_algorithm  != integrity_cipher)) )
        {
            // this cipher suite does not match the user requirements
            continue;
        }

        // unconfigured ciphersuites will not be checked/chosen
        if (pp_cfg_get_nodflt(&priv_level_str, "bmc.lan.rcmpp_mcs_priv_level[%u]", cur_cipher_suite->cipher_suite_id) == PP_SUC) {
            /* this cipher suite is a possible match */
            priv_level = pp_bmc_get_privlevel_value(priv_level_str);
            free(priv_level_str);

            if ( ((req_priv_lvl == 0) && (priv_level >= match_privlvl)) ||
                 ((req_priv_lvl != 0) && (priv_level >= req_priv_lvl)) )
            {
                /* this cipher suite matches and is better than the old one  *
                 * (either supports a higher privilege level or more secure) */
                match_auth = cur_cipher_suite->auth_algorithm;
                match_int  = cur_cipher_suite->int_algorithm;
                match_conf = cur_cipher_suite->conf_algorithm;
                match_privlvl = priv_level;
                pp_bmc_log_debug("[SEM] found matching cipher suite %d",
                                 cur_cipher_suite->cipher_suite_id);
            } else {
                pp_bmc_log_warn("[SEM] found matching cipher suite %d, "
                                "but has insufficient priv lvl (is=%d expected=%d)",
                                cur_cipher_suite->cipher_suite_id, priv_level, req_priv_lvl);
            }
        }
    }
    
    if (match_privlvl == IPMI_PRIV_UNSPEC) {
        // no matching ciphersuite
        pp_bmc_log_error("[SEM] no cipher suite found or insufficient priv lvl "
                         "(auth=%d int=%d conf=%d privlvl=%d)",
                         auth_cipher, integrity_cipher, conf_cipher, req_priv_lvl);
        return NULL;
    }

    if (req_priv_lvl != 0 && match_privlvl > req_priv_lvl) {
        // we already know that (match_privlvl >= req_priv_lvl) but we don't want a higher privlevel than requested 
        match_privlvl = req_priv_lvl;
    }

    // create new session
    session = pp_bmc_create_session();
    if (session == NULL) {
        pp_bmc_log_error("[SEM] session creation failed");
        return NULL;
    }

    session->req_priv_level = match_privlvl;

    session->authentication_cipher = match_auth;
    session->integrity_cipher = match_int;
    session->confidentiality_cipher = match_conf;
    
    session->authtype = IPMI_AUTH_RMCPP;
    
    /* currently all ipmi v2.0 sessions are for lan channel */
    session->chan = IPMI_CHAN_LAN;

    pp_bmc_log_debug("[SEM] created new 2.0 session %u, ciphersuite(%d, %d, %d), priv=%d",
                     session->session_id, match_auth, match_int, match_conf, match_privlvl);
    session->ref_count++;
    return session;
}

/* second part or session activation, sets the username and the maximum privilege level */
int pp_bmc_open_session20(imsg_session_t* session, unsigned char username_len,
                          unsigned char* username, unsigned char name_only_lookup,
                          unsigned int req_max_priv)
{
    int uid = 0;
    int priv_lvl = IPMI_PRIV_UNSPEC;
    char name[17];
    char passwd[65];
    int i;

    if (session == NULL) {
        pp_bmc_log_error("[SEM] could not open empty session");
        return PP_ERR;
    }

    if (!name_only_lookup && req_max_priv > session->req_priv_level) {
        pp_bmc_log_warn("[SEM] ignored open session with different privilege level");
        return PP_ERR;
    }
    session->req_priv_level = req_max_priv;
    
    session->name_only_lookup = name_only_lookup;
    
    // make username NULL-terminated for internal representation
    if (username_len > 16) {
	pp_bmc_log_warn("[SEM] user name longer than 16 bytes not supported");
	return PP_ERR;
    }
    memset(name, 0, 17); // includes NULL-termination
    memcpy(name, username, username_len);

    if (name_only_lookup == 0) {
        uid = pp_bmc_user_uid_by_name_privlevel(name, session->chan, req_max_priv);
    } else {
        uid = pp_bmc_user_uid_by_name(name);
    }
    if (uid < 1) {
        pp_bmc_log_debug("[SEM] unknown IPMI user '%s'", name);
        return PP_ERR;
    }

    if (!pp_bmc_user_get_enable(uid)) {
        pp_bmc_log_warn("[SEM] user %s is not enabled", name);
        return PP_ERR;
    }

    priv_lvl = pp_bmc_user_get_priv_level(uid, session->chan);
    if (priv_lvl < 0) {
        pp_bmc_log_warn("[SEM] user %s has invalid ipmi privilege level %d", name, priv_lvl);
        return PP_ERR;
    }

    if (PP_ERR == pp_bmc_user_get_password(uid, passwd, sizeof(passwd))) {
        pp_bmc_log_warn("[SEM] cant read password of user %s", name);
        return PP_ERR;
    }

    if ((int)req_max_priv > priv_lvl) {
        if (!name_only_lookup) {
            pp_bmc_log_warn("[SEM] user %s has insufficient privilege level: %.2x < %.2x",
                            name, priv_lvl, req_max_priv);
            return PP_ERR;
        }

        // cap session's priv level by user's priv level
        req_max_priv = priv_lvl;
    }

    // not sure if we should password length 16 or 20 here ...
    session->password_len = 20;
    if (session->password != NULL) free(session->password);
    session->password = malloc(session->password_len);
    memset(session->password, 0, session->password_len);
    strncpy(session->password, passwd, session->password_len);
    session->password_len = session->password_len;

    if (session->username != NULL) free(session->username);
    session->username = malloc(16);
    strncpy(session->username, name, 16);
    session->username_len = username_len;
    
    session->uid = uid & 0x1f;
    
    // choose bmc random number
    if (session->challenge != NULL) free(session->challenge);
    session->challenge = malloc(16);
    srand(time(NULL));
    for (i = 0; i < 16; i++) {
        session->challenge[i] = rand() % 0xff;
    }

    session->req_priv_level = req_max_priv;
    return PP_SUC;
}


/**
 * Get a session by its session id and increase the session reference
 * count. Use pp_bmc_session_delete to release the retrieved session.
 * @return NULL if the specified session does not exist
 */
imsg_session_t* pp_bmc_get_session(unsigned int sessionId) {
    imsg_session_t* session;
    struct list_head* act;
    
    act = session_root.next;
    
    while (act != &session_root) {
        session = list_entry(act, imsg_session_t, link);
        
        if (session->session_id == sessionId) {
            session->ref_count++;
            return session;
        }
        
        act = act->next;
    }
    
    pp_bmc_log_debug("[SEM] could not retrieve session %u", sessionId);
    return NULL;
}

/**
 * Get a session by its session handle and increase the session reference
 * count. Use pp_bmc_session_delete to release the retrieved session. This
 * function is usually only used by GetSessionInfo.
 * @return NULL if the specified session does not exist
 */
imsg_session_t* pp_bmc_get_session_by_handle(unsigned char session_handle) {
    imsg_session_t* session;
    struct list_head* act;
    
    act = session_root.next;
    
    while (act != &session_root) {
        session = list_entry(act, imsg_session_t, link);
        
        if (session->session_handle == session_handle) {
            session->ref_count++;
            return session;
        }
        
        act = act->next;
    }
    
    pp_bmc_log_debug("[SEM] could not retrieve session with handle %.2x", session_handle);
    return NULL;
}

/**
 * Get the n-th active session and increase the session reference count.
 * Index 0 is reserved. Use pp_bmc_session_delete to release the retrieved
 * session. This function is usually only used by GetSessionInfo.
 * @return NULL if the specified session does not exist
 */
imsg_session_t* pp_bmc_get_nth_session(unsigned char n) {
    imsg_session_t* session;
    struct list_head* act;
    unsigned char i;
    
    act = session_root.next;
    i = 0;
    
    while (act != &session_root) {
        i++;
        if (i == n) {
            session = list_entry(act, imsg_session_t, link);
            session->ref_count++;
            return session;
        }
        
        act = act->next;
    }
    
    pp_bmc_log_debug("[SEM] could not retrieve session at position n=%.2x", n);
    return NULL;
}

int pp_bmc_close_session(unsigned int sessionId) {
    imsg_session_t* session;
    
    session = pp_bmc_get_session(sessionId);
    if (session == NULL)
        return PP_ERR;

    // deactivate a sol session, if still running
    if (session->sol_handler != NULL && session->sol_deactivate != NULL) {
	session->sol_deactivate(session);
    }
    
    session->cur_priv_level = 0;  // deactivate session
    session->max_priv_level = 0;
    
    // remove from list and delete the global(list) reference
    list_del(&(session->link));
    pp_bmc_session_delete(session);

    // delete the pp_bmc_get_session reference
    pp_bmc_session_delete(session);
    
    return PP_SUC;
}

void pp_bmc_session_manager_close_all_sessions() {
    struct list_head *i, *n;    
    list_for_each_safe(i, n, &session_root) {
        pp_bmc_close_session((list_entry(i, imsg_session_t, link))->
			     session_id);
    }
}

void pp_bmc_session_refresh(imsg_session_t* session) {
    if (session->inactivity_timeout == -1) {
        // pp_bmc_log_debug("[SEM] Session %u has no inactivity timeout !", session->session_id);
        // this is the case for loopi channel messages, which have a session, but no timeout
        return;
    }
        
    if (pp_select_remove_to(session->inactivity_timeout) == PP_ERR) {
        pp_bmc_log_warn("[SEM] could not refresh session %u - could not remove to", session->session_id);
    }
    
    session->inactivity_timeout = pp_select_add_to(60000, 0, session_inactivity_timeout, session);
}

int pp_bmc_session_manager_init() {
    // initialize linked list for all sessions
    INIT_LIST_HEAD(&session_root);

    sessIdCnt = 1;
    sessHndlCnt = 1;
    
    return PP_SUC;
}

void pp_bmc_session_manager_cleanup() {
    /* maybe we should do some cleanup here */
    pp_bmc_session_manager_close_all_sessions();
}


unsigned char pp_bmc_session_manager_get_session_cnt(unsigned int chan_id) {
    unsigned char cnt = 0x00;
    imsg_session_t* session;
    struct list_head* act;
    
    act = session_root.next;
    
    while (act != &session_root) {
        session = list_entry(act, imsg_session_t, link);
        
        if (session->chan == chan_id) {
            ++cnt;
        }
        
        act = act->next;
    }

    return cnt;
}

unsigned char pp_bmc_session_manager_get_all_session_cnt() {
    unsigned char cnt = 0x00;
    imsg_session_t* session;
    struct list_head* act;
    
    act = session_root.next;
    
    while (act != &session_root) {
        session = list_entry(act, imsg_session_t, link);
        ++cnt;
        act = act->next;
    }

    return cnt;
}

