/**
 * utils.c
 *
 * Description: Commonly Used Utilities
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <malloc.h>
#include <string.h>
#include <fcntl.h>

#include "pp/bmc/bmc_imsg.h"
#include "pp/bmc/utils.h"
#include "pp/bmc/ipmi_sess.h"
#include "pp/bmc/debug.h"

#include "pp/base.h"
#include "pp/xdefs.h"

/*
 * Directory handling
 */

char *getdirname(const char* path)
{
    char *p = strrchr(path, '/');
    if (p) {
        int sz = p - path;
        p = strndup(path, sz + 1);
        p[sz] = '\0';
    }
    return p;
}

int mkdir_recurse(const char* path, mode_t mode)
{
    char *d = getdirname(path);
    if (d) {
        mkdir_recurse(d, mode);;
        free(d);
    }
    return mkdir(path, mode);
}

/*
 * IPMI Reservations
 */

struct reserv_s {
    unsigned char active;
    unsigned short id;
    unsigned char addr;
    unsigned char lun;
    unsigned char chan;
};

reserv_t* reserv_new()
{
    reserv_t *r = malloc(sizeof(reserv_t));
    memset(r, 0, sizeof(reserv_t));
    return r;
}

void reserv_delete(reserv_t* r)
{
    free(r);
}

unsigned short reserv_make(reserv_t* r, const imsg_t *imsg)
{
    assert(r);
    r->active = 1;
    while (!++r->id); /* inc, but skip 0 */
    r->addr = imsg->rq_addr;
    r->lun  = imsg->rq_lun;
    r->chan = imsg->chan;
    return r->id;
}

void reserv_cancel(reserv_t* r)
{
    assert(r);
    r->active = 0;
}

int reserv_check(const reserv_t* r, unsigned short reserv_id,
		 int mandatory, const imsg_t *imsg)
{
    assert(r);
    
    if ((mandatory == PP_BMC_RESERV_OPTIONAL) &
        (reserv_id == 0))
    {
        /* reservation not necessary */
        return PP_SUC;
    }
    
    /* reservation required */
    if (r->active) {
        /* check given reservation */
        return r->id == reserv_id
                && r->addr == imsg->rq_addr
                && r->lun == imsg->rq_lun
                && r->chan == imsg->chan ? PP_SUC : PP_ERR;
    } else {
        /* requested reservation does not exist */
        return PP_ERR;
    }
}

/*
 * IPMI checksum
 */

unsigned char pp_bmc_calc_ipmi_checksum(unsigned char* buffer, int begin, int end) {
    int i;
    unsigned char check;
    check = 0;
    for (i=begin; i<=end; i++) {
        check = check + buffer[i];
    }
    return -check;
}

/**
 * Convert a numeric privilege level to its String value in the config system.
 */
const char* pp_bmc_get_privlevel_string(unsigned char priv_level) {
    const char* priv_lvl_string;
    
    switch (priv_level) {
    case IPMI_PRIV_UNSPEC:
        priv_lvl_string = PP_CD_IPMIPRIVLEVEL_NO_ACCESS_STR;
        break;
    case IPMI_PRIV_CALLBACK:
        priv_lvl_string = PP_CD_IPMIPRIVLEVEL_CALLBACK_STR;
        break;
    case IPMI_PRIV_USER:
        priv_lvl_string = PP_CD_IPMIPRIVLEVEL_USER_STR;
        break;
    case IPMI_PRIV_OPERATOR:
        priv_lvl_string = PP_CD_IPMIPRIVLEVEL_OPERATOR_STR;
        break;
    case IPMI_PRIV_ADMIN:
        priv_lvl_string = PP_CD_IPMIPRIVLEVEL_ADMINISTRATOR_STR;
        break;
    case IPMI_PRIV_OEM:
        priv_lvl_string = PP_CD_IPMIPRIVLEVEL_OEM_STR;
        break;
    default:
        /* cannot happen */
        pp_bmc_log_warn("[UTILS] cannot convert privilege level %.2x to config string", priv_level);
        priv_lvl_string = PP_CD_IPMIPRIVLEVEL_NO_ACCESS_STR;
        assert(0);
    }
    
    return priv_lvl_string;
}

/**
 * Convert a privilege level string value from the config system to the
 * corresponding numeric value.
 */
unsigned char pp_bmc_get_privlevel_value(const char* priv_level_string) {
    if (priv_level_string == NULL) {
        return IPMI_PRIV_UNSPEC;
    }
    if (strcmp(priv_level_string, PP_CD_IPMIPRIVLEVEL_NO_ACCESS_STR) == 0) {
        return IPMI_PRIV_UNSPEC;
    }
    if (strcmp(priv_level_string, PP_CD_IPMIPRIVLEVEL_CALLBACK_STR) == 0) {
        return IPMI_PRIV_CALLBACK;
    }
    if (strcmp(priv_level_string, PP_CD_IPMIPRIVLEVEL_USER_STR) == 0) {
        return IPMI_PRIV_USER;
    }
    if (strcmp(priv_level_string, PP_CD_IPMIPRIVLEVEL_OPERATOR_STR) == 0) {
        return IPMI_PRIV_OPERATOR;
    }
    if (strcmp(priv_level_string, PP_CD_IPMIPRIVLEVEL_ADMINISTRATOR_STR) == 0) {
        return IPMI_PRIV_ADMIN;
    }
    if (strcmp(priv_level_string, PP_CD_IPMIPRIVLEVEL_OEM_STR) == 0) {
        return IPMI_PRIV_OEM;
    }
    
    pp_bmc_log_warn("[UTILS] cannot convert \"%s\" to privilege level", priv_level_string);
    assert(0);
    return IPMI_PRIV_UNSPEC;
}


/*
 * misc
 */

void pp_bmc_get_guid(unsigned char* guid_dest) {
    int i;
    for (i=0; i<16; i++)
        guid_dest[i] = i;
    memcpy(guid_dest, &pp_eui64, sizeof(pp_eui64_t));
}
