/**
 * host_board_id.c
 * 
 * This module provides functions to detect the motherboard type of the
 * host system. This information is needed to load the correct sensor
 * configuration.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/host_board_id.h>
#include <pp/bmc/ipmi_fru.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_fru.h>
#include <pp/bmc/rpc_host_board_id.h>
#include <pp/bmc/sensor.h>

#include <string.h>
#include <stdlib.h>
#include <fcntl.h>

#define MSI_BOARD_ID_MAX_LENGTH		4
#define AMD_BOARD_NAME_MAX_LENGTH	64
#define BOARD_ID_MAX_LENGTH		64


/* on the simulation platform we need to use a fix board id,
   in a real board we get it from the BIOS */
#define noAMD_FIX_BOARD_ID		0
#define noOPMA_EVAL

/*
 * id_value structure defines for different id_types
 */
typedef struct {
    u_short oem_id;
    u_short impl_id;
    u_short opma_version;
    char    boardname[AMD_BOARD_NAME_MAX_LENGTH];
} board_value_opma_t;

typedef struct {
    unsigned char board_id[MSI_BOARD_ID_MAX_LENGTH];
} board_value_msi_fixed_t;

typedef struct {
    u_short board_id;
    unsigned char board_subtype;
} board_value_supermicro_t;

/*
 * Supported OPMA boards
 *
 * Maximum is 7 according to spec (OPMA_MAX_BOARD_COUNT)
 */

board_value_opma_t opma_boards[] = {
    { .oem_id		= 0x1022,
      .impl_id		= 0x0001,
      .opma_version	= 0x0102,
      .boardname        = "AMD_Warthog_Rev1"
    },
    { .oem_id		= 0x1022,
      .impl_id		= 0x0002,
      .opma_version	= 0x0102,
      .boardname        = "AMD_Warthog_Rev2"
    },
    { .oem_id		= 0x1022,
      .impl_id		= 0x0006,
      .opma_version	= 0x0102,
      .boardname        = "AMD_Cheyenne_Rev1"
    },
    { .oem_id		= 0x0000,
      .impl_id		= 0x0000,
      .opma_version	= 0x0102,
      .boardname        = "NVIDIA_Reference"
    },    
    { 0, 0, 0, "" } /* end marker */
};

/*
 * static types
 */
static char hbid_value[BOARD_ID_MAX_LENGTH];
static pp_bmc_host_board_id_t hbid = {
    HOST_BOARD_ID_TYPE_UNDEFINED,
    0,
    (void*)hbid_value
};

static void board_id_add_opma_version(pp_strstream_t* s UNUSED);
static void board_id_to_strstream(pp_bmc_host_board_id_t* id, pp_strstream_t* s);
static void hbid_value_to_strstream(pp_bmc_host_board_id_t* id, pp_strstream_t* s);

#if defined (LARA_KIMMSI)
/*
 * MSI board id directly checks FRU data.
 * If something sensible is found, that number is returned 
 * If nothing is found, a zero length is returned but
 * hbid.id_type is still set to HOST_BOARD_ID_TYPE_MSI_FIXED.
 * That means, in order to retrigger a board id check,
 * the BMC has to be restarted, i.e., the platform controller has
 * to be rebooted (that's ok, don't worry)
 */
static void kimmsi_board_id_get(pp_bmc_host_board_id_t* bid) {
    ipmi_fru_header_t fru_header;
    unsigned char chassis_info_buf[8];
    ipmi_fru_chassis_info_t* chassis_info =
	(ipmi_fru_chassis_info_t*)chassis_info_buf;
    int n;
    

    // get fru header
    if ((n = pp_bmc_fru_read_data(&fru_header, 0, sizeof(ipmi_fru_header_t)))
	== PP_ERR
	|| (unsigned int)n < sizeof(ipmi_fru_header_t) // no header) 
	// get chassis info offset
	|| fru_header.offset.chassis == 0              // no chassis info
	|| (n = pp_bmc_fru_read_data(chassis_info_buf,
				     fru_header.offset.chassis * 8,
				     sizeof(chassis_info_buf)))
	== PP_ERR
	|| n < 5) {                        // no part_number in chassis info
	bid->id_type = HOST_BOARD_ID_TYPE_UNDEFINED;
	bid->id_length = 0;
    }
    // get chassis info
    else {
	bid->id_type = HOST_BOARD_ID_TYPE_MSI_FIXED;
	bid->id_length =
	    chassis_info->part_number_length > MSI_BOARD_ID_MAX_LENGTH ?
	    MSI_BOARD_ID_MAX_LENGTH : chassis_info->part_number_length;
	memcpy(bid->id_value, chassis_info->part_number, bid->id_length);
    }
}
#endif /* LARA_KIMMSI */

#if defined(PP_FEAT_OPMA_HW)
static void
kimamd_board_id_get(pp_bmc_host_board_id_t* bid)
{
#if defined(AMD_FIX_BOARD_ID)
#if defined(OPMA_EVAL)
    bid->id_type = HOST_BOARD_ID_TYPE_OPMA_EVAL;
    bid->id_length = 0;
    bid->id_value = NULL;
#else
    bid->id_type = HOST_BOARD_ID_TYPE_OPMA;
    bid->id_length = sizeof(board_value_opma_t);
    memcpy(bid->id_value, &opma_boards[AMD_FIX_BOARD_ID], sizeof(board_value_opma_t));
#endif
#else /* ! AMD_FIX_BOARD_ID */
    u_short oem_id, impl_id, opma_version;

    if (PP_SUCCED(pp_cfg_get_ushort_nodflt(&oem_id, "ipmi.oem.amd.oem_id")) &&
	PP_SUCCED(pp_cfg_get_ushort_nodflt(&impl_id, "ipmi.oem.amd.impl_id")) &&
	PP_SUCCED(pp_cfg_get_ushort_nodflt(&opma_version, "ipmi.oem.amd.opma_version"))) {

	board_value_opma_t *id = (board_value_opma_t*) bid->id_value;
	
	id->oem_id = oem_id;
	id->impl_id = impl_id;
	id->opma_version = opma_version;
	
	bid->id_type = HOST_BOARD_ID_TYPE_OPMA;
	bid->id_length = sizeof(board_value_opma_t);
    } else {
	bid->id_type = HOST_BOARD_ID_TYPE_UNDEFINED;
	bid->id_length = 0;
    }
#endif /* AMD_FIX_BOARD_ID */
}
#endif /* PP_FEAT_OPMA_HW */

#if defined(PRODUCT_SMIDC)
static void
supermicro_board_id_get(pp_bmc_host_board_id_t* bid)
{
    unsigned short board_id;
    unsigned short board_subtype;

    bid->id_type = HOST_BOARD_ID_TYPE_SUPERMICRO;
    bid->id_length = sizeof(board_value_supermicro_t);

    // get values, use defaults (0) in case of error
    pp_cfg_get_ushort(&board_id, "ipmi.oem.supermicro.board_id");
    pp_cfg_get_ushort(&board_subtype, "ipmi.oem.supermicro.board_subtype");

    board_value_supermicro_t *id = (board_value_supermicro_t*) bid->id_value;
    
    id->board_id = board_id;
    id->board_subtype = board_subtype;
}
#endif

pp_bmc_host_board_id_t* pp_bmc_host_board_id_get() {
    
    if (hbid.id_type == HOST_BOARD_ID_TYPE_UNDEFINED) {
#if defined (PP_FEAT_OPMA_HW)
	kimamd_board_id_get(&hbid);
#elif defined (LARA_KIMMSI)
	kimmsi_board_id_get(&hbid);
#elif defined (PRODUCT_LARA_BMC_TEST)
        hbid.id_type = HOST_BOARD_ID_TYPE_DUMMY;
	hbid.id_length = 0;
#elif defined (PRODUCT_ICPMMD)
	hbid.id_type = HOST_BOARD_ID_TYPE_ICP;
	hbid.id_length = 0;
#elif defined (PRODUCT_SMIDC)
	supermicro_board_id_get(&hbid);
#elif defined (PP_BMC_LOCAL_TEST)
	hbid.id_type = HOST_BOARD_ID_TYPE_TEST;
	hbid.id_length = 0;
#elif defined (KIRA_RPC)
	rpc_host_board_id_get(&hbid);
#elif defined (PP_BOARD_KIRA)
	hbid.id_type = HOST_BOARD_ID_TYPE_DUMMY;
	hbid.id_length = 0;
#endif
    }
    return &hbid;
}

#ifdef PP_FEAT_OPMA_HW
int
pp_bmc_host_board_id_set_opma(u_short oem_id, u_short impl_id,
			      u_short opma_version)
{
    board_value_opma_t *board = opma_boards;
    u_char already_set = 0, set_new_id = 1;
    
    if (hbid.id_type != HOST_BOARD_ID_TYPE_UNDEFINED) {
	already_set = 1;
    }

    if (opma_version > OPMA_VERSION_SUPPORTED) {
	pp_bmc_log_warn("[HW] unsupported OPMA version");
	return HOST_BOARD_ID_OPMA_VERSION_UNSUPPORTED;
    }
	
    while (board && board->oem_id != 0 && board->impl_id != 0) {
	if (board->oem_id == oem_id &&
	    board->impl_id == impl_id) {
	    break;
	}
	board++;
    }

    if (!board || (board->oem_id == 0 && board->impl_id == 0)) {
	pp_bmc_log_warn("[HW] unsupported OPMA oem/implementation ID");
	return HOST_BOARD_ID_OPMA_BOARD_UNSUPPORTED;
    }

    if (already_set) {
	board_value_opma_t *id = (board_value_opma_t*) hbid.id_value;
	
	if (oem_id == id->oem_id &&
	    impl_id == id->impl_id &&
	    opma_version == id->opma_version) {

	    /* got the same id as before, ignore */
	    set_new_id  = 0;
	}
    }

    if (set_new_id) {
	board_value_opma_t* current_board = (board_value_opma_t*) hbid_value;
	current_board->oem_id = oem_id;
	current_board->impl_id = impl_id;
	current_board->opma_version = opma_version;

	if (PP_FAILED(pp_cfg_set_ushort(oem_id, "ipmi.oem.amd.oem_id")) ||
	    PP_FAILED(pp_cfg_set_ushort(impl_id, "ipmi.oem.amd.impl_id")) ||
	    PP_FAILED(pp_cfg_set_ushort(opma_version, "ipmi.oem.amd.opma_version")))
	    {
		pp_bmc_log_warn("[HW] could not write opma type to cfg");
	    }

	pp_cfg_save(DO_FLUSH);
    }

    if (already_set) {
	return HOST_BOARD_ID_OPMA_ALREADY_SET;
    }
        
    // (re)start the bmc sensor part
    if ( PP_FAILED(pp_bmc_sensor_scanner_stop()) ||
         PP_FAILED(pp_bmc_sensor_topo_parse_new()) ||
	 PP_FAILED(pp_bmc_sensor_scanner_start()) )
    {
	pp_bmc_log_pfatal("[HW] restarting BMC sensor scanning after getting board-id failed");
    }
    
    return HOST_BOARD_ID_OPMA_ACCEPTED;
}

int
pp_bmc_host_board_id_get_opma(u_short *oem_id, u_short *impl_id,
			      u_short *opma_version)
{
    board_value_opma_t *board;
    
    assert(oem_id && impl_id && opma_version);
    
    if (hbid.id_type != HOST_BOARD_ID_TYPE_OPMA) {
	return PP_ERR;
    }

    board = (board_value_opma_t *) hbid.id_value;

    *oem_id = board->oem_id;
    *impl_id = board->impl_id;
    *opma_version = board->opma_version;

    return PP_SUC;
}

vector_t*
pp_bmc_host_board_id_get_supported_opma()
{
    board_value_opma_t *board = opma_boards;
    vector_t* ret;

    ret = vector_new2(NULL, OPMA_MAX_BOARD_COUNT, sizeof(opma_board_id_t), NULL);

    while (board && board->oem_id != 0 && board->impl_id != 0) {
	opma_board_id_t id = {
	    .oem_id = board->oem_id,
	    .impl_id = board->impl_id
	};
	vector_add2(ret, &id);
	board++;
    }

    return ret;
}
#endif /* PP_FEAT_OPMA_HW */

unsigned char* pp_bmc_host_board_id_to_filename(pp_bmc_host_board_id_t* id) {
    pp_strstream_t s = PP_STRSTREAM_INITIALIZER;
    board_id_to_strstream(id, &s);
    pp_strappend(&s, ".top");
    return pp_strstream_buf(&s);
}

unsigned char* pp_bmc_host_board_id_to_string(pp_bmc_host_board_id_t* id) {
    pp_strstream_t s = PP_STRSTREAM_INITIALIZER;
    board_id_to_strstream(id, &s);
    return pp_strstream_buf(&s);
}

unsigned char* pp_bmc_host_board_id_initial_filename(void)
{
#ifdef PP_FEAT_OPMA_HW  
    pp_strstream_t s = PP_STRSTREAM_INITIALIZER;
    pp_strappend(&s, "OPMA_Initial");
    board_id_add_opma_version(&s);
    pp_strappend(&s, ".top");
    return pp_strstream_buf(&s);
#else
    return NULL;
#endif
}

static void board_id_to_strstream(pp_bmc_host_board_id_t* id, pp_strstream_t* s) {
    switch (id->id_type) {
      case HOST_BOARD_ID_TYPE_MSI_FIXED:
	  if (id->id_length > 0) {
	      pp_strappend(s, "MSI_");
	      pp_strappendn(s, ((board_value_msi_fixed_t*)id->id_value)->
			    board_id, id->id_length);
	  }
          break;
      case HOST_BOARD_ID_TYPE_OPMA:
	  {
	      board_value_opma_t* board = (board_value_opma_t*) id->id_value;
	      board_value_opma_t* list  = opma_boards;
	      	      
	      /* lookup the boardname in the complete list */
	      while (list && list->oem_id != 0 && list->impl_id != 0) {
		  if (board->oem_id == list->oem_id &&
		      board->impl_id == list->impl_id) {
		      break;
		  }
		  list++;
	      }

	      pp_strappend(s, list->boardname);
	      board_id_add_opma_version(s);
	  }
	  break;
      case HOST_BOARD_ID_TYPE_ICP:
	  pp_strappendf(s, "icp-xblade");
	  break;
      case HOST_BOARD_ID_TYPE_SUPERMICRO:
          {
              board_value_supermicro_t* sid = id->id_value;
              char filename[25];
              char filename_full[100];
              int file;

              if ((sid->board_id != 0) && (sid->board_subtype)) {
                  sprintf(filename, "supermicro.%04x.%d",
			  sid->board_id, sid->board_subtype);
              } else {
                  sprintf(filename, "supermicro");
              }

              // use supermicro.top if the calculated file does not exist
              sprintf(filename_full, "/etc/bmc/%s.top", filename);
              if ((file = open(filename_full, O_RDONLY)) < 0) {
                  pp_bmc_log_notice("[HW] topology file %s does not exist, switching to default", filename);
                  sprintf(filename, "supermicro");
              }
              pp_strappend(s, filename);

              pp_bmc_log_info("[HW] get_board_id_string = %s", s);
          }
	  break;
      case HOST_BOARD_ID_TYPE_DUMMY:
	  pp_strappendf(s, "dummy");
          break;
      case HOST_BOARD_ID_TYPE_TEST:
	  pp_strappendf(s, "test");
	  break;	
      case HOST_BOARD_ID_TYPE_OPMA_EVAL:
	  pp_strappend(s, "OPMA_eval");
	  board_id_add_opma_version(s);
	  break;
      case HOST_BOARD_ID_TYPE_RPC:
	  pp_strappendn(s, id->id_value, id->id_length);
	  break;
      default:
	  pp_strappend(s, "Unknown_");
	  hbid_value_to_strstream(id, s);
    }
}

static void
board_id_add_opma_version(pp_strstream_t* s UNUSED)
{
#if defined(KIRA_KIMAMDG4)
    pp_strappend(s, "_KIMG4");  
#endif
}

static void hbid_value_to_strstream(pp_bmc_host_board_id_t* id, 
				    pp_strstream_t* s) {
    int i;
    for (i = 0; i < id->id_length; ++i) {
	pp_strappendf(s, "%hhd", ((char*)id->id_value)[i]);
    }
}
