/**
 * host_hardware.c
 * 
 * Defines abstract host hardware that can be mapped to different hosts.
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <pp/bmc/host_hardware.h>
#include <pp/bmc/tp_gpio_act.h>
#include <pp/bmc/tp_gpio_multi_act.h>
#include <pp/bmc/host_power.h>
#include <pp/bmc/debug.h>

#include <pp/bmc/serial_lowlevel.h>
#include <pp/bmc/host_sensors.h>
#include <pp/bmc/host_graceful_ops.h>
#include <pp/bmc/host_i2c_access.h>

#include <pp/selector.h>

/* gpio led - one blink per second */
#define GPIO_LED_BLINK_INTERVAL_LEN  500
#define GPIO_LED_BLINK_INTERVAL_CNT  2

#ifdef LARA_KIMAMD
# define GPIO_LED_DO_BLINK	     0
#else
# define GPIO_LED_DO_BLINK	     1
#endif

/* OPMA interface id pin values */
#define OPMA_INTERFACE_ID_KCS		0x00
#define OPMA_INTERFACE_ID_BT		0x01
#define OPMA_INTERFACE_ID_SMIC		0x02
#define OPMA_INTERFACE_ID_SSIF		0x03
/* 0x04, 0x05 reserved */
#define OPMA_INTERFACE_ID_UPGRADE	0x06
#define OPMA_INTERFACE_ID_UNINITIALIZED	0x07

#define HEARTBEAT_INTERVAL	500 // ms (duration of half period)

/* internal prototypes */
static int internal_button_id(int id);
static int id_led_blink_timeout(const int item_id UNUSED, void* ctx UNUSED);
static void update_id_led(void);


/* global variables */
typedef struct {
    /* set to 1 if button is disabled, 0 if button is enabled */
    /* power, reset, sleep, diag, nmi                         */
    int button_disable[5];
    
    /* blink timeout handler */
    int id_led_blink_to_hndl;
    /* no of blinks remaining for one second */
    int id_led_blink_cnt;
    /* remaining interval time in s */
    int id_led_interval_remaining;

    /* blinking, or not */
    int id_led_blinking;
    
    /* timeout handler and actor for clearing cmos */
    int clear_cmos_to_hndl;
    pp_tp_gpio_act_t* clear_cmos_act;

#if defined(PP_FEAT_IPMI_SERVER_HEARTBEAT)
    // heartbeat
    int heartbeat_timer_id;
    int heartbeat_last;
#endif

    /* 'global' local lockout */
    int local_lockout;
} host_hardware_globals_t;


static host_hardware_globals_t host_hardware_globals;
static host_hardware_globals_t* hhg;

#if defined(PP_FEAT_IPMI_SERVER_HEARTBEAT)
static int heartbeat_timer(const int item_id UNUSED, void* ctx UNUSED)
{
    pp_tp_gpio_act_t *act = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_HEARTBEAT);
    if (act) {
	hhg->heartbeat_last = !hhg->heartbeat_last;
	act->set_gpio(act, hhg->heartbeat_last ? 1 : 0);
    }

    // don't stop timer, if no act is there (it could be just temporarily gone)

    return PP_SUC;
}
#endif /* PP_FEAT_IPMI_SERVER_HEARTBEAT */

int pp_bmc_hardware_init() {
    int rv;
    int i;

    hhg = &host_hardware_globals;

    pp_bmc_host_sensor_init();
    
    pp_bmc_serial_lowlevel_init();

    rv = PP_SUC;
    if (pp_bmc_host_power_init() == PP_ERR) {
        pp_bmc_log_error("[HW] could not initialize bmc host power control");
        rv = PP_ERR;
    }
    
    // all buttons start enabled
    for (i=0; i<4; i++) {
        hhg->button_disable[i] = 0;
    }
    
    // id_led not blinking
    hhg->id_led_blink_to_hndl = -1;

    // no cmos clear in progress
    hhg->clear_cmos_to_hndl = -1;

    // local lockout starts disabled
    hhg->local_lockout = 0;
    
#ifdef PRODUCT_SMIDC
    if (PP_FAILED(pp_bmc_host_graceful_ops_init())) {
	pp_bmc_log_error("[HW] could not initialize bmc host graceful power control");
	rv = PP_ERR;
    }

    if (PP_FAILED(pp_bmc_host_i2c_access_init())) {
	pp_bmc_log_error("[HW] could not initialize bmc host i2c access control");
	rv = PP_ERR;
    }
#endif

#if defined(PP_FEAT_IPMI_SERVER_HEARTBEAT)
    hhg->heartbeat_last = 0;
    hhg->heartbeat_timer_id = pp_select_add_to(HEARTBEAT_INTERVAL, 1,
					       heartbeat_timer, NULL);
    if (hhg->heartbeat_timer_id == -1) {
	pp_bmc_log_error("[HW] unable to start heartbeat timer");
	rv = PP_ERR;
    }
#endif /* PP_FEAT_IPMI_SERVER_HEARTBEAT */

    return rv;
}

void pp_bmc_hardware_cleanup() {

    if (hhg->id_led_blink_to_hndl != -1) {
        hhg->id_led_interval_remaining = 0;
        update_id_led();
    }

#if defined(PP_FEAT_IPMI_SERVER_HEARTBEAT)
    if (hhg->heartbeat_timer_id != -1) {
	pp_select_remove_to(hhg->heartbeat_timer_id);
	hhg->heartbeat_timer_id = -1;
    }
#endif /* PP_FEAT_IPMI_SERVER_HEARTBEAT */

#ifdef PRODUCT_SMIDC
    pp_bmc_host_i2c_access_cleanup();
    pp_bmc_host_graceful_ops_cleanup();
#endif
    pp_bmc_host_power_cleanup();
    pp_bmc_host_sensor_cleanup();
    pp_bmc_serial_lowlevel_cleanup();
}

/**
 * convert an external host_sensor button id to an internal id (0-4)
 * for power, reset, sleep, diag. Returns -1 if button cannot be mapped.
 */
static int internal_button_id(int id) {
    if (id == PP_SENSOR_POWER_BTN)
        return 0;
    if (id == PP_SENSOR_RESET_BTN)
        return 1;
    if (id == PP_SENSOR_SLEEP_BTN)
        return 2;
    if (id == PP_SENSOR_DIAG_BTN)
        return 3;
    if (id == PP_SENSOR_NMI_BTN)
	return 4;
    return -1;
}

void pp_bmc_hardware_button_enable_set(int button, int enable) {
    int b_id;
    
    b_id = internal_button_id(button);
    assert(b_id != -1);

    if (enable == HOST_BUTTON_ENABLE)
        hhg->button_disable[b_id] = 0;
    else
        hhg->button_disable[b_id] = 1;
}

int pp_bmc_hardware_button_enable_get(int button) {
    int b_id;
    
    b_id = internal_button_id(button);
    assert(b_id != -1);
    
    if (hhg->button_disable[b_id] == 0)
        return HOST_BUTTON_ENABLE;
    else
        return HOST_BUTTON_DISABLE;
}

int pp_bmc_hardware_button_enable_check(int button UNUSED) {
    // disabling buttons is possible for all our buttons
    return PP_SUC;
}

int pp_bmc_hardware_frontpanel_lockout_get() {
    if ((hhg->button_disable[0] == 1)  &
	(hhg->button_disable[1] == 1)) 
    {
        return PP_SUC;
    } else {
        return PP_ERR;
    }
}

int pp_bmc_hardware_frontpanel_hard_lockout_set(int lockout) {
    pp_tp_gpio_act_t* lock_act;

    assert(lockout == 0 || lockout == 1);
    
    if ((lock_act = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_LOCAL_LOCK)) == NULL) {
        /* no actor = not supported */
        return PP_ERR;
    }

    lock_act->set_gpio(lock_act, lockout);
    hhg->local_lockout = lockout;

    return PP_SUC;
}

int pp_bmc_hardware_frontpanel_hard_lockout_get() {
    if (pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_LOCAL_LOCK) == NULL) {
        return -1;
    }
    
    return hhg->local_lockout;
}

int pp_bmc_hardware_diag_int_exists() {
    if (pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_DIAG_INT) == NULL)
        return PP_ERR;
    else
        return PP_SUC;
}

int pp_bmc_hardware_chassis_intrusion_exists() {
    if (pp_bmc_host_sensor_get_sensor(PP_SENSOR_CHASSIS_INTRUSION) == NULL)
        return PP_ERR;
    else
        return PP_SUC;
}

int pp_bmc_hardware_chassis_closed() {
    pp_tp_sensdev_t* sens;
    
    sens = pp_bmc_host_sensor_get_sensor(PP_SENSOR_CHASSIS_INTRUSION);
    
    if (sens == NULL) return PP_SUC;
    
    if (pp_tp_sensdev_get_reading(sens) == 0) return PP_ERR;
            
    return PP_SUC;
}

int pp_bmc_hardware_clr_intrusion_exists() {
    if (pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_CLR_INTRUSION) == NULL)
        return PP_ERR;
    else
        return PP_SUC;
}

int pp_bmc_hardware_clr_intrusion_flag() {
    pp_tp_gpio_act_t *act = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_CLR_INTRUSION);
    if  (!act) return PP_ERR;

    pp_bmc_log_debug("[HW] clearing chassis intrusion flag");
    act->set_gpio(act, 1); // set 1 to clear intrusion flag, auto-resets to 0
    return PP_SUC;
}

int pp_bmc_hardware_drive_health() {
    pp_tp_sensdev_t* sens;
    
    sens = pp_bmc_host_sensor_get_sensor(PP_SENSOR_DRIVE_HEALTH);
    
    if (sens == NULL) return PP_SUC;
    
    if (pp_tp_sensdev_get_reading(sens) == 1) return PP_ERR;
        
    return PP_SUC;
}

int pp_bmc_hardware_fan_health() {
    pp_tp_sensdev_t* sens;
    
    sens = pp_bmc_host_sensor_get_sensor(PP_SENSOR_FAN_HEALTH);
    
    if (sens == NULL) return PP_SUC;
    
    if (pp_tp_sensdev_get_reading(sens) == 1) return PP_ERR;
        
    return PP_SUC;
}

int pp_bmc_hardware_chassis_identify_device_exists() {
    if (pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_ID_LED) == NULL)
        return PP_ERR;
    else
        return PP_SUC;
}

void pp_bmc_hardware_chassis_identify_set(int interval) {
    if (pp_bmc_hardware_chassis_identify_device_exists() == PP_ERR)
        return;
    
    pp_bmc_log_debug("[HW] setting identify interval to %d", interval);
    
    hhg->id_led_interval_remaining = interval;
    hhg->id_led_blinking = GPIO_LED_DO_BLINK;
    
    update_id_led();
}


static void update_id_led(void) {
    pp_tp_gpio_act_t* id_led;
    id_led = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_ID_LED);
    
    if (id_led == NULL)
        return;
 
    if (hhg->id_led_interval_remaining == 0) {
        // stop timer (should be necessary)
        if (hhg->id_led_blink_to_hndl != -1) {
            pp_select_remove_to(hhg->id_led_blink_to_hndl);
            hhg->id_led_blink_to_hndl = -1;
        }
        // turn id led off
        id_led->set_gpio(id_led, 0);
    } else {
        // check the timer (must be started)
        if (hhg->id_led_blink_to_hndl == -1) {
            // start timer
            hhg->id_led_blink_to_hndl = pp_select_add_to(GPIO_LED_BLINK_INTERVAL_LEN, 1,
                                 id_led_blink_timeout, NULL);
            // new timer, counts within blink interval can be reset
            hhg->id_led_blink_cnt = 0;
        }
        // set led status
        if (!hhg->id_led_blinking ||
	    (hhg->id_led_blink_cnt % 2) == 0) {
            // first half of cycle (or not blinking), turn led on
            id_led->set_gpio(id_led, 1);
        } else {
            // second half of cycle, turn led off
            id_led->set_gpio(id_led, 0);
        }
    }
}

static int id_led_blink_timeout(const int item_id UNUSED, void* ctx UNUSED)
{
    hhg->id_led_blink_cnt++;
    if (hhg->id_led_blink_cnt == GPIO_LED_BLINK_INTERVAL_CNT) {
        hhg->id_led_blink_cnt = 0;
        if (hhg->id_led_interval_remaining > 0)
            hhg->id_led_interval_remaining--;
    }
    update_id_led();
    
    return PP_SUC;
}

static int
clear_cmos_to_hndl(const int item_id UNUSED, void* ctx UNUSED)
{
    hhg->clear_cmos_act->set_gpio(hhg->clear_cmos_act, 0);
    hhg->clear_cmos_to_hndl = -1;

    return PP_SUC;
}

int
pp_bmc_hardware_clear_cmos(void)
{
    int ret = PP_ERR;
    
    if (hhg->clear_cmos_to_hndl != -1) {
	pp_bmc_log_debug("[HW] %s: clear_cmos already in progress", ___F);
	goto bail_success;
    }

    hhg->clear_cmos_act = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_CLEAR_CMOS);
    if (hhg->clear_cmos_act == NULL) {
	pp_bmc_log_debug("[HW] %s: actor not found", ___F);
	goto bail;
    }
    
    hhg->clear_cmos_act->set_gpio(hhg->clear_cmos_act, 1);
    
    /* TODO variable timeout depending on the board we are running on,
            AMD Warthog requires 2s */

    if ((hhg->clear_cmos_to_hndl = pp_select_add_to(2000, 0, clear_cmos_to_hndl,
						    NULL)) == PP_ERR) {
	goto bail;	
    }

 bail_success:
    ret = PP_SUC;
    
 bail:
    return ret;
}

/**
 * Receive a sensor reading for the frontpanel buttons (hardwired in host_sensor.c
 */
void pp_bmc_hardware_receive_reading(int sensor_reading, void* ctx) {
    int sensor_id = (int)(long)ctx;
    pp_tp_gpio_act_t* go;
    
    //pp_bmc_log_debug("[HW] received new sensor event %d for sensor_id %d", sensor_reading, sensor_id);
    
    switch (sensor_id) {

    case HOST_BUTTON_POWER:
    case HOST_BUTTON_RESET:
    case HOST_BUTTON_STANDBY:
    case HOST_BUTTON_DIAG_INT:
    case HOST_BUTTON_NMI:
        // block button-down events
        if (sensor_reading == 1) {
            // this is a button activate event

            // block power button only if power is on, other buttons always
            if (sensor_id != HOST_BUTTON_POWER || pp_bmc_host_power_state_is_on()) {
                // check if button is disabled
                if (hhg->button_disable[internal_button_id(sensor_id)] == 1) {
                     /* button is disabled, completely ignore */
                    return;
                }
            }
        }
        
        // button not blocked, forward sensor event
        if ((sensor_id == HOST_BUTTON_POWER) ||
            (sensor_id == HOST_BUTTON_RESET))
            pp_bmc_host_power_receive_reading(sensor_reading, ctx);
        
        if (sensor_id == HOST_BUTTON_STANDBY) {
            go = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_STANDBY);
            if (go != NULL)
                go->set_gpio(go, sensor_reading);
            //pp_bmc_log_debug("[HW] new standby button value %d", sensor_reading);
        }

        if (sensor_id == HOST_BUTTON_DIAG_INT) {
            go = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_DIAG_INT);
            if (go != NULL)
                go->set_gpio(go, sensor_reading);
            //pp_bmc_log_debug("[HW] new diag int button value %d", sensor_reading);
        }

        if (sensor_id == HOST_BUTTON_NMI) {
            go = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_NMI);
            if (go != NULL)
                go->set_gpio(go, sensor_reading);
            //pp_bmc_log_debug("[HW] new nmi int button value %d", sensor_reading);
        }
	break;

    case HOST_BUTTON_ID:
        // turn of id functionality if button is pressed
        if (sensor_reading == 1) {
            hhg->id_led_interval_remaining = 0;
            update_id_led();
        }
        break;
    default:
        pp_bmc_log_fatal("[HW] received unknown frontpanel button event");
    }
}

#ifdef PP_FEAT_OPMA_HW
int
pp_bmc_hardware_setup_interface_id(pp_bmc_smi_mode_t mode)
{
    pp_tp_gpio_multi_act_t *act_id;
    u_char value;
    int ret = 0;
    
    switch (mode) {
      case PP_SMI_MODE_KCS:
	  value = OPMA_INTERFACE_ID_KCS;
	  break;
      case PP_SMI_MODE_BT:
	  value = OPMA_INTERFACE_ID_BT;
	  break;
      case PP_SMI_MODE_SMIC:
	  value = OPMA_INTERFACE_ID_SMIC;
	  break;
      case PP_SMI_MODE_UNKNOWN:
      case PP_SMI_MODE_LOOPBACK:	  
      default:
	  value = OPMA_INTERFACE_ID_UNINITIALIZED;
	  break;
    }

    /* get ID actors */
    if ((act_id = (pp_tp_gpio_multi_act_t*) pp_bmc_host_sensor_get_actor(PP_ACTOR_MCARD_ID)) == NULL) {
	pp_bmc_log_error("[HW] %s: cannot get multi actor for mcard id", ___F);
	goto bail;
    }

    if (PP_FAILED(pp_tp_gpio_multi_act_set(act_id, value & 0x07, 0))) {
	pp_bmc_log_error("[HW] %s: cannot set multi actor for mcard id", ___F);
	goto bail;
    }
	
    ret = PP_SUC;
    
 bail:
     return ret;
}
#else /* !PP_FEAT_OPMA_HW */
#if defined(PP_FEAT_IPMI_SERVER_SMI_CHAN)
int
pp_bmc_hardware_setup_interface_id(pp_bmc_smi_mode_t mode UNUSED)
{
    return PP_SUC;
}
#endif
#endif /* !PP_FEAT_OPMA_HW */
