/**
 * host_i2c_access.c
 *
 * Implements Supermicro's I2C access control.
 *
 * I2C access has to be synchronized between BIOS and BMC. After
 * Power On or Reset, BIOS should have control. Additionally,
 * there are OEM cmd to allow BIOS to manually request/release control.
 *
 * (c) 2006 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <pp/bmc/host_power.h>
#include <pp/bmc/host_graceful_ops.h>
#include <pp/bmc/bmc_event.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/host_hardware.h>
#include <pp/bmc/tp_cond.h>
#include <pp/bmc/tp_static_cond.h>
#include <pp/bmc/host_i2c_access.h>

#include <pp/base.h>
#include <pp/selector.h>

/*
 * Global values
 */

#define SMB_ACCESS_TIMEOUT 30000

static int timer_id;
static int last_power_state;

/*
 * Internal functions
 */

static int timer(int id, void* ctx);

enum i2c_accessor_e { BIOS, BMC };

static int i2c_access(enum i2c_accessor_e accessor)
{
    pp_tp_cond_t* cond = pp_bmc_host_get_condition(PP_BMC_COND_I2C_ACCESS);
    if (!cond) {
	pp_bmc_log_warn("[HW] unable to change I2C control: no 'i2c_access' cond available");
	return PP_ERR;
    }

    // stop any timer
    if (timer_id != -1) {
	pp_select_remove_to(timer_id);
	timer_id = -1;
    }

    if (accessor == BIOS) {
	// start a (new) time
	timer_id = pp_select_add_to(SMB_ACCESS_TIMEOUT, 0, timer, NULL);
	if (timer_id == -1) {
	    pp_bmc_log_warn("[HW] unable to create timer for I2C control");
	}
    }

    pp_bmc_tp_static_cond_set((pp_tp_static_cond_t*)cond, accessor == BMC ? 1 : 0);
    return PP_SUC;
}

static int timer(int id UNUSED, void* ctx UNUSED)
{
    timer_id = -1;
    pp_bmc_log_info("[HW] I2C access enabled by timeout");
    i2c_access(BMC); // switch I2C access to BMC again
    return PP_SUC;
}

static void power_control_hndl(int control_action)
{
    if (control_action == HOST_POWER_CONTROL_RESET) {
	pp_bmc_log_info("[HW] I2C access disabled by RESET");
	i2c_access(BIOS);
    }
}

static void power_state_hndl(int new_state UNUSED)
{
    int on = pp_bmc_host_power_state_is_on();
    if (on != last_power_state) {
	if (on) {
	    pp_bmc_log_info("[HW] I2C access disabled by POWER ON");
	    i2c_access(BIOS);
	}
	last_power_state = on;
    }
}

/*
 * Public functions
 */

int pp_bmc_host_i2c_access_init(void)
{
    // presence of i2c_access cond not checked here, since topology not yet parsed

    last_power_state = pp_bmc_host_power_state_is_on();

    // register hander to get reset events
    pp_bmc_host_power_control_subscribe(power_control_hndl);
    pp_bmc_host_power_state_subscribe(power_state_hndl);

    // initially start a time
    timer_id = pp_select_add_to(SMB_ACCESS_TIMEOUT, 0, timer, NULL);
    if (timer_id == -1) {
	pp_bmc_log_warn("[HW] unable to create initial timer for I2C control");
    }

    return PP_SUC;
}

void pp_bmc_host_i2c_access_cleanup(void)
{
    if (timer_id != -1) pp_select_remove_to(timer_id);
    pp_bmc_host_power_state_unsubscribe(power_state_hndl);
    pp_bmc_host_power_control_unsubscribe(power_control_hndl);
}

int pp_bmc_host_i2c_access_enable(void)
{
    pp_bmc_log_info("[HW] I2C access enabled by BIOS OEM cmd");
    return i2c_access(BMC);
}

int pp_bmc_host_i2c_access_disable(void)
{
    pp_bmc_log_info("[HW] I2C access disabled by BIOS OEM cmd");
    return i2c_access(BIOS);
}

int pp_bmc_host_i2c_access_get_state(void)
{
    pp_tp_cond_t* cond = pp_bmc_host_get_condition(PP_BMC_COND_I2C_ACCESS);
    if (!cond) {
	pp_bmc_log_warn("[HW] unable to get I2C control state: no 'i2c_access' cond avalable");
	return 1; // return "BMC has access" by default
    }

    return pp_bmc_tp_cond_is_true(cond) ? 1 : 0;
}
