/**
 * host_power.h
 *
 * Provides an abstraction to the host power control. Offers functions to
 * - control the host power
 * - track host power actions
 * - query the host power status
 * - track the host power status
 * - options: (power_cycle_interval, ...)
 * 
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <pp/bmc/host_power.h>
#include <pp/bmc/debug.h>

#include <pp/vector.h>
#include <pp/base.h>
#include <pp/selector.h>
#include <pp/cfg.h>

#include <pp/bmc/topo_classes.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_gpio_dev.h>

#include <pp/bmc/host_sensors.h>
#include <pp/bmc/bmc_chassis.h>

/*
 * More (internal) power control states 
 */
#define HOST_POWER_CONTROL_USER      -2   // user presses button
#define HOST_POWER_CONTROL_NONE      -1   // no active control action (idle)


/* global vars */
typedef struct {
    vector_t* power_state_listeners;
    vector_t* power_control_listeners;

    /* the current power state                                    */
    int host_power_state;
    /* PP_ERR = no success, PP_SUC = success / not initialized    */
    int last_control_success;
    /* currently active power control actions                     */
    int host_power_control;
    
    /* power cycle time interval                                        */
    /* this field will be evaluated right after power_off is finished   */
    /* if its value is >= 0, then a power_on will be scheduled          */
    /*                                                                  */
    /* this value should only be changed while power_state = power_off  */
    int power_cycle_time;
    
    /* power_cycle timeout handler id, initiates power_on on expiration */
    /* the timer is started right after the power_down has finished     */
    /*                                                                  */
    /* this value is valid while power_state is idle and will be reset  */
    /* before power_on                                                  */
    int power_cycle_to_hndl;
    
    /* the source of the power cycle (stored for power_on source)       */
    u_char power_cycle_source;
    u_char power_cycle_channel;
    
} host_power_globals_t;

/* nonvolatile vars, use hp_get_nv and hp_set_nv to retrieve this struct */
typedef struct {
    int power_cycle_interval;  /* in seconds */
    unsigned char power_restore_policy;
    // poh counter
} host_power_globals_nv_t;

/* internal prototypes */
static int initiate_host_power_control(int action);
static int finish_host_power_control(const int item_id UNUSED, void* ctx);
static int power_cycle_power_on(const int item_id, void* ctx);
static void update_power_cycle_timer(void);

static void notify_power_control_subscribers(int control_action);
static void notify_power_state_subscribers(void);

static void hp_get_nv(host_power_globals_nv_t* nv_vars);
static void hp_set_nv(host_power_globals_nv_t* nv_vars);

/* vars */
static host_power_globals_t host_power_globals;
static host_power_globals_t* hpg;


int pp_bmc_host_power_init() {
    hpg = &host_power_globals;
    
    hpg->power_state_listeners = vector_new(NULL, 0, NULL);
    hpg->power_control_listeners = vector_new(NULL, 0, NULL);
 
    hpg->host_power_state = HOST_POWER_STATE_UNKNOWN;
    hpg->last_control_success = PP_SUC;
    hpg->host_power_control = HOST_POWER_CONTROL_NONE;

    hpg->power_cycle_to_hndl = -1;
    hpg->power_cycle_time = -1;

    pp_bmc_log_debug("[POWER] host power control initialized");

    return PP_SUC;
}

void pp_bmc_host_power_cleanup() {
    // no need to remove the listeners as we have a NULL free function    
    vector_delete(hpg->power_state_listeners);
    vector_delete(hpg->power_control_listeners);

    // stop the power_cycle timeout if running
    if (hpg->power_cycle_to_hndl != -1) {
        pp_select_remove_to(hpg->power_cycle_to_hndl);
    }

    pp_bmc_log_debug("[POWER] host power control cleaned up");
}

/* convenience function to retrieve nv power_cycle_interval */
static int get_power_cycle_interval(void) {
    host_power_globals_nv_t hpg_nv;
    hp_get_nv(&hpg_nv);
    return hpg_nv.power_cycle_interval;
}

int pp_bmc_host_power_get_state() {
    return hpg->host_power_state;
}

int pp_bmc_host_power_simplify_state(int state) {
    if (state == HOST_POWER_STATE_UNKNOWN) {
        return HOST_POWER_STATE_UNKNOWN;
    }
    
    if (HOST_POWER_STATE_IS_OFF(state)) {
        return HOST_POWER_STATE_LEGOFF;
    } else {
        return HOST_POWER_STATE_LEGON;
    }
}

int pp_bmc_host_power_state_is_on() {
    return HOST_POWER_STATE_IS_ON(hpg->host_power_state);
}

void pp_bmc_host_power_set_state(int new_state) {
    int old_state = hpg->host_power_state;
    int old_state_simple = pp_bmc_host_power_simplify_state(old_state);
    int new_state_simple = pp_bmc_host_power_simplify_state(new_state);
    hpg->host_power_state = new_state;
    pp_bmc_set_sys_acpi_pwr_state(new_state);
    
    if (old_state == new_state) {
        /* nothing has changed */
        return;
    }
    
    if (old_state_simple == new_state_simple)
    {
        unsigned char accept_new_state;
        /* (state cannot be unkown because unknown has no substates) */
        assert(old_state_simple != HOST_POWER_STATE_UNKNOWN);
        
        accept_new_state = 0;
        
        /* substate changed, check if new reading is more detailed than old one */
        if (old_state_simple == HOST_POWER_STATE_LEGON) {
            /* all power on states are sorted (by chance) */
            if (new_state < old_state) {
                accept_new_state = 1;
            }
        } else {
            /* old_state_simple = HOST_POWER_STATE_LEGOFF */
            switch (old_state) {
            case HOST_POWER_STATE_LEGOFF:
                /* everything is better than legacy off */
                accept_new_state = 1;
                break;
            case HOST_POWER_STATE_S45:
                /* only legacy off state is less detailed */
                if (new_state != HOST_POWER_STATE_LEGOFF) {
                    accept_new_state = 1;
                }
                break;
            case HOST_POWER_STATE_S5:
                if ((new_state == HOST_POWER_STATE_S4) ||
                    (new_state == HOST_POWER_STATE_MECHOFF) ||
                    (new_state == HOST_POWER_STATE_S5OVR))
                {
                    accept_new_state = 1;
                }
                break;
            case HOST_POWER_STATE_S4:
                if ((new_state == HOST_POWER_STATE_S5) ||
                    (new_state == HOST_POWER_STATE_MECHOFF) ||
                    (new_state == HOST_POWER_STATE_S5OVR))
                {
                    accept_new_state = 1;
                }
                break;
            }
        }
        
        if (accept_new_state) {
            /* we have a valid new substate */
            notify_power_state_subscribers();
            /* power control source is within system and is not of interest for us */
        }
    }
    else
    {
        /* major power state change */
        notify_power_state_subscribers();
            
        /* detect on/off power control events for notify_power_control */
        if (old_state_simple != new_state_simple)
        {
             /* try to extract the originating power control */
             switch(new_state_simple) {
             case HOST_POWER_STATE_LEGON:
                 notify_power_control_subscribers(HOST_POWER_CONTROL_ON);
                 break;
             case HOST_POWER_STATE_LEGOFF:
                 notify_power_control_subscribers(HOST_POWER_CONTROL_OFF);
                 break;
             case HOST_POWER_STATE_UNKNOWN:
                 /* oh oh, we moved from a known state to an unknown state, should not happen */
                 pp_bmc_log_error("[POWER] lost track of ACPI power state, state now unknown");
                 break;
             }
        }
    }
}

/**
 * initiate a host power control action.
 * Action may be on, off, cycle, reset
 * @returns PP_SUC
 * @returns PP_ERR if the control action could not be initiated
 *                 (for example if no actor exists)
 */
static int initiate_host_power_control(int action) {
    int duration = 0;   // pulse time in ms
    int actor_id = -1;  // actor id from host_sensors
    pp_tp_gpio_act_t* actor;

    assert (hpg->host_power_control == HOST_POWER_CONTROL_NONE);
    assert (hpg->power_cycle_to_hndl == -1);

    pp_bmc_log_debug("[POWER] power control event %d", action);
    
    if (hpg->host_power_control != HOST_POWER_CONTROL_NONE) {
        // very odd, should not happen
        pp_bmc_log_warn("[POWER] initiating power action while other action is active");
    }
    
    switch (action) {
    case HOST_POWER_CONTROL_ON:
        duration = 500;
        actor_id = PP_ACTOR_POWER;
        break;
    case HOST_POWER_CONTROL_CYCLE:
        action = HOST_POWER_CONTROL_OFF;
        duration = 5000;
        actor_id = PP_ACTOR_POWER;
        break;
    case HOST_POWER_CONTROL_OFF:
        assert (hpg->power_cycle_time == -1);
        
        duration = 5000;
        actor_id = PP_ACTOR_POWER;
        break;
    case HOST_POWER_CONTROL_RESET:
        duration = 1000;
        actor_id = PP_ACTOR_RESET;
        break;
    case HOST_POWER_CONTROL_SOFT_OFF:
        duration = 500;
        actor_id = PP_ACTOR_POWER;
        break;
    default:
        // must not happen
        assert(0);
        return PP_ERR;
    }
    
    actor = pp_bmc_host_sensor_get_gpio_actor(actor_id);
    
    if (actor == NULL) {
        pp_bmc_log_warn("[POWER] power actor or reset actor not registered");
        return PP_ERR;
    }

    // assert signal
    actor->set_gpio(actor, 1);
    
    // add to, put actor in context
    pp_select_add_to(duration, 0, finish_host_power_control,
		     (void*)(long)action);
    
    // declare this power control action
    hpg->host_power_control = action;
    
    pp_bmc_log_debug("[POWER] pulse initiated");
    return PP_SUC;
}

/**
 * timeout handler to release the signal that was set by initiate_host_power_control().
 */
static int finish_host_power_control(const int item_id UNUSED, void* ctx) {
    int action = (int)(long)ctx;
    pp_tp_gpio_act_t* actor = NULL;
    
    switch (action) {
    case HOST_POWER_CONTROL_ON:
    case HOST_POWER_CONTROL_OFF:
    case HOST_POWER_CONTROL_SOFT_OFF:
        actor = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_POWER);
        break;
    case HOST_POWER_CONTROL_RESET:
        actor = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_RESET);
        break;
    }
    
    // deassert signal
    actor->set_gpio(actor, 0);
    
    update_power_cycle_timer();
    
    // generate reset sensor event if no reset sensor is known
    if (action == HOST_POWER_CONTROL_RESET
	&& pp_bmc_host_sensor_get_sensor(PP_SENSOR_SOFT_RESET) == NULL) {
        pp_bmc_host_power_receive_reading(1, (void*)PP_SENSOR_SOFT_RESET);
        pp_bmc_host_power_receive_reading(0, (void*)PP_SENSOR_SOFT_RESET);
    }
    
    // check if desired state reached
    if (action == HOST_POWER_CONTROL_ON) {
        if (pp_bmc_host_power_simplify_state(pp_bmc_host_power_get_state()) == HOST_POWER_STATE_LEGON) {
            hpg->last_control_success = PP_SUC;
	} else {
            hpg->last_control_success = PP_ERR;
	}
    } else if (action == HOST_POWER_CONTROL_OFF || action == HOST_POWER_CONTROL_SOFT_OFF) {
        if (pp_bmc_host_power_simplify_state(pp_bmc_host_power_get_state()) == HOST_POWER_STATE_LEGOFF) {
            hpg->last_control_success = PP_SUC;
	} else {
            hpg->last_control_success = PP_ERR;
	}
    }
    
    hpg->host_power_control = HOST_POWER_CONTROL_NONE;
    pp_bmc_log_debug("[POWER] pulse ended");
    
    return PP_SUC;
}

/**
 * called cycle_interval seconds after power_cycle power off to turn the
 * host on again.
 */
static int power_cycle_power_on(const int item_id UNUSED, void* ctx UNUSED) {
    // initiate power on
    
    hpg->power_cycle_to_hndl = -1;
    hpg->power_cycle_time = -1;
    
    pp_bmc_host_power_control(HOST_POWER_CONTROL_ON, hpg->power_cycle_source, hpg->power_cycle_channel);
    
    return PP_SUC;
}

/**
 * check the power_cycle_power_on timer and start or stop it
 * according to the current value of power_cycle_time
 */
static void update_power_cycle_timer(void) {
    // stop timer
    if (hpg->power_cycle_to_hndl != -1) {
        pp_select_remove_to(hpg->power_cycle_to_hndl);
        hpg->power_cycle_to_hndl = -1;
    }
    if (hpg->power_cycle_time != -1) {
        // (re)start timer with actual value
        hpg->power_cycle_to_hndl = 
            pp_select_add_to((hpg->power_cycle_time*1000), 0, power_cycle_power_on, NULL);
    }
}


int pp_bmc_host_power_control(int action, int source, unsigned char channel) {
    pp_bmc_log_debug("[POWER] initiating power control action %d, source %d, state %d",
		     action, source, hpg->host_power_control);
    int rv;
    
    rv = PP_SUC;
    
    /* 
     * if the desired action can be executed depends on whether there is
     * another active power control action or not and if the desired action
     * can be matched with the active action
     */
    switch (hpg->host_power_control) {
         
    case HOST_POWER_CONTROL_USER:
        // very bad, we cannot predict what the user is doing
        // we better reject the action
        rv = PP_NOT_APPLICABLE;
        break;
        
    case HOST_POWER_CONTROL_NONE:
        if (hpg->power_cycle_to_hndl == -1) {
            // no active control, everything can be done (if it makes sense)
        
            switch (action) {
            case HOST_POWER_CONTROL_ON:
                if (pp_bmc_host_power_simplify_state(pp_bmc_host_power_get_state()) != HOST_POWER_STATE_LEGON) {
                    rv = initiate_host_power_control(action);
                }
                break;
            case HOST_POWER_CONTROL_CYCLE:
                if (pp_bmc_host_power_simplify_state(pp_bmc_host_power_get_state()) != HOST_POWER_STATE_LEGOFF) {
                    hpg->power_cycle_source = source;
                    hpg->power_cycle_channel = channel;
                    hpg->power_cycle_time = get_power_cycle_interval();
                    rv = initiate_host_power_control(action);
                }
                break;
            case HOST_POWER_CONTROL_RESET:
            case HOST_POWER_CONTROL_OFF:
            case HOST_POWER_CONTROL_SOFT_OFF:
                if (pp_bmc_host_power_simplify_state(pp_bmc_host_power_get_state()) != HOST_POWER_STATE_LEGOFF) {
                    rv = initiate_host_power_control(action);
                }
                break;
            default:
                assert(0);
                break;
            }
        } else {
            // a power cycle timer is active
            
            switch (action) {
            case HOST_POWER_CONTROL_ON:
                // shorten the power cycle time to 0
                hpg->power_cycle_time = 0;
                hpg->power_cycle_source = source;
                hpg->power_cycle_channel = channel;
                update_power_cycle_timer();
                rv = PP_SUC;
                break;
            case HOST_POWER_CONTROL_OFF:
            case HOST_POWER_CONTROL_SOFT_OFF:
                // we already are off, stop cycle to make sure that we stay off
                hpg->power_cycle_time = -1;
                update_power_cycle_timer();
                rv = PP_SUC;
                break;
            case HOST_POWER_CONTROL_CYCLE:
                // a power control cycle is running, no need to do anything
                hpg->power_cycle_source = source;
                hpg->power_cycle_channel = channel;
                rv = PP_SUC;
                break;
            case HOST_POWER_CONTROL_RESET:
                // impossible to solve the conflict
                rv = PP_NOT_APPLICABLE;
                break;
            default:
                assert(0);
                break;
            }
        }
        break;
        
    case HOST_POWER_CONTROL_ON:
        // host is currently powering up
        
        switch (action) {
        case HOST_POWER_CONTROL_ON:
        case HOST_POWER_CONTROL_RESET:
            // host power will be up in a moment, silently 'ignore' new command
            rv = PP_SUC;
            break;
        case HOST_POWER_CONTROL_CYCLE:
        case HOST_POWER_CONTROL_OFF:
        case HOST_POWER_CONTROL_SOFT_OFF:
            // impossible to solve the conflict
            rv = PP_NOT_APPLICABLE;
            break;
        default:
            assert(0);
            break;
        }
        break;
    
    case HOST_POWER_CONTROL_OFF:
        if (hpg->power_cycle_time == -1) {
            // host is powering down

            switch (action) {
            case HOST_POWER_CONTROL_RESET:
                // impossible to solve the conflict
                rv = PP_NOT_APPLICABLE;
                break;
            case HOST_POWER_CONTROL_OFF:
            case HOST_POWER_CONTROL_SOFT_OFF:
                // power will be off in a moment, silently 'ignore' new command
                rv = PP_SUC;
                break;
            case HOST_POWER_CONTROL_CYCLE:
                // power_cycle can overwrite power_off
                hpg->power_cycle_source = source;
                hpg->power_cycle_channel = channel;
                hpg->power_cycle_time = get_power_cycle_interval();
                rv = PP_SUC;
                break;
            case HOST_POWER_CONTROL_ON: 
                // power_cycle with 0 interval instead of power_off
                hpg->power_cycle_source = source;
                hpg->power_cycle_channel = channel;
                hpg->power_cycle_time = 0;
                rv = PP_SUC;
                break;
            default:
                assert(0);
                break;
            }
        }
        else {
            // host is powering down for power cycle
            
            switch (action) {
            case HOST_POWER_CONTROL_ON:
                // power up immediately after this power off
                hpg->power_cycle_time = 0;
                rv = PP_SUC;
                break;
            case HOST_POWER_CONTROL_OFF:
            case HOST_POWER_CONTROL_SOFT_OFF:
                // stay powered off after this power cycle off
                hpg->power_cycle_time = -1;
                rv = PP_SUC;
                break;
            case HOST_POWER_CONTROL_CYCLE:
                // this is what we're doing, ignore
                rv = PP_SUC;
                break;
            case HOST_POWER_CONTROL_RESET:
                // impossible to solve the conflict
                rv = PP_NOT_APPLICABLE;
                break;
            default:
                assert(0);
                break;
            }
        }
        break;
        
    case HOST_POWER_CONTROL_RESET:
        // host is resetting
        switch (action) {
        case HOST_POWER_CONTROL_ON:
        case HOST_POWER_CONTROL_RESET:
            // we will be on and reset in a moment
            rv = PP_SUC;
            break;
        case HOST_POWER_CONTROL_OFF:
        case HOST_POWER_CONTROL_SOFT_OFF:
        case HOST_POWER_CONTROL_CYCLE:
            // impossible to solve the conflict
            rv = PP_ERR;
            break;
        default:
            assert(0);
            break;
        }
        break;

    default:
        assert(0);
        break;
    }
    
    if (rv == PP_SUC) {
#if defined (PP_FEAT_IPMI_SERVER_CHASSIS_CMDS)
	pp_bmc_dev_chassis_set_restart_cause(action, source, channel);
#endif
	pp_bmc_log_debug("[POWER] control successful");
    }
    if (rv == PP_ERR) {
	pp_bmc_log_error("[POWER] control unsuccessful");
    }
    if (rv == PP_NOT_APPLICABLE) {
	pp_bmc_log_warn("[POWER] control rejected");
    }
    
    return rv;
}

int pp_bmc_host_power_control_result() {
    return hpg->last_control_success;
}


void pp_bmc_host_power_state_subscribe(pp_bmc_host_power_state_hndl_t power_state_hndl) {
    vector_add(hpg->power_state_listeners, power_state_hndl);
}

int pp_bmc_host_power_state_unsubscribe(pp_bmc_host_power_state_hndl_t power_state_hndl) {
    u_int i;
    for (i = 0; i < vector_size(hpg->power_state_listeners); ++i) {
	pp_bmc_host_power_state_hndl_t h;
	h = (pp_bmc_host_power_state_hndl_t)vector_get(hpg->power_state_listeners, i);
        if (h == power_state_hndl) {
            vector_remove_dont_delete(hpg->power_state_listeners, i);
	    return PP_SUC;
        }
    }
    return PP_ERR;
}


void pp_bmc_host_power_control_subscribe(pp_bmc_host_power_control_hndl_t
					 power_control_hndl) {
    vector_add(hpg->power_control_listeners, power_control_hndl);
}

int pp_bmc_host_power_control_unsubscribe(pp_bmc_host_power_control_hndl_t power_control_hndl) {
    u_int i;
    for (i = 0; i < vector_size(hpg->power_control_listeners); ++i) {
	pp_bmc_host_power_control_hndl_t h;
	h = (pp_bmc_host_power_control_hndl_t)vector_get(hpg->power_control_listeners, i);
        if (h == power_control_hndl) {
            vector_remove_dont_delete(hpg->power_control_listeners, i);
            return PP_SUC;
        }
    }
    return PP_ERR;
}

static void notify_power_control_subscribers(int control_action) {
    pp_bmc_host_power_control_hndl_t cb;
    unsigned int i;
    
    for (i=0; i<vector_size(hpg->power_control_listeners); i++) {
        cb = vector_get(hpg->power_control_listeners, i);
        cb(control_action);
    }
}

static void notify_power_state_subscribers(void) {
    pp_bmc_host_power_state_hndl_t cb;
    unsigned int i;
    
    for (i=0; i<vector_size(hpg->power_state_listeners); i++) {
        cb = vector_get(hpg->power_state_listeners, i);
        cb(hpg->host_power_state);
    }
}


void pp_bmc_host_power_policy_set(unsigned char policy) {
    host_power_globals_nv_t nv_vars;
    
    hp_get_nv(&nv_vars);
    nv_vars.power_restore_policy = policy;
    hp_set_nv(&nv_vars);
}

unsigned char pp_bmc_host_power_policy_get() {
    host_power_globals_nv_t nv_vars;
    
    hp_get_nv(&nv_vars);
    return nv_vars.power_restore_policy;
}

void pp_bmc_host_power_cycle_interval_set(unsigned char interval) {
    host_power_globals_nv_t hpg_nv;
    
    hp_get_nv(&hpg_nv);
    hpg_nv.power_cycle_interval = interval;
    hp_set_nv(&hpg_nv);
}

/* get power supply status */
int pp_bmc_host_power_supply_status() {
    pp_tp_sensdev_t* sens;
    if (pp_bmc_host_sensor_get_sensor(PP_SENSOR_POWER_INTERLOCK) != NULL) {
        sens = pp_bmc_host_sensor_get_sensor(PP_SENSOR_POWER_INTERLOCK);
        if (pp_tp_sensdev_get_reading(sens) == 1) {
            return HOST_POWER_SUPPLY_INTERLOCK;
        }
    }

    sens = pp_bmc_host_sensor_get_sensor(PP_SENSOR_POWER_HEALTH);
    if (sens != NULL && pp_tp_sensdev_get_reading(sens) == 1) 
        return HOST_POWER_SUPPLY_FAULT;
        
    return HOST_POWER_SUPPLY_OK;
}

int pp_bmc_host_power_interlock_exists() {
    if (pp_bmc_host_sensor_get_sensor(PP_SENSOR_POWER_INTERLOCK) == NULL)
        return PP_ERR;
    else 
        return PP_SUC;
}

/**
 * Get last power event byte as defined in table 28-3, byte 3
 */
unsigned char pp_bmc_host_power_get_last_event() {
    // we cannot really find out anything for this event
    // TODO: eventually, we could implement bit 4: last power on entered via IPMI command
    
    // tweb: MSI return 0x01 for that... for the moment, do that as well...
//    return 1;
    return 0;
}

void pp_bmc_host_power_receive_reading(int reading, void* ctx) {
    pp_tp_gpio_act_t* act = NULL;
    int sensor_id = (int)(long)ctx;
    int new_power_state;
    
    switch (sensor_id) {
    case PP_SENSOR_ACPI_POWER_STATE:
        /* acpi_power_state is a discrete IPMI sensor that returns
	 * a 16 bit bitmask */
	if (reading > 0) {
	    new_power_state = ffs(reading) - 1;
	} else {
	    new_power_state = HOST_POWER_STATE_UNKNOWN;
	}
        /* power_set_state will notify the power state and
	 * control subscribers */
        pp_bmc_host_power_set_state(new_power_state);
        
	break;
    case PP_SENSOR_SOFT_RESET:
        /* power_state does not change, notify power_control_subscribers only */
        if (reading == 1) {
            notify_power_control_subscribers(HOST_POWER_CONTROL_RESET);
        }
        break;
    case PP_SENSOR_POWER_BTN:
    case PP_SENSOR_RESET_BTN:
        /* frontpanel button presses */
        if (sensor_id == PP_SENSOR_POWER_BTN)
            act = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_POWER);
        else if (sensor_id == PP_SENSOR_RESET_BTN)
            act = pp_bmc_host_sensor_get_gpio_actor(PP_ACTOR_RESET);

        if (act != NULL) {
            // the user is pressing buttons that are not disabled by host_hardware
            if (reading == 1) {
                if (hpg->host_power_control == HOST_POWER_CONTROL_NONE) {
                    if (hpg->power_cycle_to_hndl != -1) {
                        hpg->power_cycle_time = -1;
                        update_power_cycle_timer();
                    }
                    // okay, we can forward this button press
#ifndef PRODUCT_SMIDC // Supermicro: Reset/Power button sense may stuck to GND during Power Off!
                    hpg->host_power_control = HOST_POWER_CONTROL_USER;
#endif

#if !defined(PRODUCT_SMIDC) && !defined(PP_FEAT_OPMA_HW) // not needed for OPMA
                    act->set_gpio(act, 1);
#endif
                }
            } else {
#ifndef PRODUCT_SMIDC // Supermicro: Reset only 5ms, so button down signal might be missed
                if (hpg->host_power_control ==HOST_POWER_CONTROL_USER) {
#endif
                    hpg->host_power_control = HOST_POWER_CONTROL_NONE;
#if !defined(PRODUCT_SMIDC) && !defined(PP_FEAT_OPMA_HW) // not needed for OPMA
                    act->set_gpio(act,0);
#endif

                    /* reset events will not be detected by power state changes */
                    if (sensor_id == PP_SENSOR_RESET_BTN) {
                        /* reset events will be detected by the SOFT_RESET sensor if we have one */
                        if (pp_bmc_host_sensor_get_sensor(PP_SENSOR_SOFT_RESET) == NULL) {
                            /* no SOFT_RESET sensor to detect the event, notify directly */
                            notify_power_control_subscribers(HOST_POWER_CONTROL_RESET);
                        }
                    }
#ifndef PRODUCT_SMIDC // see above
                }
#endif
            }
        }
        break;
    default:
        pp_bmc_log_error("[POWER] unknown sensor event from sensor %d", sensor_id);
        break;
    }
}


static void hp_get_nv(host_power_globals_nv_t* nv_vars) {
    char* s;
    int i;
    
    pp_cfg_get_nodflt(&s, "bmc.power.restore_policy");
    if (strcmp(s, "on") == 0) {
        nv_vars->power_restore_policy = HOST_POWER_POLICY_ON;
    } else
    if (strcmp(s, "off") == 0) {
        nv_vars->power_restore_policy = HOST_POWER_POLICY_OFF;
    } else
    if (strcmp(s, "restore") == 0) {
        nv_vars->power_restore_policy = HOST_POWER_POLICY_RESTORE;
    } else
    if (strcmp(s, "unknown") == 0) {
        nv_vars->power_restore_policy = HOST_POWER_POLICY_UNKNOWN;
    } else {
        nv_vars->power_restore_policy = HOST_POWER_POLICY_UNKNOWN;
        assert(0);
    }
    
    pp_cfg_get_int_nodflt(&i, "bmc.power.cycle_interval");
    nv_vars->power_cycle_interval = i;
}

static void hp_set_nv(host_power_globals_nv_t* nv_vars) {
    switch (nv_vars->power_restore_policy) {
        case HOST_POWER_POLICY_ON:
            pp_cfg_set("on", "bmc.power.restore_policy");
            break;
        case HOST_POWER_POLICY_OFF:
            pp_cfg_set("off", "bmc.power.restore_policy");
            break;
        case HOST_POWER_POLICY_RESTORE:
            pp_cfg_set("restore", "bmc.power.restore_policy");
            break;
        case HOST_POWER_POLICY_UNKNOWN:
            pp_cfg_set("unknown", "bmc.power.restore_policy");
            break;
    }
    
    pp_cfg_set_int(nv_vars->power_cycle_interval, "bmc.power.cycle_interval");
}

