/**
 * host_sensors.h
 * 
 * Registry of sensors and actors for host_power, host_hardware and other
 * libpp_hardware files. Provides a callback function for libpp_sensor
 * and automatically registers with relevant sensors. Sensor events are
 * automatically forwarded to interested (hardwired) consumers.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */
#include <string.h>
#include <pp/base.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/host_sensors.h>
#include <pp/bmc/host_hardware.h>
#include <pp/bmc/host_power.h>
#include <pp/bmc/topo_classes.h>
#include <pp/bmc/tp_sensdev.h>
#include <pp/bmc/tp_gpio_act.h>
#include <pp/bmc/tp_cond.h>


/* internal actor and sensor registry */
static pp_tp_sensdev_t*    sensor_registry[PP_SENSOR_MAX];
static pp_tp_actor_t*      actor_registry[PP_ACTOR_MAX];
static pp_tp_pwm_act_t*    pwm_actor_registry[PP_PWM_ACTOR_MAX];
static pp_tp_cond_t*       cond_registry[PP_BMC_COND_MAX];
static pp_tp_i2c_comdev_t* i2c_comdev_registry[PP_BMC_I2C_COMDEV_MAX];
static pp_tp_rs485_chip_t* rs485_chip_registry[PP_BMC_RS485_CHIP_MAX];
#ifdef KIRA_RPC
static pp_tp_ctrl_t*       ctrl_registry[PP_BMC_CTRL_MAX];
#endif /* KIRA_RPC */
/* subscriptions for libpp_hardware (host_power, host_hardware) */
static pp_tp_sensdev_subscriber_t sensor_sub[PP_SENSOR_MAX];



static void host_sens_hw_receive_reading(pp_tp_sensdev_subscriber_t* o,
					 pp_tp_sensdev_t* source UNUSED,
					 int reading)
{
    // find my index
    int idx = ((char*)o - (char*)sensor_sub) / sizeof(pp_tp_sensdev_subscriber_t);
    pp_bmc_hardware_receive_reading(reading, (void*)(long)idx);
}

static void host_sens_power_receive_reading(pp_tp_sensdev_subscriber_t* o,
					    pp_tp_sensdev_t* source UNUSED,
        				    int reading)
{
    // find my index
    int idx = ((char*)o - (char*)sensor_sub) / sizeof(pp_tp_sensdev_subscriber_t);
    pp_bmc_host_power_receive_reading(reading, (void*)(long)idx);
}

static void bmc_register_sensor(pp_tp_sensdev_t* obj) {
    int idx;
    int reg;
    unsigned char* id_str;
    
    id_str = ((pp_tp_obj_t*)obj)->id;

    idx = -1;
    
    reg = 0;
    /* set reg to register at the sensor                  *
     * reg=0 dont register sensor subscriber              *
     * reg=1 registers at hardware_lib (e.g. buttons)     *
     * reg=2 registers at power_lib (e.g. powersensors)   */ 

    /* please keep this if in the same order as the list in host_sensors.h */
    if (strcmp("power_btn", id_str) == 0) {
        idx = PP_SENSOR_POWER_BTN;
        reg = 1;
    } else if (strcmp("reset_btn", id_str) == 0) {
        idx = PP_SENSOR_RESET_BTN;
        reg = 1;
    } else if (strcmp("sleep_btn", id_str) == 0) {
        idx = PP_SENSOR_SLEEP_BTN;
        reg = 1;
    } else if (strcmp("id_btn", id_str) == 0) {
        idx = PP_SENSOR_ID_BTN;
        reg = 1;
    } else if (strcmp("diag_btn", id_str) == 0) {
        idx = PP_SENSOR_DIAG_BTN;
        reg = 1;
    } else if (strcmp("soft_reset", id_str) == 0) {
        idx = PP_SENSOR_SOFT_RESET;
        reg = 2;
    } else if (strcmp("chas_intr", id_str) == 0) {
        idx = PP_SENSOR_CHASSIS_INTRUSION;
    } else if (strcmp("drv_health", id_str) == 0) {
        idx = PP_SENSOR_SOFT_RESET;
    } else if (strcmp("fan_health", id_str) == 0) {
        idx = PP_SENSOR_SOFT_RESET;
    } else if (strcmp("ps_health", id_str) == 0) {
        idx = PP_SENSOR_POWER_HEALTH;
    } else if (strcmp("ps_ilock", id_str) == 0) {
        idx = PP_SENSOR_POWER_INTERLOCK;
    } else if (strcmp("nmi_btn", id_str) == 0) {
	idx = PP_SENSOR_NMI_BTN;
	reg = 1;
    } else if (strcmp("acpi_power_state", id_str) == 0) {
	idx = PP_SENSOR_ACPI_POWER_STATE;
	reg = 2;
#ifdef PRODUCT_ICPMMD
    } else if (strcmp("psu0_sens_ac1_ok", id_str) == 0) {
	idx = PP_SENSOR_PSU0_AC1_OK;
    } else if (strcmp("psu0_sens_ac2_ok", id_str) == 0) {
	idx = PP_SENSOR_PSU0_AC2_OK;
    } else if (strcmp("psu1_sens_ac1_ok", id_str) == 0) {
	idx = PP_SENSOR_PSU1_AC1_OK;
    } else if (strcmp("psu1_sens_ac2_ok", id_str) == 0) {
	idx = PP_SENSOR_PSU1_AC2_OK;
    } else if (strcmp("psu2_sens_ac1_ok", id_str) == 0) {
	idx = PP_SENSOR_PSU2_AC1_OK;
    } else if (strcmp("psu2_sens_ac2_ok", id_str) == 0) {
	idx = PP_SENSOR_PSU2_AC2_OK;
    } else if (strcmp("psu3_sens_ac1_ok", id_str) == 0) {
	idx = PP_SENSOR_PSU3_AC1_OK;
    } else if (strcmp("psu3_sens_ac2_ok", id_str) == 0) {
	idx = PP_SENSOR_PSU3_AC2_OK;
    } else if (strcmp("ibsw0_sens_slotid", id_str) == 0) {
	idx = PP_SENSOR_IBSW0_SLOTID;
    } else if (strcmp("ibsw0_sens_pwr_good", id_str) == 0) {
	idx = PP_SENSOR_IBSW0_POWER_OK;
    } else if (strcmp("ibsw1_sens_slotid", id_str) == 0) {
	idx = PP_SENSOR_IBSW1_SLOTID;
    } else if (strcmp("ibsw1_sens_pwr_good", id_str) == 0) {
	idx = PP_SENSOR_IBSW1_POWER_OK;
#endif /* PRODUCT_ICPMMD */
    } else {
        // this is a sensor that is not of interest
        return;
    }
    
    // new valid sensor
    if ((sensor_registry[idx]) == NULL) {
        pp_tp_obj_duplicate((pp_tp_obj_t*)obj);
        sensor_registry[idx] = obj;
        
        // register at this sensor
        if (reg == 1) {
	    assert(sensor_sub[idx].recv_reading == NULL);
	    sensor_sub[idx].recv_reading = host_sens_hw_receive_reading;
	    pp_bmc_tp_sensdev_subscribe(obj, &sensor_sub[idx]);
        }
	else if (reg == 2) {
	    assert(sensor_sub[idx].recv_reading == NULL);
	    sensor_sub[idx].recv_reading = host_sens_power_receive_reading;
	    pp_bmc_tp_sensdev_subscribe(obj, &sensor_sub[idx]);
        }
    } else {
        pp_bmc_log_debug("[HOST] cannot register sensor %s twice", id_str);
    }
}

pp_tp_sensdev_t* pp_bmc_host_sensor_get_sensor(int type) {
    assert (type < PP_SENSOR_MAX);
    return sensor_registry[type];
}

/* stores actors and pwm_actors, but in different storages */
static void bmc_register_actor(pp_tp_actor_t* obj) {
    char* id = ((pp_tp_obj_t*)obj)->id;
    int idx;

    idx = -1;
    if (strcmp("power_out", id) == 0) {
        idx = PP_ACTOR_POWER;
    } else if (strcmp("reset_out", id) == 0) {
        idx = PP_ACTOR_RESET;
    } else if (strcmp("nmi_out", id) == 0) {
        idx = PP_ACTOR_NMI;
    } else if (strcmp("smi_out", id) == 0) {
        idx = PP_ACTOR_SMI;
    } else if (strcmp("stdby_out", id) == 0) {
        idx = PP_ACTOR_STANDBY;
    } else if (strcmp("diag_int", id) == 0) {
        idx = PP_ACTOR_DIAG_INT;
    } else if (strcmp("id_led", id) == 0) {
        idx = PP_ACTOR_ID_LED;
    } else if (strcmp("smux" , id) == 0) {
	idx = PP_ACTOR_SMUX;
    } else if (strcmp("clear_cmos" , id) == 0) {
	idx = PP_ACTOR_CLEAR_CMOS;
    } else if (strcmp("clr_intrusion" , id) == 0) {
	idx = PP_ACTOR_CLR_INTRUSION;
    } else if (strcmp("heartbeat" , id) == 0) {
	idx = PP_ACTOR_HEARTBEAT;
    }
#if defined(LARA_KIMMSI) || defined(PP_FEAT_OPMA_HW)
    else if (strncmp("pwm", id, 3) == 0) {
        char* temp1;
        char* temp2;
        int n;
        
        temp1 = id+3;
        n = strtol(temp1, &temp2, 0);
	if ((*temp1 != '\0') && (*temp2 == '\0')) {
            // string contains chars and was completely parsed, number is ok
            if ((n < PP_PWM_ACTOR_MAX) && (pwm_actor_registry[n] == NULL)) { 
		pp_tp_obj_duplicate((pp_tp_obj_t*)obj);
                pwm_actor_registry[n] = (pp_tp_pwm_act_t*)obj;
            } else {
                pp_bmc_log_debug("[HOST] cannot register pwm actor %s twice", id);
	    }
	} else {
            // remaining chars, not a pwmXXX identifier, ignore
	    pp_bmc_log_debug("[HOST] Invalid actor identifier: %s", ((pp_tp_obj_t*)obj)->id);
	}
    }
#endif /* LARA_KIMMSI || PP_FEAT_OPMA_HW */
#ifdef PP_FEAT_OPMA_HW
    else if (strcmp("local_lock", id) == 0) {
	idx = PP_ACTOR_LOCAL_LOCK;
    } else if (strcmp("mcard_id", id) == 0) {
	idx = PP_ACTOR_MCARD_ID;
    }
#endif /* PP_FEAT_OPMA_HW */
#ifdef PRODUCT_ICPMMD
    else if (strcmp("cfu0_act_fan1_ok", id) == 0) {
	idx = PP_ACTOR_CFU0_FAN1_OK;
    } else if (strcmp("cfu0_act_fan2_ok", id) == 0) {
	idx = PP_ACTOR_CFU0_FAN2_OK;
    } else if (strcmp("cfu1_act_fan1_ok", id) == 0) {
	idx = PP_ACTOR_CFU1_FAN1_OK;
    } else if (strcmp("cfu1_act_fan2_ok", id) == 0) {
	idx = PP_ACTOR_CFU1_FAN2_OK;
    } else if (strcmp("cfu2_act_fan1_ok", id) == 0) {
	idx = PP_ACTOR_CFU2_FAN1_OK;
    } else if (strcmp("cfu2_act_fan2_ok", id) == 0) {
	idx = PP_ACTOR_CFU2_FAN2_OK;
    } else if (strcmp("cfu3_act_fan1_ok", id) == 0) {
	idx = PP_ACTOR_CFU3_FAN1_OK;
    } else if (strcmp("cfu3_act_fan2_ok", id) == 0) {
	idx = PP_ACTOR_CFU3_FAN2_OK;
    } else if (strcmp("psu0_act_fan_ok", id) == 0) {
	idx = PP_ACTOR_PSU0_FAN_OK;
    } else if (strcmp("psu0_act_dc_ok", id) == 0) {
	idx = PP_ACTOR_PSU0_DC_OK;
    } else if (strcmp("psu1_act_fan_ok", id) == 0) {
	idx = PP_ACTOR_PSU1_FAN_OK;
    } else if (strcmp("psu1_act_dc_ok", id) == 0) {
	idx = PP_ACTOR_PSU1_DC_OK;
    } else if (strcmp("psu2_act_fan_ok", id) == 0) {
	idx = PP_ACTOR_PSU2_FAN_OK;
    } else if (strcmp("psu2_act_dc_ok", id) == 0) {
	idx = PP_ACTOR_PSU2_DC_OK;
    } else if (strcmp("psu3_act_fan_ok", id) == 0) {
	idx = PP_ACTOR_PSU3_FAN_OK;
    } else if (strcmp("psu3_act_dc_ok", id) == 0) {
	idx = PP_ACTOR_PSU3_DC_OK;
    } else if (strcmp("ibsw0_act_fault_led", id) == 0) {
	idx = PP_ACTOR_IBSW0_FAULT_LED;
    } else if (strcmp("ibsw0_act_good_led", id) == 0) {
	idx = PP_ACTOR_IBSW0_GOOD_LED;
    } else if (strcmp("ibsw0_act_reset", id) == 0) {
	idx = PP_ACTOR_IBSW0_RESET;
    } else if (strcmp("ibsw1_act_fault_led", id) == 0) {
	idx = PP_ACTOR_IBSW1_FAULT_LED;
    } else if (strcmp("ibsw1_act_good_led", id) == 0) {
	idx = PP_ACTOR_IBSW1_GOOD_LED;
    } else if (strcmp("ibsw1_act_reset", id) == 0) {
	idx = PP_ACTOR_IBSW1_RESET;
    } else if (strstr(id, "_pwm_fans") != NULL) {
	/* HACK */
	static u_int pwm_idx = 0;
	if (pwm_idx < PP_PWM_ACTOR_MAX && pwm_actor_registry[pwm_idx] == NULL) { 
	    pp_tp_obj_duplicate((pp_tp_obj_t*)obj);
	    pwm_actor_registry[pwm_idx++] = (pp_tp_pwm_act_t*)obj;
	} else {
	    pp_bmc_log_debug("[HOST] cannot register pwm actor %s twice", id);
	}
    }
#endif /* PRODUCT_ICPMMD */
#ifdef KIRA_RPC
    /*
     * examples
     * r00_act_state
     */
    int base, no;
    char* eptr;
    const char str_act[] = "_act_";
    const char str_sta[] = "state";
    
    if (strncmp(str_act, &id[3], sizeof(str_act)-1) == 0) {
	if (strncmp(str_sta, &id[8], sizeof(str_sta)-1) == 0) {
	    base = PP_ACTOR_R0_STATE;
	} else {
	    base = -1;
	}
	
	if (base >= 0) {
	    /* found the pattern, try to get receptacle number */
	    no = strtol(&id[1], &eptr, 10);
	    if (&id[1] != eptr && no >= 0 && no < 24) {
		idx = base + no;
		pp_bmc_log_debug("RPC register actor %s  -> idx %d", id,idx);
	    }
	}
    }
#endif /* KIRA_RPC */

    if (idx == -1) {		
        // this is an actor that is not of interest or an pwm actor that was treated separately
        return;
    }

    if (actor_registry[idx] == NULL) {
        // new valid actor
        pp_tp_obj_duplicate((pp_tp_obj_t*)obj);
        actor_registry[idx] = obj;
    } else {
        pp_bmc_log_debug("[HOST] cannot register actor %s twice", id);
    }
}

pp_tp_actor_t* pp_bmc_host_sensor_get_actor(int type) {
    assert (type < PP_ACTOR_MAX);
    return actor_registry[type];
}

pp_tp_gpio_act_t* pp_bmc_host_sensor_get_gpio_actor(int type) {
    pp_tp_actor_t* a = pp_bmc_host_sensor_get_actor(type);
    if (a != NULL && PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_ACT, a)) {
        return (pp_tp_gpio_act_t*)a;
    }
    return NULL;
}

pp_tp_pwm_act_t* pp_bmc_host_sensor_get_pwm_actor(int idx) {
    assert (idx < PP_PWM_ACTOR_MAX);
    return pwm_actor_registry[idx];
}


static void bmc_register_condition(pp_tp_cond_t* obj) {
    int idx  = -1;

#ifdef LARA_KIMMSI
    if (strcmp("cpu0_cond", pp_tp_cond_to_string(obj)) == 0) {
        idx = PP_BMC_COND_CPU0_PRESENCE;
    } else if (strcmp("cpu1_cond", pp_tp_cond_to_string(obj)) == 0) {
        idx = PP_BMC_COND_CPU1_PRESENCE;
    }
#endif /* LARA_KIMMSI */

#ifdef PRODUCT_SMIDC
    if (strcmp("i2c_access", pp_tp_cond_to_string(obj)) == 0) {
        idx = PP_BMC_COND_I2C_ACCESS;
    }
#endif

    if (idx == -1) {
        // this is an condition that is not of interest
        return;
    }
    
    // new valid condition
    if ((cond_registry[idx]) == NULL) {
        cond_registry[idx] = pp_tp_cond_duplicate(obj);
    } else {
        pp_bmc_log_debug("[HOST] cannot register condition %s twice",
			 pp_tp_cond_to_string(obj));
    }
}

pp_tp_cond_t* pp_bmc_host_get_condition(int type) {
    assert (type < PP_BMC_COND_MAX);
    return cond_registry[type];
}    


static void bmc_register_i2c_comdev(pp_tp_i2c_comdev_t* obj) {
    char * id_str = ((pp_tp_obj_t*)obj)->id;
    int idx  = -1;
    if (strcmp("ipmb0_i2c_bus", id_str) == 0) {
        idx = PP_BMC_I2C_COMDEV_IPMB0;
    } else if (strcmp("ipmb1_i2c_bus", id_str) == 0) {
        idx = PP_BMC_I2C_COMDEV_IPMB1;
#ifdef PRODUCT_ICPMMD
    } else if (strcmp("gesw_i2c_bus", id_str) == 0) {
        idx = PP_BMC_I2C_COMDEV_GESW;
#endif /* PRODUCT_ICPMMD */
    } else {
        return;
    }
    
    if ((i2c_comdev_registry[idx]) == NULL) {
	i2c_comdev_registry[idx] = pp_tp_i2c_comdev_duplicate(obj);
    } else {
        pp_bmc_log_debug("[HOST] cannot register i2c_comdev %s twice", id_str);
    }
}

pp_tp_i2c_comdev_t* pp_bmc_host_get_i2c_comdev(int type) {
    assert (type < PP_BMC_I2C_COMDEV_MAX);
    return i2c_comdev_registry[type];
}

static void bmc_register_rs485_chip(pp_tp_rs485_chip_t* obj) {
    char * id_str = ((pp_tp_obj_t*)obj)->id;
    int idx  = -1;
    if (strcmp("b0_chip", id_str) == 0) {
        idx = PP_BMC_RS485_CHIP_RPC0;
    } else if (strcmp("b1_chip", id_str) == 0) {
        idx = PP_BMC_RS485_CHIP_RPC1;
    } else if (strcmp("b2_chip", id_str) == 0) {
        idx = PP_BMC_RS485_CHIP_RPC2;
    } else if (strcmp("b3_chip", id_str) == 0) {
        idx = PP_BMC_RS485_CHIP_RPC3;
    } else if (strcmp("b4_chip", id_str) == 0) {
        idx = PP_BMC_RS485_CHIP_RPC4;
    } else {
        return;
    }

    assert (idx < PP_BMC_RS485_CHIP_MAX);
    if ((rs485_chip_registry[idx]) == NULL) {
	rs485_chip_registry[idx] = pp_tp_rs485_chip_duplicate(obj);
    } else {
        pp_bmc_log_debug("[HOST] cannot register rs485-chip %s twice", id_str);
    }
}

pp_tp_rs485_chip_t* pp_bmc_host_get_rs485_chip(int type) {
    assert (type < PP_BMC_RS485_CHIP_MAX);
    return rs485_chip_registry[type];
}

#ifdef KIRA_RPC
/*
 * examples
 * r00_ctrl_led
 */
static void bmc_register_ctrl(pp_tp_ctrl_t* obj) {
    char * id = ((pp_tp_obj_t*)obj)->id;
    int idx  = -1;
    int base, no;
    char* eptr;
    const char str_ctrl[] = "_ctrl_";
    const char str_led[] = "led";
    
    if (strncmp(str_ctrl, &id[3], sizeof(str_ctrl)-1) == 0) {
	if (strncmp(str_led, &id[9], sizeof(str_led)-1) == 0) {
	    base = PP_BMC_CTRL_R0_LED;
	} else {
	    base = -1;
	}
	
	if (base >= 0) {
	    /* found the pattern, try to get receptacle number */
	    no = strtol(&id[1], &eptr, 10);
	    if (&id[1] != eptr && no >= 0 && no < 24) {
		idx = base + no;
		pp_bmc_log_debug("RPC register controller %s  -> idx %d",
                                 id, idx);
	    }
	}
    }
    if (idx < 0) return; // nothing matched
    
    assert (idx < PP_BMC_CTRL_MAX);
    if ((ctrl_registry[idx]) == NULL) {
	ctrl_registry[idx] = pp_tp_ctrl_duplicate(obj);
    } else {
        pp_bmc_log_debug("[HOST] cannot register controller %s twice", id);
    }
}

pp_tp_ctrl_t* pp_bmc_host_get_ctrl(int type) {
    assert (type < PP_BMC_CTRL_MAX);
    return ctrl_registry[type];
}
#endif /* KIRA_RPC */

void pp_bmc_host_sensor_create(pp_tp_obj_t* obj) {
    // note: always check more specific classes before more generic classes!
    if (PP_TP_OBJ_IS_TYPE(PP_TP_COND, obj)) {
        bmc_register_condition((pp_tp_cond_t*)obj);
    } else if (PP_TP_OBJ_IS_TYPE(PP_TP_SENS_DEV, obj)) {
        bmc_register_sensor((pp_tp_sensdev_t*)obj);
    } else if (PP_TP_OBJ_IS_TYPE(PP_TP_ACTOR, obj)) {
        bmc_register_actor((pp_tp_actor_t*)obj);
    } else if (PP_TP_OBJ_IS_TYPE(PP_TP_I2C_COM_DEV, obj)) {
	bmc_register_i2c_comdev((pp_tp_i2c_comdev_t*)obj);
    } else if (PP_TP_OBJ_IS_TYPE(PP_TP_RS485_CHIP, obj)) {
	bmc_register_rs485_chip((pp_tp_rs485_chip_t*)obj);
#ifdef KIRA_RPC
    } else if (PP_TP_OBJ_IS_TYPE(PP_TP_CTRL, obj)) {
	bmc_register_ctrl((pp_tp_ctrl_t*)obj);
#endif /* KIRA_RPC */
    }
}

int pp_bmc_host_sensor_init() {
    memset(sensor_registry, 0, sizeof(sensor_registry));
    memset(actor_registry, 0, sizeof(actor_registry));
    memset(pwm_actor_registry, 0, sizeof(pwm_actor_registry));
    memset(cond_registry, 0, sizeof(cond_registry));
    memset(i2c_comdev_registry, 0, sizeof(i2c_comdev_registry));
    memset(rs485_chip_registry, 0, sizeof(rs485_chip_registry));
#ifdef KIRA_RPC
    memset(ctrl_registry, 0, sizeof(ctrl_registry));
#endif /* KIRA_RPC */
    memset(sensor_sub, 0, sizeof(sensor_sub));
    pp_bmc_log_debug("[HOST] sensor initialized");
    return PP_SUC;
}

void pp_bmc_host_sensor_cleanup() {
    int i;
    for (i=0; i<PP_SENSOR_MAX; i++) {
        if (sensor_registry[i] != NULL) {
            if (sensor_sub[i].recv_reading != NULL) {
		pp_bmc_tp_sensdev_unsubscribe(sensor_registry[i], 
					      &sensor_sub[i]);
		sensor_sub[i].recv_reading = NULL;
            }
            pp_tp_obj_release((pp_tp_obj_t*)sensor_registry[i]);
            sensor_registry[i] = NULL;
        }
    }
    for (i=0; i<PP_ACTOR_MAX; i++) {
        if (actor_registry[i] != NULL) {
            pp_tp_obj_release((pp_tp_obj_t*)actor_registry[i]);
            actor_registry[i] = NULL;
        }
    }
    for (i=0; i<PP_PWM_ACTOR_MAX; i++) {
        if (pwm_actor_registry[i] != NULL) {
            pp_tp_obj_release((pp_tp_obj_t*)pwm_actor_registry[i]);
            pwm_actor_registry[i] = NULL;
        }
    }
    for (i=0; i<PP_BMC_COND_MAX; i++) {
        if (cond_registry[i] != NULL) {
            pp_tp_obj_release((pp_tp_obj_t*)cond_registry[i]);
            cond_registry[i] = NULL;
        }
    }
    for (i=0; i<PP_BMC_I2C_COMDEV_MAX; i++) {
        if (i2c_comdev_registry[i] != NULL) {
            pp_tp_obj_release((pp_tp_obj_t*)i2c_comdev_registry[i]);
            i2c_comdev_registry[i] = NULL;
        }
    }
    for (i=0; i<PP_BMC_RS485_CHIP_MAX; i++) {
        if (rs485_chip_registry[i] != NULL) {
            pp_tp_obj_release((pp_tp_obj_t*)rs485_chip_registry[i]);
            rs485_chip_registry[i] = NULL;
        }
    }
#ifdef KIRA_RPC
    for (i=0; i<PP_BMC_CTRL_MAX; i++) {
        if (ctrl_registry[i] != NULL) {
            pp_tp_obj_release((pp_tp_obj_t*)ctrl_registry[i]);
            ctrl_registry[i] = NULL;
        }
    }
#endif /* KIRA_RPC */
    pp_bmc_log_debug("[HOST] sensor cleanup");
}
