/**
 * host_power.h
 *
 * Provides an abstraction to the host power control. Offers functions to
 * - control the host power
 * - track host power actions
 * - query the host power status
 * - track the host power status
 * - options: (power_cycle_interval, ...)
 *
 * TODO:
 * - subscribe functions cannot fail. return handler_id as void* instead of int
 * 
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef _PP_BMC_HARDWARE_HOST_POWER_H
#define _PP_BMC_HARDWARE_HOST_POWER_H

#include <pp/bmc/topo_factory.h>

/**
 * Host power states (ACPI)
 *
 * values represent the sensor offsets in IPMI spec, ON/OFF is mapping to the
 * simple power state. Note: S1234 is treated as 'on' although S4 is off.
 * 
 * Values defined according to Table 42-3, SensorType 0x22
 */
#define HOST_POWER_STATE_S0		0x00	// ON   S0/G0 (working)
#define HOST_POWER_STATE_S1		0x01	// ON   S1 (sleeping with sys h/w & processor context maintained)
#define HOST_POWER_STATE_S2		0x02	// ON   S2 (sleeping; processor context lost)
#define HOST_POWER_STATE_S3		0x03	// ON   S3 (sleeping; processor & h/w context lost; memory retained)
#define HOST_POWER_STATE_S4		0x04	// OFF  S4 (non volatile sleep / suspend-to disk)
#define HOST_POWER_STATE_S5		0x05	// OFF  S5/G2 (soft-off)
#define HOST_POWER_STATE_S45		0x06	// OFF  S4/S5 (soft-off; cannot differentiate S4 and S5)
#define HOST_POWER_STATE_MECHOFF	0x07	// OFF  G3 (mechanical off)
#define HOST_POWER_STATE_S123		0x08	// ON   S1/S2/S3 (sleeping; cannot differentiate S1, S2 and S3)
#define HOST_POWER_STATE_S1234		0x09	// ON   S1/S2/S3/S4 (sleeping; cannot differentiate S1, S2, S3 and S4)
#define HOST_POWER_STATE_S5OVR		0x0A	// OFF  S5 (entered by override)
#define HOST_POWER_STATE_LEGON		0x0B	// ON   Legacy ON state
#define HOST_POWER_STATE_LEGOFF		0x0C	// OFF  Legacy OFF state
#define HOST_POWER_STATE_UNKNOWN	0x0E	//      unknown power state

/**
 * convenience macros for ACPI power state. Returns true if state is off,
 * false if state is on or unknown.
 * (use GNU C extension to avoid side effects of multiple evaluation of state)
 */
#define HOST_POWER_STATE_IS_OFF(state)			\
    ({int _state = (state);				\
	(   _state == HOST_POWER_STATE_S4		\
         || _state == HOST_POWER_STATE_S5               \
	 || _state == HOST_POWER_STATE_S45		\
	 || _state == HOST_POWER_STATE_MECHOFF		\
         || _state == HOST_POWER_STATE_S5OVR            \
	 || _state == HOST_POWER_STATE_LEGOFF);})
#define HOST_POWER_STATE_IS_ON(state) (!HOST_POWER_STATE_IS_OFF(state))

/**
 * convenience macro for POH counter condition. Returns true if state is S0 
 * or LEGON
 */
#define HOST_POWER_STATE_POH_IS_ON(state)		\
    ({int _state = (state);				\
	(   _state == HOST_POWER_STATE_S0		\
	 || _state == HOST_POWER_STATE_LEGON);})
#define HOST_POWER_STATE_POH_IS_OFF(state) (!HOST_POWER_STATE_POH_IS_ON(state))


/**
 * Host power control actions.
 * Used by watchdog(reset, powercycle, poweroff), chassis (all),
 * pef(reset, powercycle, poweroff)
 */
#define HOST_POWER_CONTROL_ON         0   // power on
#define HOST_POWER_CONTROL_OFF        1   // power down
#define HOST_POWER_CONTROL_RESET      2   // reset (hard)
#define HOST_POWER_CONTROL_CYCLE      3   // power cycle, will report as power_off/power_on
#define HOST_POWER_CONTROL_SOFT_RESET 4   // soft reset, cannot be triggered by bmc, but can eventually be detected
#define HOST_POWER_CONTROL_SOFT_OFF   5   // soft off, via ACPI (power button shortly pressed)

/**
 * Host power control sources (restart only)
 */
#define HOST_POWER_CONTROL_SOURCE_UNKNOWN              0
#define HOST_POWER_CONTROL_SOURCE_CHASSIS_CONTROL      1
#define HOST_POWER_CONTROL_SOURCE_RESET_BUTTON         2
#define HOST_POWER_CONTROL_SOURCE_POWER_BUTTON_ON      3
#define HOST_POWER_CONTROL_SOURCE_WATCHDOG             4
#define HOST_POWER_CONTROL_SOURCE_OEM                  5
#define HOST_POWER_CONTROL_SOURCE_POLICY_ON            6
#define HOST_POWER_CONTROL_SOURCE_POLICY_RESTORE       7
#define HOST_POWER_CONTROL_SOURCE_PEF_RESET            8
#define HOST_POWER_CONTROL_SOURCE_PEF_CYCLE            9
#define HOST_POWER_CONTROL_SOURCE_SOFT_RESET          10
#define HOST_POWER_CONTROL_SOURCE_RTC                 11

/* return value for power control actions */
#define PP_NOT_APPLICABLE      1

/**
 * Set the power state that can be retrieved by the ipmi commands
 * get/set-acpi-power-state. This can differ from the real power state.
 */
void pp_bmc_set_sys_acpi_pwr_state(unsigned char power_state);

/**
 * Get the current power status of the host.
 * @returns HOST_POWER_STATE_XXX. If a power control action is in
 * progress, the result may be misleading. (For example the status during
 * power_cycle is POWER_STATE_S5 (off) although the host will certainly
 * return to POWER_STATE_S0 (on) without further interaction.
 * Dont use this function if you cannot handle all results.
 */
int pp_bmc_host_power_get_state(void);

/**
 * Simplify the provided power state to on, off, unknown.
 * @returns HOST_POWER_STATE_LEGON, HOST_POWER_STATE_LEGOFF or
 *          HOST_POWER_STATE_UNKNOWN. Other power state values are
 *          mapped to these three states.
 */
int pp_bmc_host_power_simplify_state(int state);

/**
 * Set the current (logical) power status of the host. The new state
 * will override the current state if it is different or more detailed
 * than the current state. This means that for example a 'legacy on'
 * will never overwrite 'S0' (because S0 is included in legacy_on).
 * The command will not change the real power state of the system
 * (it will not power the host on or off).
 * 
 * If the new power state differs from sensor data, it may be overwritten
 * by sensor measures sooner or later. This is merely a way for the OS to
 * deliver more detailed states than the power sensors might provide.
 */
void pp_bmc_host_power_set_state(int state);

/**
 * @returns != 0 if the host power state is on (see HOST_POWER_STATE_IS_ON macro above).
 * @returns 0 if the host power state off
 */
int pp_bmc_host_power_state_is_on(void);

/**
 * Initiate a host power control action. The source-channel must only be
 * provided if the source is a chassis control command.
 * 
 * @returns PP_SUC if the command could be executed.
 * @returns PP_ERR if the command failed due to hardware reasons
 *                 (for example no actor registered)
 * @returns PP_NOT_APPLICABLE if the command cannot be executed in the current
 *                 power context (other action already in progress)
 */
int pp_bmc_host_power_control(int action, int source, unsigned char channel);

/**
 * @returns PP_SUC if the last power control action succeded.
 * @returns PP_ERR if the last power control action failed.
 */
int pp_bmc_host_power_control_result(void);

/**
 * set power button status, external direct power control. can be
 * suppresed by internal actions. called by host_hardware.
 */
void pp_bmc_host_power_set_power_button(unsigned char value);

/**
 * set reset button status, external direct power control. can be
 * suppresed by internal actions. called by host_hardware.
 */
void pp_bmc_host_power_set_reset_button(unsigned char value);

/**
 * A subscription mechanism for changes of the power state.
 * Called whenever the host enters a new power state.
 */
typedef void(*pp_bmc_host_power_state_hndl_t)(int new_state);

/**
 * Subscribe a new host_power_state_hndl.
 */
void pp_bmc_host_power_state_subscribe(pp_bmc_host_power_state_hndl_t power_state_hndl);

/**
 * @returns PP_SUC if the specified handler could be unsubscribed
 * @returns PP_ERR if the specified handler does not exist
 */
int pp_bmc_host_power_state_unsubscribe(pp_bmc_host_power_state_hndl_t power_state_hndl);



/**
 * A subscription mechanism for power control actions.
 * Called whenever a power control action occurs.
 */
typedef void(*pp_bmc_host_power_control_hndl_t)(int control_action);

/**
 * Subscribe a new host_power_control_hndl.
 */
void pp_bmc_host_power_control_subscribe(pp_bmc_host_power_control_hndl_t
                                         power_control_hndl);

/**
 * @returns PP_SUC if the specified handler could be unsubscribed
 * @returns PP_ERR if the specified handler does not exist
 */
int pp_bmc_host_power_control_unsubscribe(pp_bmc_host_power_control_hndl_t);



/* Host power policy options */
#define HOST_POWER_POLICY_OFF       0   // power stays off if AC returns
#define HOST_POWER_POLICY_RESTORE   1   // restore power to previous state
#define HOST_POWER_POLICY_ON        2   // always power on if AC returns
#define HOST_POWER_POLICY_UNKNOWN   3   // no policy exists

/**
 * set the host power policy.
 */
void pp_bmc_host_power_policy_set(unsigned char policy);

/**
 * get the host power policy
 */
unsigned char pp_bmc_host_power_policy_get(void);

/**
 * Set the host power cycle interval (in seconds). Querying this values is
 * not supported.
 */
void pp_bmc_host_power_cycle_interval_set(unsigned char interval);

/**
 * Get the power supply status
 */
#define HOST_POWER_SUPPLY_OK          0
#define HOST_POWER_SUPPLY_OVERLOAD    1
#define HOST_POWER_SUPPLY_INTERLOCK   2
#define HOST_POWER_SUPPLY_FAULT       3
/* get power supply status */
int pp_bmc_host_power_supply_status(void);

/**
 * Returns PP_SUC if the host has a power interlock device.
 */
int pp_bmc_host_power_interlock_exists(void);

/**
 * Get last power event byte as defined in table 28-3, byte 3
 */
unsigned char pp_bmc_host_power_get_last_event(void);

/**
 * Get the last system restart cause (according to the source passed
 * in bmc_host_power_control()).
 */
unsigned char pp_bmc_host_power_get_restart_cause(void);
unsigned char pp_bmc_host_power_get_restart_cause_channel(void);


/**
 * This function is called whenever a new sensor reading is received.
 * ctx contains the index of the sensor
 */
void pp_bmc_host_power_receive_reading(int reading, void* ctx);


/**
 * Init the host_power module, called by bmc_host_hardware_init()
 */
int pp_bmc_host_power_init(void);

/**
 * Cleanup the host_power module, called by bmc_host_hardware_init()
 */
void pp_bmc_host_power_cleanup(void);


#endif /* _PP_BMC_HARDWARE_HOST_POWER_H */
