/**
 * host_sensors.h
 * 
 * Registry of sensors and actors for host_power, host_hardware and other
 * libpp_hardware files. Provides a callback function for libpp_sensor
 * and automatically registers with relevant sensors. Sensor events are
 * automatically forwarded to interested (hardwired) consumers.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef _PP_BMC_HARDWARE_HOST_SENSORS_H
#define _PP_BMC_HARDWARE_HOST_SENSORS_H

#include <pp/bmc/tp_sensdev.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_rs485_chip.h>
#ifdef KIRA_RPC
#include <pp/bmc/tp_ctrl.h>
#endif /* KIRA_RPC */
#include <pp/bmc/tp_gpio_act.h>
#include <pp/bmc/tp_actor.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/tp_cond.h>

/* currently known sensor types      id          function                  usage                  */
/* -----------------------------------------------------------------------------------------------*/
enum {
  PP_SENSOR_POWER_BTN,            // power_btn   Frontpanel power button   1=pressed 0=released
  PP_SENSOR_RESET_BTN,            // reset_btn   Frontpanel reset button   1=pressed 0=released
  PP_SENSOR_SLEEP_BTN,            // sleep_btn   Frontpanel sleep button   1=pressed 0=released
  PP_SENSOR_ID_BTN,               // id_btn      Frontpanel id    button   1=pressed 0=released
  PP_SENSOR_DIAG_BTN,             // diag_btn    Frontpanel diag_int btn   1=pressed 0=released

  PP_SENSOR_SOFT_RESET,           // soft_reset  (from chipset)     1=system_reset 0=system_on

  PP_SENSOR_CHASSIS_INTRUSION,    // chas_intr   chassis intrusion  1=closed       0=open
  PP_SENSOR_DRIVE_HEALTH,         // drv_health  drive health       1=drive_fault  0=drive_ok
  PP_SENSOR_FAN_HEALTH,           // fan_health  fan health         1=fan_fault    0=fan_ok
  PP_SENSOR_POWER_HEALTH,         // ps_health   power supply       1=ps_fault     0=ps_ok
  PP_SENSOR_POWER_INTERLOCK,      // ps_ilock    power interlock    1=ps_locked    0=ps_ok
  PP_SENSOR_NMI_BTN,              // nmi_btn     frontpanel nmi button
  PP_SENSOR_ACPI_POWER_STATE,     // acpi_power_state  ACPI power state
#ifdef PRODUCT_ICPMMD
  PP_SENSOR_PSU0_AC1_OK,
  PP_SENSOR_PSU0_AC2_OK,
  PP_SENSOR_PSU1_AC1_OK,
  PP_SENSOR_PSU1_AC2_OK,
  PP_SENSOR_PSU2_AC1_OK,
  PP_SENSOR_PSU2_AC2_OK,
  PP_SENSOR_PSU3_AC1_OK,
  PP_SENSOR_PSU3_AC2_OK,
  PP_SENSOR_IBSW0_SLOTID,
  PP_SENSOR_IBSW0_POWER_OK,
  PP_SENSOR_IBSW1_SLOTID,
  PP_SENSOR_IBSW1_POWER_OK,
#endif /* PRODUCT_ICPMMD */
  PP_SENSOR_MAX
};

/* currently known actor types (non-pwm)    id          function       */
/* --------------------------------------------------------------------*/
enum {
  PP_ACTOR_POWER,                     // power_out   assert (pulse, long) to power up/down
  PP_ACTOR_RESET,                     // reset_out   assert (pulse) to reset
  PP_ACTOR_STANDBY,                   // stdby_out   assert (pulse) to activate standby mode
  PP_ACTOR_DIAG_INT,                  // diag_int    assert (pulse) to activate diagnostic interrupt
  PP_ACTOR_ID_LED,                    // id_led      assert to turn on chassis id led
  PP_ACTOR_SMUX,                      // smux        serial mux
  PP_ACTOR_NMI,                       // nmi_out     non maskable interrupt handler (cause the OS to core dump)
  PP_ACTOR_SMI,                       // smi_out     system management interrupt actor
  PP_ACTOR_CLEAR_CMOS,                // clear_cmos  assert to clear the system bios cmos
  PP_ACTOR_LOCAL_LOCK,                // local_lock  disable front panel buttons with board hw support
  PP_ACTOR_PWM,                       // pwm#        pwm line for fan control
  PP_ACTOR_CLR_INTRUSION,             // clr_intrusion clears chassis intrusion flag (if required)
  PP_ACTOR_HEARTBEAT,                 // heartbeat   heartbeat of BMC (should be oscilated by BMC)
#ifdef PP_FEAT_OPMA_HW
  PP_ACTOR_MCARD_ID,                  // mcard_id#   3 bit binary encoded value to signal interface type
#endif /* PP_FEAT_OPMA_HW */
#ifdef PRODUCT_ICPMMD
  PP_ACTOR_CFU0_FAN1_OK,
  PP_ACTOR_CFU0_FAN2_OK,
  PP_ACTOR_CFU1_FAN1_OK,
  PP_ACTOR_CFU1_FAN2_OK,
  PP_ACTOR_CFU2_FAN1_OK,
  PP_ACTOR_CFU2_FAN2_OK,
  PP_ACTOR_CFU3_FAN1_OK,
  PP_ACTOR_CFU3_FAN2_OK,
  PP_ACTOR_PSU0_FAN_OK,
  PP_ACTOR_PSU0_DC_OK,
  PP_ACTOR_PSU1_FAN_OK,
  PP_ACTOR_PSU1_DC_OK,
  PP_ACTOR_PSU2_FAN_OK,
  PP_ACTOR_PSU2_DC_OK,
  PP_ACTOR_PSU3_FAN_OK,
  PP_ACTOR_PSU3_DC_OK,
  PP_ACTOR_IBSW0_FAULT_LED,
  PP_ACTOR_IBSW0_GOOD_LED,
  PP_ACTOR_IBSW0_RESET,
  PP_ACTOR_IBSW1_FAULT_LED,
  PP_ACTOR_IBSW1_GOOD_LED,  
  PP_ACTOR_IBSW1_RESET,
#endif /* PRODUCT_ICPMMD */
#ifdef KIRA_RPC
  PP_ACTOR_R0_STATE,
  PP_ACTOR_R1_STATE,
  PP_ACTOR_R2_STATE,
  PP_ACTOR_R3_STATE,
  PP_ACTOR_R4_STATE,
  PP_ACTOR_R5_STATE,
  PP_ACTOR_R6_STATE,
  PP_ACTOR_R7_STATE,
  PP_ACTOR_R8_STATE,
  PP_ACTOR_R9_STATE,
  PP_ACTOR_R10_STATE,
  PP_ACTOR_R11_STATE,
  PP_ACTOR_R12_STATE,
  PP_ACTOR_R13_STATE,
  PP_ACTOR_R14_STATE,
  PP_ACTOR_R15_STATE,
  PP_ACTOR_R16_STATE,
  PP_ACTOR_R17_STATE,
  PP_ACTOR_R18_STATE,
  PP_ACTOR_R19_STATE,
  PP_ACTOR_R20_STATE,
  PP_ACTOR_R21_STATE,
  PP_ACTOR_R22_STATE,
  PP_ACTOR_R23_STATE,  
#endif /* KIRA_RPC */
  PP_ACTOR_MAX
};

/* currently known actors (pwm)                                                  *
 * PWM_ACTORs are stored in a separate storage that can hold multiple instances. *
 * The actors must be declared as 'pwmXXX' and can be retrieved by their number. *
 * For details see function signatures below.                                    */
#define PP_PWM_ACTOR_MAX         10        // maximum number of possible pwm actors

/* currently known conditions                               */
/* ---------------------------------------------------------*/

enum {
#ifdef LARA_KIMMSI
  PP_BMC_COND_CPU0_PRESENCE,
  PP_BMC_COND_CPU1_PRESENCE,
#endif
#ifdef PRODUCT_SMIDC
  PP_BMC_COND_I2C_ACCESS, // Supermicro: I2C access sync'd with BIOS
#endif
  PP_BMC_COND_MAX
};

#ifdef LARA_KIMMSI

/* ATTENTION:
 * funktion 'oem_msi_cmd_set_cpu_info' (bmc_devf_oem_msi.c)
 * assumes that CPU-presence conditions have consecutive numbers
 * starting from
 *      PP_BMC_COND_CPU_PRESENCE_BASE
 * and not exceeding
 *      PP_BMC_COND_CPU_PRESENCE_LAST
 *
 * _DO NOT CHANGE_ these defines without following these rules or,
 * if needed, change code in bmc_devf_oem_msi.c.
 */

#  define PP_BMC_COND_CPU_PRESENCE_BASE PP_BMC_COND_CPU0_PRESENCE
#  define PP_BMC_COND_CPU_PRESENCE_LAST PP_BMC_COND_CPU1_PRESENCE

#endif

/* currently known i2c communication devices                */
/* ---------------------------------------------------------*/
enum {
	PP_BMC_I2C_COMDEV_IPMB0,
	PP_BMC_I2C_COMDEV_IPMB1,
#ifdef PRODUCT_ICPMMD
	PP_BMC_I2C_COMDEV_GESW,
#endif

	PP_BMC_I2C_COMDEV_MAX
};

/* currently known rs485 chip devices                       */
/* ---------------------------------------------------------*/
enum {
	PP_BMC_RS485_CHIP_RPC0,
	PP_BMC_RS485_CHIP_RPC1,
	PP_BMC_RS485_CHIP_RPC2,
	PP_BMC_RS485_CHIP_RPC3,
	PP_BMC_RS485_CHIP_RPC4,

	PP_BMC_RS485_CHIP_MAX
};

#define PP_BMC_RS485_CHIP_PDU_MAX (PP_BMC_RS485_CHIP_RPC4 + 1)

#ifdef KIRA_RPC
/* currently known controller devices                       */
/* ---------------------------------------------------------*/
enum {
	PP_BMC_CTRL_R0_LED,
	PP_BMC_CTRL_R1_LED,
	PP_BMC_CTRL_R2_LED,
	PP_BMC_CTRL_R3_LED,
	PP_BMC_CTRL_R4_LED,
	PP_BMC_CTRL_R5_LED,
	PP_BMC_CTRL_R6_LED,
	PP_BMC_CTRL_R7_LED,
	PP_BMC_CTRL_R8_LED,
	PP_BMC_CTRL_R9_LED,
	PP_BMC_CTRL_R10_LED,
	PP_BMC_CTRL_R11_LED,
	PP_BMC_CTRL_R12_LED,
	PP_BMC_CTRL_R13_LED,
	PP_BMC_CTRL_R14_LED,
	PP_BMC_CTRL_R15_LED,
	PP_BMC_CTRL_R16_LED,
	PP_BMC_CTRL_R17_LED,
	PP_BMC_CTRL_R18_LED,
	PP_BMC_CTRL_R19_LED,
	PP_BMC_CTRL_R20_LED,
	PP_BMC_CTRL_R21_LED,
	PP_BMC_CTRL_R22_LED,
	PP_BMC_CTRL_R23_LED,

	PP_BMC_CTRL_MAX
};
#endif /* KIRA_RPC */

/**
 * get a sensor from the registry. may return null if the sensor is
 * not yet initialized (topology not parsed)
 */
pp_tp_sensdev_t* pp_bmc_host_sensor_get_sensor(int type);

/**
 * get an non-pwm actor from the registry. may return null if the
 * actor is not yet initialized (topology nor parsed)
 */
pp_tp_actor_t* pp_bmc_host_sensor_get_actor(int type);
pp_tp_gpio_act_t* pp_bmc_host_sensor_get_gpio_actor(int type);

/**
 * Get a pwm actor from the registry. Returns null if the specified
 * pwm actor is not yet initialized.
 */
pp_tp_pwm_act_t* pp_bmc_host_sensor_get_pwm_actor(int idx);

/**
 * get a condition object from registry.
 * may return NULL if not present in topo-file
 */
pp_tp_cond_t* pp_bmc_host_get_condition(int type);

/**
 * get a i2c communication device from registry.
 * may return NULL if not present in topo-file
 */
pp_tp_i2c_comdev_t* pp_bmc_host_get_i2c_comdev(int type);

/**
 * get a rs485 communication device from registry.
 * may return NULL if not present in topo-file
 */
pp_tp_rs485_chip_t* pp_bmc_host_get_rs485_chip(int type);

#ifdef KIRA_RPC
/**
 * get a controller device from registry.
 * may return NULL if not present in topo-file
 */
pp_tp_ctrl_t* pp_bmc_host_get_ctrl(int type);
#endif /* KIRA_RPC */

/**
 * Callback for topology parser to notify module of new sensors.
 */
void pp_bmc_host_sensor_create(pp_tp_obj_t* obj);

/**
 * host_sensor initializer, called by host_hardware.c
 */
int pp_bmc_host_sensor_init(void);

/**
 * host_sensor cleanup, called by host_hardware.c
 */
void pp_bmc_host_sensor_cleanup(void);


#endif /* _PP_BMC_HARDWARE_HOST_SENSORS_H */


