/**
 * serial_lowelevel.h
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * A serial port lowlevel abstraction that can easily be mapped
 * to different motherboard/serial port sharing solutions. The
 * abstraction assumes that the smux can have three states and 
 * that there can be no more than one handler for each state.
 * Currently these handlers are SOL, snoop mode, and standard
 * serial mode(basic mode/CLI).
 * 
 * Each serial handler has to implement activation and deactivation
 * callbacks. When a mux state switch is received from an external
 * source, the active serial handler is deactivated. This may take
 * some time so a callback is provided. Then the new serial handler
 * is activated and can start operation. Please note that the command
 * to change the mux returns immediately, but the corresponding
 * handler is not operational until the old handler is deactivated.
 */

#ifndef __SERIAL_LOWLEVEL_H__
#define __SERIAL_LOWLEVEL_H__

typedef enum pp_serial_mux_state_e {
    PP_SMUX_HOST_CHASSIS,
    PP_SMUX_BMC_CHASSIS,
    PP_SMUX_HOST_BMC
} pp_bmc_serial_mux_state_t;

#if PP_HWID_INT == 0x10 || PP_HWID_INT == 0x20
/* MSI || OPMAG4 */
# define PP_BMC_SERIAL_TTY "/dev/ttyS0"
#else
# define PP_BMC_SERIAL_TTY "/dev/ttyS1"
#endif

/**
 * returns the current state of the mux switch
 */
pp_bmc_serial_mux_state_t pp_bmc_serial_mux_get_state(void);

/**
 * Initiate a serial mux switch to the given state. Please note that
 * the new state might not immediately be available after the function
 * returns.
 * @returns PP_ERR if a switch is in progress and the request cannot be fullfilled
 */
int pp_bmc_serial_mux_switch(pp_bmc_serial_mux_state_t newstate);

/**
 * serial handler activation type.
 * This function is called to activate a handler associated with a
 * new switch state. The filedescriptor for the serial connection can be 
 * retrieved with pp_bmc_serial_lowlevel_get_fd() and the handler
 * becomes responsible for handling the connection (read, write,
 * register to selector, ...).
 */
typedef void (*pp_bmc_serial_mux_activate_hndl)(pp_bmc_serial_mux_state_t oldstate);

/**
 * serial handler deactivation type.
 * This function is called to deactivate an active serial handler.
 * This must happen as soon as possible. After deactivation, the
 * handler must call 'deactivate_finished()' to notify the mux that
 * deactivation is finished.
 */
typedef void (*pp_bmc_serial_mux_deactivate_hndl)(pp_bmc_serial_mux_state_t newstate);

/**
 * serial deactivation finished callback.
 * This function must be called by the deactivate_hndl once the
 * deactivation is finished.
 */
void pp_bmc_serial_mux_deactivate_finished(void);

/**
 * Register a handler for a serial mux switch state. If another handler is
 * already registered for this state, it will be unregistered first.
 */
void pp_bmc_serial_mux_register_handler(pp_bmc_serial_mux_state_t state,
                                        pp_bmc_serial_mux_activate_hndl activate_hndl,
                                        pp_bmc_serial_mux_deactivate_hndl deactivate_hndl);

/**
 * Clear a handler for a serial mux switch state. The existing handler
 * is deactivated (if necessary) and unregistered.
 */
void pp_bmc_serial_mux_clear_handler(pp_bmc_serial_mux_state_t state);


/**
 * Initialize the serial lowlevel module.
 */
int pp_bmc_serial_lowlevel_init(void);

/**
 * Clean up the serial lowlevel module.
 */
void pp_bmc_serial_lowlevel_cleanup(void);

#endif
