/**
 * lan_arp.c
 *
 * BMC-generated ARP handling
 *
 * (c) 2006 Peppercon AG, inva@raritan.com
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <linux/if.h>
#include <linux/if_ether.h>
#include <linux/if_packet.h>

#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/xdefs.h>
#include <pp/selector.h>
#include <pp/bmc/bmc_watchdog.h>

#include "lan_arp.h"

static int arp_item_id = -1;
static int arp_fd;

static volatile int reconfigure = 1;
static volatile int suspended = 0;
static char *interface = NULL;

static const char *network_options[] = {
    "network.eth_interface",
    "network.ipaddr",
    "network.netmask",
    "network.mac",
    NULL
};

struct arp_packet_t {
    unsigned char  eth_dst[ETH_ALEN];
    unsigned char  eth_src[ETH_ALEN];
    unsigned short eth_proto_be16;
    unsigned short arp_hw_id_be16;
    unsigned short arp_proto_id_be16;
    unsigned char  arp_hw_size;
    unsigned char  arp_proto_size;
    unsigned short arp_opcode_be16;
    unsigned char  arp_sender_mac[ETH_ALEN];
    in_addr_t      arp_sender_ip;
    unsigned char  arp_target_mac[ETH_ALEN];
    in_addr_t      arp_target_ip;
} __attribute__ ((packed));

static struct arp_packet_t packet = {
    .eth_dst           = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, // Broadcast
    .eth_proto_be16    = cpu_to_be16_const(0x0806),              // ARP
    .arp_hw_id_be16    = cpu_to_be16_const(0x0001),              // Ethernet
    .arp_proto_id_be16 = cpu_to_be16_const(0x0800),              // IP
    .arp_hw_size       = ETH_ALEN,
    .arp_proto_size    = 4,
    .arp_opcode_be16   = cpu_to_be16_const(0x0001),              // Query
    .arp_target_mac    = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }  // Broadcast
};

static inline int get_ifindex(void)
{
    struct ifreq req;
    if (!interface) return -1;
    strncpy(req.ifr_name, interface, sizeof(req.ifr_name)-1);
    req.ifr_name[sizeof(req.ifr_name)-1] = '\0';
    if (ioctl(arp_fd, SIOCGIFINDEX, &req) < 0) return -1;
    return req.ifr_ifindex;
}

static inline int prepare_packet(void)
{
    int ret = PP_ERR;
    char *ipaddr_str = NULL;
    char *mac_str = NULL;
    char *netmask_str = NULL;
    unsigned int mac_uint[6];
    in_addr_t ipaddr, netmask;
    int i;

    free(interface);
    interface = NULL;

    if ( PP_FAILED(pp_cfg_get(&interface, "network.eth_interface"))
      || PP_FAILED(pp_cfg_get_nodflt(&ipaddr_str, "network.ipaddr"))
      || PP_FAILED(pp_cfg_get_nodflt(&mac_str, "network.mac"))
      || PP_FAILED(pp_cfg_get_nodflt(&netmask_str, "network.netmask")) ) {
	goto bail;
    }

    if (sscanf(mac_str, "%x:%x:%x:%x:%x:%x",
		mac_uint  , mac_uint+1, mac_uint+2,
		mac_uint+3, mac_uint+4, mac_uint+5) != 6) {
	goto bail;
    };

    ipaddr = inet_addr(ipaddr_str);
    netmask = inet_addr(netmask_str);
    if (ipaddr == INADDR_NONE || netmask == INADDR_NONE) goto bail;

    for (i = 0; i < ETH_ALEN; i++) {
	packet.eth_src[i] = mac_uint[i];
	packet.arp_sender_mac[i] = mac_uint[i];
    }
    packet.arp_sender_ip = ipaddr;
    packet.arp_target_ip = ipaddr;
    reconfigure = 0;
    ret = PP_SUC;

bail:
    free(ipaddr_str);
    free(mac_str);
    free(netmask_str);
    return ret;
}

static int send_packet(const int item_id UNUSED, void *ctx UNUSED)
{
    struct sockaddr_ll sa;

    if (suspended) return PP_SUC;
    if (reconfigure) {
	if (PP_FAILED(prepare_packet())) return PP_ERR;
    }

    sa.sll_family = AF_PACKET;
    sa.sll_ifindex = get_ifindex();
    if (sa.sll_ifindex < 0) return PP_SUC;
    sa.sll_protocol = htons(ETH_P_ALL);

    sendto(arp_fd, &packet, sizeof(packet), 0,
	    (struct sockaddr *)&sa, sizeof(sa));

    return PP_SUC;
}

/* configuration change handlers */
static int network_reconf_handler(pp_cfg_chg_ctx_t *ctx UNUSED)
{
    reconfigure = 1;
    return PP_SUC;
}

static int arp_reconf_handler(pp_cfg_chg_ctx_t *ctx UNUSED)
{
    int generate, interval;
    if ( PP_FAILED(pp_cfg_is_enabled(&generate, "bmc.lan.arp.generate"))
      || PP_FAILED(pp_cfg_get_int(&interval, "bmc.lan.arp.interval")) ) {
	return PP_ERR;
    }

    if (arp_item_id >= 0) {
	pp_select_remove_to(arp_item_id);
	arp_item_id = -1;
    }

    if (generate) {
	arp_item_id = pp_select_add_to((interval+1) * 500, 1, send_packet, NULL);
    }

    return PP_SUC;
}

#if defined (PP_FEAT_IPMI_SERVER_WATCHDOG)
static void watchdog_callback(pp_bmc_dev_watchdog_event_t event)
{
    if (event == PP_BMC_DEV_WATCHDOG_SET
	    || event == PP_BMC_DEV_WATCHDOG_RESET
	    || event == PP_BMC_DEV_WATCHDOG_TIMEOUT) {
	pp_lan_arp_suspend(0);
    }
}
#endif

int pp_lan_arp_suspend(int suspend)
{
    suspended = suspend;
    return (arp_item_id >= 0) && !suspended;
}

/* c'tor / d'tor */
int pp_lan_arp_init()
{
    int i;
    for (i = 0; network_options[i] != NULL; i++) {
	pp_cfg_add_post_tx_change_listener(network_reconf_handler, network_options[i]);
    }
    pp_cfg_add_post_tx_change_listener(arp_reconf_handler, "bmc.lan.arp");
    arp_fd = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_ALL));
    arp_reconf_handler(NULL);
#if defined (PP_FEAT_IPMI_SERVER_WATCHDOG)
    pp_bmc_dev_watchdog_add_callback(watchdog_callback);
#endif
    return PP_SUC;
}

void pp_lan_arp_cleanup()
{
    int i;
#if defined (PP_FEAT_IPMI_SERVER_WATCHDOG)
    pp_bmc_dev_watchdog_rem_callback(watchdog_callback);
#endif
    for (i = 0; network_options[i] != NULL; i++) {
	pp_cfg_rem_change_listener(network_reconf_handler, network_options[i]);
    }
    pp_cfg_rem_change_listener(arp_reconf_handler, "bmc.lan.arp");
    if (arp_item_id >= 0) {
	pp_select_remove_to(arp_item_id);
	arp_item_id = -1;
    }
    close(arp_fd);
    free(interface);
}

