/**
 * lan_dev_conf.c
 *
 * IPMI LAN commands
 *
 * (c) 2005 Peppercon AG, Thomas Weber <tweb@peppercon.de>
 */

#include "pp/base.h"

#include "pp/cfg.h"
#include "pp/xdefs.h"

#include "pp/bmc/debug.h"

#include "pp/bmc/bmc_core.h"
#include "pp/bmc/bmc_router.h"
#include "pp/bmc/utils.h"

#include "pp/bmc/ipmi_cmd.h"
#include "pp/bmc/ipmi_err.h"
#include "pp/bmc/ipmi_chan.h"
#include "pp/bmc/ipmi_sess.h"

#include "pp/bmc/lan_cipher_suites.h"

#include "lan_dev_conf.h"
#include "lan_arp.h"

static unsigned char lanconf_set_in_progress = 0; /* set complete */

static const char *bmc_lan_alert_community_str = "bmc.lan.alert.community";
static const char *bmc_lan_alert_dest_sz_str = "bmc.lan.alert.dest._s_";
static const char *bmc_lan_alert_dest_acknowledge_str =
                      "bmc.lan.alert.dest[%u].acknowledge";
static const char *bmc_lan_alert_dest_retry_no_str =
                      "bmc.lan.alert.dest[%u].retry_no";
static const char *bmc_lan_alert_dest_timeout_str =
                      "bmc.lan.alert.dest[%u].timeout";
static const char *bmc_lan_alert_dest_trap_addr_str =
                      "bmc.lan.alert.dest[%u].trap_addr";
static const char *bmc_lan_alert_dest_type_str =
                      "bmc.lan.alert.dest[%u].type";
static const char *bmc_lan_gateway_mac_str = "bmc.lan.gateway_mac";
static const char *bmc_lan_backup_gateway_str = "bmc.lan.backup_gateway";
static const char *bmc_lan_backup_gateway_mac_str =
                      "bmc.lan.backup_gateway_mac";
static const char *bmc_lan_backup_gateway_enabled_str =
                      "bmc.lan.backup_gateway_enabled";
static const char *bmc_lan_rmcpp_mcs_priv_level =
                      "bmc.lan.rcmpp_mcs_priv_level[%u]";
static const char *bmc_lan_arp_generate_str = "bmc.lan.arp.generate";
static const char *bmc_lan_arp_interval_str = "bmc.lan.arp.interval";
static const char *network_gateway_str = "network.gateway";
static const char *network_ip_auto_config_proto_str =
                      "network.ip_auto_config_proto";
static const char *network_ip_addr_str = "network.ipaddr";
static const char *network_netmask_str = "network.netmask";
static const char *network_ip_addr_source_str = "network.ipaddr_source";
static const char *network_mac_str = "network.mac";

/* implemented configuration types */
#define LANCONF_AUTH_IMPLEMENTED 0x17 /* none, MD2, MD5, plain */

#define IP_V4_FROM_CFG_CORE(_data, _cfgkey, _idx) { \
    char *_val = NULL; \
    if (PP_FAILED(pp_cfg_get_nodflt(&_val, _cfgkey, _idx)) || !_val[0]) { \
        memset(_data, 0, 4); \
    } else { \
        *(in_addr_t*)_data = inet_addr(_val); \
    } \
    free(_val); \
}

#define IP_V4_FROM_CFG(_data, _cfgkey) \
    IP_V4_FROM_CFG_CORE(_data, _cfgkey, NULL)

#define IP_V4_TO_CFG_CORE(_data, _cfgkey, _idx) \
    pp_cfg_set(inet_ntoa(*(struct in_addr*)_data), _cfgkey, _idx)

#define IP_V4_TO_CFG(_data, _cfgkey) \
    IP_V4_TO_CFG_CORE(_data, _cfgkey, NULL)

#define MAC_FROM_CFG(_data, _cfgkey) { \
    char *_val = NULL; \
    memset(_data, 0, 6); \
    if (PP_SUCCED( pp_cfg_get_nodflt(&_val, _cfgkey) )) { \
        int k; \
        unsigned int mac_uint[6]; \
        sscanf(_val, "%x:%x:%x:%x:%x:%x", mac_uint, mac_uint+1, mac_uint+2, mac_uint+3, mac_uint+4, mac_uint+5); \
        for (k = 0; k < 6; k++) _data[k] = mac_uint[k]; \
    } \
    free(_val); \
}

#define MAC_TO_CFG(_data, _cfgkey) { \
    char macstring[18]; \
    snprintf(macstring, sizeof(macstring), \
                    "%02x:%02x:%02x:%02x:%02x:%02x", \
                    _data[0], _data[1], _data[2], \
                    _data[3], _data[4], _data[5]); \
    pp_cfg_set(macstring, _cfgkey); \
}

/* get the rmcpp privilege level for ciphersuite idx and write to *data */
static void lanconf_get_priv_nibble(int idx, unsigned char *data) {
    unsigned char priv_level;
    unsigned char current;
    char *val;
    
    priv_level = IPMI_PRIV_UNSPEC;
    if(PP_ERR != pp_cfg_get_nodflt(&val, bmc_lan_rmcpp_mcs_priv_level, idx) ) {
        priv_level = pp_bmc_get_privlevel_value(val);
        free(val);
    }
    
    /* set nibble but do not change the rest */
    current = data[idx >> 1];
    if(idx & 0x00000001) {
        /* 1, 3, 5...: set upper nibble of current char */
        current = (current & 0x0f) | (priv_level << 4);
    } else {
        /* 0, 2, 4...: set lower nibble of current char */
        current = (current & 0xf0) | priv_level;
    }
    data[idx >> 1] = current;
}

/* write the specified ciphersuite configuration nibble to the config system */
static unsigned char lanconf_set_priv_nibble(int idx, unsigned char *data) {
    unsigned char priv_level;
    unsigned char current = data[idx >> 1];
    
    if(idx & 0x00000001) {
        /* 1, 3, 5...: get upper nibble of current char */
        priv_level = current >> 4;
    } else {
        /* 0, 2, 4...: get lower nibble of current char */
        priv_level = current & 0x0f;
    }
    
    return pp_cfg_set(pp_bmc_get_privlevel_string(priv_level), bmc_lan_rmcpp_mcs_priv_level, idx);
}

#define CHECK_PDATA_LEN(s) \
    if (psize < s) { ierr = IPMI_ERR_REQ_DATA_LEN_INVALID; break; }

#define SET_CFG(c) \
    if (PP_FAILED(c)) { \
        pp_bmc_log_error("[LANCONF] can't write param #%d to cfg", param); \
        ierr = IPMI_ERR_UNSPECIFIED; \
        break; \
    }

static int lanconf_cmd_set_conf(imsg_t *imsg) {
    /* IPMI message data */
    unsigned char channel = imsg->data[0] & 0x0f;
    unsigned char param = imsg->data[1];
    unsigned char *pdata = &imsg->data[2];
    int psize = imsg->data_size - 2;

    unsigned char ierr = IPMI_ERR_SUCCESS;
    int dirty = 0;
    pp_cfg_tx_t* cfg_trans;
    char val[255];
    int i;

    if(channel != IPMI_CHAN_LAN) {
        pp_bmc_log_warn("[LANCONF] invalid lan channel id %u, should be %u",
                        channel, IPMI_CHAN_LAN);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    cfg_trans = pp_cfg_tx_begin(1);
    if (!cfg_trans) {
        pp_bmc_log_error("[LANCONF] can't create cfg transaction");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }

    pp_bmc_log_info("[LANCONF] set param %u", param);
    switch (param) {
        case 0: /* set in progress */
            CHECK_PDATA_LEN(1);
            switch (pdata[0] & 0x03) {
                case 0: /* set complete */
                    lanconf_set_in_progress = 0;
                    break;
                case 1: /* set in progress */
                    if (lanconf_set_in_progress == 1) {
                        ierr = IPMI_ERR_LANCONF_SET_IN_PROGRESS;
                    }
                    lanconf_set_in_progress = 1;
                    break;
                default:
                    ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
            }
            break;
        case 1: /* authentication type support */
            ierr = IPMI_ERR_LANCONF_READ_ONLY;
            break;
        case 2: /* authentication type enables */
            CHECK_PDATA_LEN(5);

            /* check pdata for correctness */
            if ((pdata[0] & (~LANCONF_AUTH_IMPLEMENTED)) != 0) ierr = IPMI_ERR_INVALID_DATA_FIELD;
            if ((pdata[1] & (~LANCONF_AUTH_IMPLEMENTED)) != 0) ierr = IPMI_ERR_INVALID_DATA_FIELD;
            if ((pdata[2] & (~LANCONF_AUTH_IMPLEMENTED)) != 0) ierr = IPMI_ERR_INVALID_DATA_FIELD;
            if ((pdata[3] & (~LANCONF_AUTH_IMPLEMENTED)) != 0) ierr = IPMI_ERR_INVALID_DATA_FIELD;
            if ((pdata[4] & (~LANCONF_AUTH_IMPLEMENTED)) != 0) ierr = IPMI_ERR_INVALID_DATA_FIELD;
            if (ierr != IPMI_ERR_SUCCESS) break;

            /* pdata is correct, write to config fs */
            SET_CFG(pp_cfg_set_short(pdata[0], "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_CALLBACK_STR));
            SET_CFG(pp_cfg_set_short(pdata[1], "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_USER_STR));
            SET_CFG(pp_cfg_set_short(pdata[2], "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_OPERATOR_STR));
            SET_CFG(pp_cfg_set_short(pdata[3], "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_ADMINISTRATOR_STR));
            SET_CFG(pp_cfg_set_short(pdata[4], "bmc.channel_access[%u].authentication_type_enable[%s]", channel, PP_CD_IPMIPRIVLEVEL_OEM_STR));
            dirty = 1;
            break;
        case 3: /* IP address */
            CHECK_PDATA_LEN(4);
            if (pdata[0] == 0 && pdata[1] == 0 && pdata[2] == 0 && pdata[3] == 0) {
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
                pp_bmc_log_warn("[LANCONF] Set LAN Config Params failed (invalid IP address 0.0.0.0)");
                break;
            }
            SET_CFG(IP_V4_TO_CFG(pdata, network_ip_addr_str));
            dirty = 1;
            break;
        case 4: /* IP address source */
            CHECK_PDATA_LEN(1);
            pdata[0] &= 0x0f;  // mask out reserved bits
            if (pdata[0] > 4) {
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
                pp_bmc_log_warn("[LANCONF] Set LAN Config Params failed (invalid IP address source %d)", pdata[0]);
                break;
            }
            switch (pdata[0]) {
                case 2: // DHCP
                    SET_CFG(pp_cfg_set(PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_DHCP_STR,
                            network_ip_auto_config_proto_str));
                    break;
                case 4: // other (here: BOOTP)
                    SET_CFG(pp_cfg_set(PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_BOOTP_STR,
                            network_ip_auto_config_proto_str));
                    break;
                default:
                    SET_CFG(pp_cfg_set(PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_NONE_STR,
                            network_ip_auto_config_proto_str));
            }
            // set describing value too
            SET_CFG(pp_cfg_set_int(pdata[0], network_ip_addr_source_str));
            dirty = 1;
            break;
        case 5: /* MAC address */
            {
                char macstring[18];
                CHECK_PDATA_LEN(6);
                snprintf(macstring, sizeof(macstring),
                        "%02x:%02x:%02x:%02x:%02x:%02x",
                        pdata[0], pdata[1], pdata[2],
                        pdata[3], pdata[4], pdata[5]);
                SET_CFG(pp_cfg_set_at_layer(PP_PROFILE_SYSTEM, macstring, network_mac_str));
                pp_cfg_save_layer(PP_PROFILE_SYSTEM, DO_FLUSH);
                dirty = 1;
            }
            break;
        case 6: /* subnet mask */
            CHECK_PDATA_LEN(4);
            if (pdata[0] == 0 && pdata[1] == 0 && pdata[2] == 0 && pdata[3] == 0) {
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
                pp_bmc_log_warn("[LANCONF] Set LAN Config Params failed (invalid net mask 0.0.0.0)");
                break;
            }
            SET_CFG(IP_V4_TO_CFG(pdata, network_netmask_str));
            dirty = 1;
            break;
        case 7: /* IPv4 header parameters */
            /* not applicable to our IP stack */
            ierr = IPMI_ERR_LANCONF_READ_ONLY;
            break;
        case 8: /* primary RMCP port number (optional) */
            /* read only parameter (not specified!) */
            ierr = IPMI_ERR_LANCONF_READ_ONLY;
            break;
        case 9: /* secondary RMCP port number (optional) */
            /* read only parameter (not specified!) */
            ierr = IPMI_ERR_LANCONF_READ_ONLY;
            break;
        case 10: /* BMC-generated ARP control (optional) */
            CHECK_PDATA_LEN(1);
            // we don't support to disable ARP responses, just ignore that flag
            //   pdata[0] & 0x02
            // Gratuitous ARP
            SET_CFG(pp_cfg_set_enabled(pdata[0] & 0x01, bmc_lan_arp_generate_str));
            dirty = 1;
            break;
        case 11: /* gratuitous ARP interval (optional) */
            CHECK_PDATA_LEN(1);
            SET_CFG(pp_cfg_set_int(pdata[0], bmc_lan_arp_interval_str));
            dirty = 1;
            break;
        case 12: /* default gateway address */
            CHECK_PDATA_LEN(4);
            SET_CFG(IP_V4_TO_CFG(pdata, network_gateway_str));
            dirty = 1;
            break;
        case 13: /* default gateway MAC address */
            CHECK_PDATA_LEN(6);
            SET_CFG(MAC_TO_CFG(pdata, bmc_lan_gateway_mac_str));
            dirty = 1;
            break;
        case 14: /* backup gateway address */
            CHECK_PDATA_LEN(4);
            SET_CFG(IP_V4_TO_CFG(pdata, bmc_lan_backup_gateway_str));
            dirty = 1;
            break;
        case 15: /* backup gateway MAC address */
            CHECK_PDATA_LEN(6);
            SET_CFG(MAC_TO_CFG(pdata, bmc_lan_backup_gateway_mac_str));
            dirty = 1;
            break;
        case 16: /* community string */
            CHECK_PDATA_LEN(18);
            strncpy(val, pdata, 18);
            val[18] = '\0';
            SET_CFG(pp_cfg_set(val, bmc_lan_alert_community_str));
            dirty = 1;
            break;
        case 17: /* number of destinations */
            /* read only parameter (not specified!) */
            ierr = IPMI_ERR_LANCONF_READ_ONLY;
            break;
        case 18: /* destination type */
            CHECK_PDATA_LEN(4);
            pdata[0] &= 0x0f; // mask out reserved bits
            /* enable if pdata[1] & 0x80 == 0x80 */
            SET_CFG(pp_cfg_set_enabled(pdata[1] & 0x80,
                            bmc_lan_alert_dest_acknowledge_str, pdata[0]));
            /* MAIL if pdata[1] & 0x07 != 0x00, as we only support OEM1 */
            if((pdata[1] & 0x07) == 0x00) {
                SET_CFG(pp_cfg_set("PET", bmc_lan_alert_dest_type_str, pdata[0]));
            } else if((pdata[1] & 0x07) == 0x06) {
                SET_CFG(pp_cfg_set("MAIL", bmc_lan_alert_dest_type_str, pdata[0]));
            } else {
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
                break;
            }
            SET_CFG(pp_cfg_set_int(pdata[2], bmc_lan_alert_dest_timeout_str, pdata[0]));
            SET_CFG(pp_cfg_set_int(pdata[3] & 0x07, bmc_lan_alert_dest_retry_no_str, pdata[0]));
            dirty = 1;
            break;
        case 19: /* destination addresses */
            CHECK_PDATA_LEN(13);
            pdata[0] &= 0x0f; // mask out reserved bits
            if((pdata[1] & 0xf0) != 0x00) { // mask out reserved bits
                /* we only provide IPv4 + MAC addressing (see spec) */
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
                break;
            }
            SET_CFG(pp_cfg_set_enabled(pdata[2] & 0x01,
                            bmc_lan_backup_gateway_enabled_str));
            SET_CFG(IP_V4_TO_CFG_CORE(&pdata[3], bmc_lan_alert_dest_trap_addr_str, pdata[0]));
            // MAC is not stored, we always use arp cache and default gateway

            dirty = 1;
            break;
        // case 20: /* VLAN id */
            /* parameter not supported (at the moment) */
        // case 21: /* VLAN priority */
            /* parameter not supported (at the moment) */
        case 22: /* RMCP+ messaging cifer suite entry support */
            ierr = IPMI_ERR_LANCONF_READ_ONLY;
            break;
        case 23: /* RMCP+ messaging cifer suite entries */
            ierr = IPMI_ERR_LANCONF_READ_ONLY;
            break;
        case 24: /* RMCP+ messaging cifer suite privilege levels */
            CHECK_PDATA_LEN(9);
            for(i = 0; i < 15; ++i) {
                if (PP_FAILED(lanconf_set_priv_nibble(i, &pdata[1]))) {
                    ierr = IPMI_ERR_UNSPECIFIED;
                    break;
                }
            }
            dirty = 1;
            break;
        // case 25: /* destination address VLAN tags */
            /* parameter not supported (at the moment) */
        case  192: /* OEM command, set email alert address */
            CHECK_PDATA_LEN(65);
            pdata[0] &= 0x0F;
            // this field is quite long, maybe we should split this in several blocks ?
            strncpy(val, pdata + 1, 64);
            val[64] = 0;
            SET_CFG(pp_cfg_set(val, "bmc.lan.alert.dest[%u].mail_dest", pdata[0]));
            dirty = 1;
            break;
        default: /* parameter not supported */
            ierr = IPMI_ERR_LANCONF_PARAM_NOT_SUPPORTED;
    }

    if (dirty && ierr == IPMI_ERR_SUCCESS) {
        if (PP_FAILED(pp_cfg_tx_commit(cfg_trans))) {
            pp_bmc_log_error("[LANCONF] can't commit cfg transaction");
            ierr = IPMI_ERR_UNSPECIFIED;
        }
        pp_cfg_save(DO_FLUSH);
    } else {
        pp_cfg_tx_rollback(cfg_trans);
    }

    return pp_bmc_router_resp_err(imsg, ierr);
}

#define GET_CFG(c) \
    if (PP_FAILED(c)) { \
        pp_bmc_log_error("[LANCONF] can't read param #%d from cfg", param); \
        ierr = IPMI_ERR_UNSPECIFIED; \
        break; \
    }

static int lanconf_cmd_get_conf(imsg_t *imsg) {
    /* local data buffer */
    unsigned char pdata[256];
    int psize = 0;

    /* IPMI message data */
    unsigned char channel = imsg->data[0] & 0x0f;
    unsigned char param = imsg->data[1];
    unsigned char set_sel = imsg->data[2];
    char *resp;
    unsigned char ierr = IPMI_ERR_SUCCESS;

    unsigned short us;
    char *val;
    int i;

    if(channel != IPMI_CHAN_LAN) {
        pp_bmc_log_warn("[LANCONF] invalid lan channel id %u, should be %u",
                        channel, IPMI_CHAN_LAN);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    if (imsg->data[0] & 0x80) {
        // return param rev only
        psize = 0;
    } else {
        pp_bmc_log_info("[LANCONF] get param %u", param);
        switch (param) {
            case 0: /* set in progress */
                psize = 1;
                pdata[0] = lanconf_set_in_progress & 0x03;
                break;
            case 1: /* authentication type support */
                psize = 1;
                /* all implemented types are supported */
                pdata[0] = LANCONF_AUTH_IMPLEMENTED;
                break;
            case 2: /* authentication type enables */
                psize = 5;
                pp_cfg_get_short_nodflt(&us,
                    "bmc.channel_access[%u].authentication_type_enable[%s]",
                    channel, PP_CD_IPMIPRIVLEVEL_CALLBACK_STR);
                pdata[0] = 0x37 & us;
                pp_cfg_get_short_nodflt(&us,
                    "bmc.channel_access[%u].authentication_type_enable[%s]",
                    channel, PP_CD_IPMIPRIVLEVEL_USER_STR);
                pdata[1] = 0x37 & us;
                pp_cfg_get_short_nodflt(&us,
                    "bmc.channel_access[%u].authentication_type_enable[%s]",
                    channel, PP_CD_IPMIPRIVLEVEL_OPERATOR_STR);
                pdata[2] = 0x37 & us;
                pp_cfg_get_short_nodflt(&us,
                    "bmc.channel_access[%u].authentication_type_enable[%s]",
                    channel, PP_CD_IPMIPRIVLEVEL_ADMINISTRATOR_STR);
                pdata[3] = 0x37 & us;
                pp_cfg_get_short_nodflt(&us,
                    "bmc.channel_access[%u].authentication_type_enable[%s]",
                    channel, PP_CD_IPMIPRIVLEVEL_OEM_STR);
                pdata[4] = 0x37 & us;
                break;
            case 3: /* IP address */
                psize = 4;
                IP_V4_FROM_CFG(pdata, network_ip_addr_str);
                break;
            case 4: /* IP address source */
                psize = 1;
                GET_CFG(pp_cfg_get_int(&i, network_ip_addr_source_str));

                // check describing value agains actual setting
                GET_CFG(pp_cfg_get(&val, network_ip_auto_config_proto_str));
                if (strcmp(val, PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_DHCP_STR) == 0) {
                    i = 2; // DHCP
                } else if (strcmp(val, PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_BOOTP_STR) == 0) {
                    i = 4; // other (here: BOOTP)
                } else {
                    if (i == 2 || i == 4) i = 1; // static
                }
                free(val);

                pdata[0] = i;
                break;
            case 5: /* MAC address */
                psize = 6;
                MAC_FROM_CFG(pdata, network_mac_str);
                break;
            case 6: /* subnet mask */
                psize = 4;
                IP_V4_FROM_CFG(pdata, network_netmask_str);
                break;
            case 7: /* IPv4 header parameters */
                psize = 3;
                /* we only provide defaults */
                pdata[0] = 0x40;
                pdata[1] = 0x40;
                pdata[2] = 0x10;
                break;
            case 8: /* primary RMCP port number (optional) */
                psize = 2;
                /* default: RMCP Aux Bus Shunt port */
                pdata[0] = 0x02;
                pdata[1] = 0x6f;
                break;
            case 9: /* secondary RMCP port number (optional) */
                psize = 2;
                /* default: RMCP Secure Aux Bus port */
                pdata[0] = 0x02;
                pdata[1] = 0x98;
                break;
            case 10: /* BMC-generated ARP control (optional) */
                psize = 1;
                // we don't support controlling ARP responses, it is always ON
                pdata[0] = 0x02;
                // Gratuitous ARP
                GET_CFG(pp_cfg_is_enabled(&i, bmc_lan_arp_generate_str));
                if (i) pdata[0] |= 0x01;
                break;
            case 11: /* gratuitous ARP interval (optional) */
                psize = 1;
                GET_CFG(pp_cfg_get_int(&i, bmc_lan_arp_interval_str));
                pdata[0] = i;
                break;
            case 12: /* default gateway address */
                psize = 4;
                IP_V4_FROM_CFG(pdata, network_gateway_str);
                break;
            case 13: /* default gateway MAC address */
                psize = 6;
                MAC_FROM_CFG(pdata, bmc_lan_gateway_mac_str);
                break;
            case 14: /* backup gateway address */
                psize = 4;
                IP_V4_FROM_CFG(pdata, bmc_lan_backup_gateway_str);
                break;
            case 15: /* backup gateway MAC address */
                psize = 6;
                MAC_FROM_CFG(pdata, bmc_lan_backup_gateway_mac_str);
                break;
            case 16: /* community string */
                psize = 18;
                memset(pdata, 0, 18);
                GET_CFG(pp_cfg_get(&val, bmc_lan_alert_community_str));
                /* copy at most 18 bytes from string without tailing \0 */
                strncpy(pdata, val, 18);
                free(val);
                break;
            case 17: /* number of destinations */
                psize = 1;
                GET_CFG(pp_cfg_get_int(&i, bmc_lan_alert_dest_sz_str));
                pdata[0] = 0x0f & i;
                break;
            case 18: /* destination type */
                psize = 4;
                pdata[0] = set_sel & 0x0f; // mask out reserved bits
                pp_cfg_is_enabled(&i, bmc_lan_alert_dest_acknowledge_str,
                    pdata[0]);
                pdata[1] = i << 7;
                if(PP_ERR != pp_cfg_get_nodflt(&val,
                                            bmc_lan_alert_dest_type_str,
                                            pdata[0])) {
                    if(!strcmp(val, "MAIL")) {
                        pdata[1] |= 0x06; /* set OEM1 param */
                    } /* else do not set anything, default is PET */
                    free(val);
                }
                pp_cfg_get_int(&i, bmc_lan_alert_dest_timeout_str, pdata[0]);
                pdata[2] = i;
                pp_cfg_get_int(&i, bmc_lan_alert_dest_retry_no_str, pdata[0]);
                pdata[3] = i & 0x07; // mask out reserved bits
                break;
            case 19: /* destination addresses */
                psize = 13;
                pdata[0] = set_sel & 0x0f; // mask out reserved bits
                pdata[1] = 0x00; // default, IPv4 address + MAC
                pp_cfg_is_enabled(&i, bmc_lan_backup_gateway_enabled_str);
                pdata[2] = 0x01 & i; // mask out reserved bits
                IP_V4_FROM_CFG_CORE(&pdata[3], bmc_lan_alert_dest_trap_addr_str, pdata[0]);
                /* TODO! get destination MAC address from ARP */
                memset(&pdata[7], 0, 6);
                break;
//            case 20: /* VLAN id */
                /* parameter not supported (at the moment) */
//            case 21: /* VLAN priority */
                /* parameter not supported (at the moment) */
            case 22: /* RMCP+ messaging cifer suite entry support */
                psize = 1;
                pdata[0] = PP_BMC_LAN_CIPHER_SUITE_COUNT; // 0..3, 6..8, 11, 12
                break;
            case 23: /* RMCP+ messaging cipher suite entries */
                psize = 1 + PP_BMC_LAN_CIPHER_SUITE_COUNT;
                pdata[0] = 0x00; /* reserved */
                for (i = 0; i < PP_BMC_LAN_CIPHER_SUITE_COUNT; i++) {
                    pdata[1 + i] = pp_bmc_lan_enabled_cipher_suites[i].cipher_suite_id;
                }
                break;
            case 24: /* RMCP+ messaging cipher suite privilege levels */
                psize = 9;
                memset(pdata, 0, 9); /* initialize rs-bitfield to 0 */
                /* pdata[0] = 0x00; reserved */
                for(i = 0; i < 15; ++i) {
                    lanconf_get_priv_nibble(i, &pdata[1]);
                }
                break;
    //        case 25: /* destination address VLAN tags */
                /* parameter not supported (at the moment) */
            case  192: /* OEM command, set email alert address */
                set_sel = set_sel & 0x0F;
                // this is quite long for one response, maybe we should split this in several blocks ?
                psize = 65;
                pdata[0] = set_sel;
                memset((pdata+1), 0, 64);
                if (pp_cfg_get_nodflt(&val, "bmc.lan.alert.dest[%u].mail_dest", set_sel) == PP_SUC) {
                    strncpy((pdata+1), val, 64);
                    free(val);
                }
                break;
            default: /* parameter not supported */
                ierr = IPMI_ERR_LANCONF_PARAM_NOT_SUPPORTED;
        }
    }

    if (ierr != IPMI_ERR_SUCCESS) {
        return pp_bmc_router_resp_err(imsg, ierr);
    }

    resp = pp_bmc_imsg_resp(imsg, ierr, psize + 1);
    resp[0] = 0x11;
    memcpy(resp + 1, pdata, psize);

    return pp_bmc_router_send_msg(imsg);
}

static int lanconf_cmd_suspend_arp(imsg_t *imsg) {
    unsigned char ierr = IPMI_ERR_SUCCESS;
    char result = 0;
    char *imsg_data;

    if (imsg->data_size != 1) {
	ierr = IPMI_ERR_REQ_DATA_LEN_INVALID;
    } else {
	result = pp_lan_arp_suspend(imsg->data[0] & 0x01);
    }

    imsg_data = pp_bmc_imsg_resp(imsg, ierr, 1);
    imsg_data[0] = result;
    return pp_bmc_router_send_msg(imsg);
}

/********************************************************************
 * PEF device c'tor/d'tor
 */

static const dev_cmd_entry_t lanconf_cmd_tab[] = {
    {
        .cmd_hndlr = lanconf_cmd_set_conf, 
        .netfn = IPMI_NETFN_TRANSPORT,
        .cmd = IPMI_CMD_SET_LAN_CONFIGURATION,
        .min_data_size = 3,
        .min_priv_level = IPMI_PRIV_ADMIN 
    },
    {
        .cmd_hndlr = lanconf_cmd_get_conf, 
        .netfn = IPMI_NETFN_TRANSPORT,
        .cmd = IPMI_CMD_GET_LAN_CONFIGURATION,
        .min_data_size = 4,
        .min_priv_level = IPMI_PRIV_OPERATOR 
    },
    {
        .cmd_hndlr = lanconf_cmd_suspend_arp,
        .netfn = IPMI_NETFN_TRANSPORT,
        .cmd = IPMI_CMD_SUSPEND_BMC_ARP,
        .min_data_size = 1,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    { .cmd_hndlr = NULL }
};

int pp_lan_dev_conf_init() {
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(lanconf_cmd_tab)) return PP_ERR;

    pp_bmc_log_info("[LANCONF] device started");

    return PP_SUC;
    
}

void pp_lan_dev_conf_cleanup() {
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(lanconf_cmd_tab);

    pp_bmc_log_info("[LANCONF] device shut down");
}

