/**
 * lan_lowlevel.c
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * The lowlevel routines of the lan adapter. This code is
 * responsible for managing the socket and receiving
 * ip/udp packets either via a dedicated interface or
 * from the sideband link to the hosts nic.
 */

#include <poll.h>

#include <pp/selector.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/bmc/debug.h>

#include <pp/bmc/lan_serv.h>
#include <pp/bmc/lan_lowlevel.h>

/* whats the correct maximum input size ? */
#define LAN_RAW_FRAME_SIZE 1024

typedef struct {
    int fd;     // the sockets fd
    int sel_h;  // select_handler
    
    pp_bmc_lan_receive_handle_t receiver;
} lan_lowlevel_globals_t;

static lan_lowlevel_globals_t lan_lowlevel_globals;
static lan_lowlevel_globals_t *llg;


/**
 * Called by selector whenever a new incoming
 * packet can be received.
 */
static int handle_receive(int itemid UNUSED, int fd UNUSED, short event UNUSED, void* context UNUSED) {
    unsigned char data[LAN_RAW_FRAME_SIZE];
    struct sockaddr addr;
    socklen_t addr_len;
    int len;
    addr_len = sizeof(struct sockaddr);
    
    len = recvfrom(llg->fd, data, LAN_RAW_FRAME_SIZE, 0, &(addr), &(addr_len));
    if (len < 0) {
        pp_bmc_log_pdebug("[LAN] receive failed");
        return PP_ERR;
    }

    if (llg->receiver == NULL)
        return PP_SUC;
    else
        return llg->receiver(data, len, &addr, &addr_len);
}

/**
 * send the specified payload as udp packet
 */
int pp_bmc_lan_lowlevel_send(unsigned char* data, int len, struct sockaddr* addr, socklen_t* addr_len) {
    if (sendto(llg->fd, data, len, 0, addr, *addr_len) == len)
        return PP_SUC;
    else
        return PP_ERR;
}

/**
 * Initialize a raw socket.
 */
static int init_socket(void)
{
    int rv;
    int on = 1;
    int retry;
    
    struct sockaddr_in saddr;
    
    unsigned short port;
    if (PP_FAILED(pp_cfg_get_short_nodflt(&port, "bmc.lan.port"))) {
        port = IPMI_LAN_PORT;
    }

    memset (&saddr, 0, sizeof(struct sockaddr_in));
    saddr.sin_family = AF_INET;
    saddr.sin_port = htons(port);
    saddr.sin_addr.s_addr = inet_addr("0.0.0.0");  // obsolete, use atoi_addr()
    
    llg->fd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (llg->fd < 0) {
        pp_bmc_log_pfatal("[LAN] could not create socket");
        return PP_ERR;
    }

    rv = setsockopt(llg->fd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on));
    if (rv == -1) {
        pp_bmc_log_pfatal("[LAN] could not set reuse option on socket");
        return PP_ERR;
    }

    retry = 0;
rebind:
    rv = bind(llg->fd, &saddr, sizeof(struct sockaddr_in));
    if (rv == -1) {
        if (retry < 15) {
            usleep(1000000);
            retry++;
            goto rebind;
        }
        pp_bmc_log_pfatal("[LAN] could not bind socket");
        return PP_ERR;
    }

    return PP_SUC;
}

void pp_bmc_lan_lowlevel_set_handler(pp_bmc_lan_receive_handle_t lan_rcv_handler) {
    llg->receiver = lan_rcv_handler;
}

/**
 * Initialize the LAN lowlevel interface (hardware interface)
 * (Register in selector, etc)
 */
int pp_bmc_lan_lowlevel_init() {
    llg = &lan_lowlevel_globals;
    
    llg->receiver = NULL;
    
    if (init_socket() != PP_SUC) {
        return PP_ERR;
    }
    
    llg->sel_h = pp_select_add_fd(llg->fd, POLLIN , handle_receive, NULL);
    
    return PP_SUC;
}

/**
 * Cleanup the LAn interface
 */
int pp_bmc_lan_lowlevel_cleanup() {
    close (llg->fd);

    return PP_SUC;
}

