/**
 * lan_serv.c
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * A lan channel adapter for the pp_bmc_core.
 */

#include <pp/base.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/lan_serv.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/lan_serv.h>
#include <pp/bmc/utils.h>

#include <pp/bmc/lan_lowlevel.h>
#include <pp/bmc/session_manager.h>

#include "lan_dev_conf.h"
#include "lan_cmds.h"
#include "lan_sol.h"
#include "lan_sol_cmds.h"
#include "lan_ipmi15.h"
#include "lan_ipmi20.h"
#include "rmcp.h"
#include "lan_alert.h"
#include "lan_arp.h"
#include "lan_serv_intern.h"

static const pp_bmc_router_chan_adapter_config_t* lan_channel_config = NULL;

int pp_bmc_lanserv_ipmi_handle_receive(unsigned char* data, int ipmi_msg_len,
                                       imsg_session_t* session,
				       unsigned char userlevel_only, 
                                       lan_addr_t *addr)
{
    imsg_t *imsg;
    
    // check msg
    if (data[2] != pp_bmc_calc_ipmi_checksum(data, 0, 1))
    {
        pp_bmc_log_warn("[LAN] received imsg with invalid checksum 1");
        return PP_ERR;
    }
    if (data[ipmi_msg_len-1] != pp_bmc_calc_ipmi_checksum(data, 3, ipmi_msg_len-2))
    {
        pp_bmc_log_warn("[LAN] received imsg with invalid checksum 2 (msglen=%d)", ipmi_msg_len);
        return PP_ERR;
    }
    
    // construct the imsg

    imsg = pp_bmc_imsg_new(ipmi_msg_len - 7);

    imsg->buf = malloc(sizeof(lan_addr_t));
    imsg->buf_size = sizeof(lan_addr_t);

    imsg->rs_addr =  data[0];
    imsg->netfn   = (data[1] >> 2) & 0x3F;
    imsg->rs_lun  = (data[1]) & 0x03;
    imsg->rq_addr =  data[3];
    imsg->rq_seq  = (data[4] >> 2) & 0x3F;
    imsg->rq_lun  = (data[4]) & 0x03;

    imsg->cmd     =  data[5];
    memcpy(imsg->data, data+6, imsg->data_size);

    // other values in imsg
    imsg->chan = IPMI_CHAN_LAN;
    imsg->session = session;
    if (session != NULL) session->ref_count++;  /* reassigning the session is like making a copy, so increase refcount  */
    
    if (session == NULL)
        imsg->priv_level = IPMI_PRIV_UNSPEC;
    else {
        imsg->priv_level = session->cur_priv_level;
        if (userlevel_only == 1) {
            if (imsg->priv_level > IPMI_PRIV_USER) {
                /* unauthenticated message with 'no_user_level_authentication' */
                /* limit message to user level                                 */
                imsg->priv_level = IPMI_PRIV_USER;
            }
        }

        // also store lan addr in session
        if(!session->addr) {
            session->addr = malloc(sizeof(lan_addr_t));
        }
        memcpy(session->addr, addr, sizeof(lan_addr_t));
    }
    
    memcpy(imsg->buf, addr, sizeof(lan_addr_t));
    
    return pp_bmc_core_handle_msg(imsg);
}

static int pp_lan_ipmi_send_msg(imsg_t* imsg) {
    unsigned char* buf;
    int rv;
    int msglen;
    
    buf = malloc(imsg->data_size + 7);
    
    // construct imsg in buf
    // caveat: the completion code is part of imsg->data
    msglen = imsg->data_size+7;
    buf[0] = imsg->rq_addr;
    buf[1] = (imsg->netfn << 2) | imsg->rs_lun;
    buf[2] = pp_bmc_calc_ipmi_checksum(buf,0,1);
    buf[3] = imsg->rs_addr;
    buf[4] = (imsg->rq_seq << 2) | imsg->rs_lun;
    buf[5] = imsg->cmd;
  //msgbuffer[6] = imsg->data[0];   // completion code - this byte is copied with data
    memcpy(buf+6, imsg->data, imsg->data_size);
    buf[imsg->data_size+6] = pp_bmc_calc_ipmi_checksum(buf,3,imsg->data_size+5);

    if (imsg->session == NULL) {
        if ( (((lan_addr_t*)imsg->buf)->ipmi_version) == IPMI_LAN_VERSION_15) {
            rv = pp_ipmi15_send_msg(buf, msglen, NULL, (lan_addr_t*)imsg->buf);
        } else {
            rv = pp_ipmi20_send_payload(buf, msglen, IPMI_PAYLOAD_IPMI, NULL, (lan_addr_t*)imsg->buf);
        }
    } else {
        if (imsg->session->authtype == IPMI_AUTH_RMCPP) {
            // IPMIv20 message
            rv = pp_ipmi20_send_payload(buf, msglen, IPMI_PAYLOAD_IPMI, imsg->session, (lan_addr_t*)imsg->buf);
        } else {
            // should be an IPMIv15 message
            rv = pp_ipmi15_send_msg(buf, msglen, imsg->session, (lan_addr_t*)imsg->buf);
        }
    }
    
    pp_bmc_imsg_delete(imsg);
    free(buf);
    
    return rv;
}

/**
 * Returns pointer to the socket address information stored in a given session
 *
 * @param session desired session
 * @return NULL if not a lan channel session, 
 *         assigned socket address information otherwhise
 */
struct sockaddr_in* pp_bmc_lanserv_get_sockaddr(imsg_session_t* session) {
    assert(session);
    
    if(session->chan != IPMI_CHAN_LAN) {
        return NULL;
    }
    
    return((struct sockaddr_in*)&session->addr->addr);
}

const
pp_bmc_router_chan_adapter_config_t* get_lan_channel_config() {
    return lan_channel_config;
}

static void lan_update_config(void) {
    pp_bmc_log_debug("[LAN] configuration update handler not implemented");
    // TODO: forward internally
}

static const pp_bmc_router_chan_adapter_info_t lan_chan_info = {
    .medium =       IPMI_CHAN_MED_LAN,
    .proto =        IPMI_CHAN_PROT_IPMB,
    .sess_supp =    IPMI_CHAN_SESS_MULTI,
};

int pp_bmc_lanserv_init() {
    if (pp_bmc_lan_lowlevel_init() != PP_SUC) {
        return PP_ERR;
    }    
    
    pp_bmc_lan_lowlevel_set_handler(pp_rmcp_handle_receive);

    lan_channel_config = pp_bmc_router_reg_chan(IPMI_CHAN_LAN, pp_lan_ipmi_send_msg,
                               NULL, lan_update_config, &lan_chan_info);
                               
    if ((lan_channel_config == NULL)
        || PP_FAILED(pp_lan_dev_conf_init())
#ifndef PP_FEAT_BMC_OEMCMDS_ONLY
        || PP_FAILED(pp_lan_cmd_init())
        || PP_FAILED(pp_sol_init())
        || PP_FAILED(pp_sol_cmd_init())
        || PP_FAILED(pp_bmc_lan_alert_init())
#endif
        || PP_FAILED(pp_lan_arp_init())) {
            return PP_ERR;
    }

    pp_bmc_log_info("[LAN] channel initialized");

    return PP_SUC;
}

void pp_bmc_lanserv_cleanup() {
    pp_lan_arp_cleanup();
#ifndef PP_FEAT_BMC_OEMCMDS_ONLY
    pp_bmc_lan_alert_cleanup();
    pp_sol_cmd_cleanup();
    pp_sol_cleanup();
    pp_lan_cmd_cleanup();
#endif
    pp_lan_dev_conf_cleanup();
    pp_bmc_router_unreg_chan(IPMI_CHAN_LAN);

    if (pp_bmc_lan_lowlevel_cleanup() != PP_SUC) {
        pp_bmc_log_notice("[LAN] failed to cleanup lowlevel functions");
    }
    pp_bmc_log_info("[LAN] channel shut down");
}
