/**
 * lan_sol.c
 *
 * Serial over LAN implementation
 * 
 * (c) 2005 Peppercon AG, 2005/4/29, tbr@peppecon.de
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>

#include <pp/base.h>
#include <pp/selector.h>
#include <pp/termios.h>
#include <pp/rbuf.h>
#include <pp/cfg.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/serial_lowlevel.h>

#include "lan_ipmi20.h"
#include "lan_sol.h"

/*
 * local definitions
 */
#define noDEBUG
#ifdef DEBUG
# define D(fmt, args...) printf(fmt, ##args);
#else
# define D(fmt, args...)    
#endif

#define IN_RBUF_SIZE               (10 * 1024)
#define OUT_RBUF_SIZE              1024
#define TRUE                       1
#define FALSE                      0
#define SOL_MAX_DATA_LEN           224
#define SOL_RETRY_TO_2_MS(to)      (10 * (to))
#define SOL_ACCU_TO_2_MS(to)       (5 * (to))

/*
 * SOL payload BMC to Remote Console
 */
struct sol_payload_bmc2rc_s {
    unsigned char seq_no;
    unsigned char ack_seq_no;
    unsigned char accept_char_cnt;
    BITFIELD7(unsigned char,      \
	      reserved0 : 2,      \
	      break_cond : 1,     \
	      overflow : 1,       \
	      sol_deactive : 1,   \
	      sol_unavailable : 1,\
	      is_nack : 1,        \
	      reserved1 : 1);
    char data[0];
} __attribute__ ((packed));
typedef struct sol_payload_bmc2rc_s sol_payload_bmc2rc_t;

/*
 * SOL payload Remote Console to BMC
 */
struct sol_payload_rc2bmc_s {
    unsigned char seq_no;
    unsigned char ack_seq_no;
    unsigned char accept_char_cnt;
    BITFIELD8(unsigned char,       \
	      flush_outbound : 1,  \
	      flush_inbound : 1,   \
	      deassert_dcd_dsr : 1,\
	      deassert_pause : 1,  \
	      gen_break : 1,       \
	      ring_wor  : 1,       \
	      is_nack : 1,         \
	      reserved : 1);
    char data[0];
} __attribute__ ((packed));
typedef struct sol_payload_rc2bmc_s sol_payload_rc2bmc_t;

/*
 * SOL config options
 */
typedef struct sol_config_s {
    unsigned char send_threshold;  // number of chars
    unsigned char char_accumulate; // 5ms increment
    unsigned char retry_cnt;       // max 7, 0 = no retries
    unsigned char retry_to;        // 10ms increment, 0 = back-to-back (stupid)
    int tty_speed;
} sol_config_t;

/*
 * state machines states and transitions
 */
typedef enum sol_sender_state_e {
    SOLS_STATE_ACCU,        // accumulating data in buffer
    SOLS_STATE_ACKPENDING,  // data send, expect ack
    SOLS_STATE_SUSPENDED    // suspended
} sol_sender_state_t;

typedef enum sol_sender_trans_e {
    SOLS_TRANS_SEND_DATA,     // data ready for send
    SOLS_TRANS_SEND_ACK,      // send an ack
    SOLS_TRANS_ACK_DATA,      // ack has been received
    SOLS_TRANS_NACK_DATA,     // nack has been received
    SOLS_TRANS_RETRY_TIMEOUT, // timeout for data packet occured
    SOLS_TRANS_ACCU_TIMEOUT,  // char accumulation timeout
} sol_sender_trans_t;

typedef struct sol_sender_s {
    sol_sender_state_t state;
    unsigned char seq_no;      // only bits [3:0], 0x0 = ACK-only packet
    int accu_to_hndl;
    int retry_to_hndl;
    unsigned char retry_cnt;
    int send_pending;
    int flush_pending;
    sol_payload_bmc2rc_t* pkt;
    unsigned short pkt_data_len;
    sol_payload_bmc2rc_t* ackpkt;
    imsg_session_t* session;
} sol_sender_t;

typedef enum sol_recver_state_e {
    SOL_RECV_RECEIVING,
    SOL_RECV_SUSPENDING,
} sol_recver_state_t;

typedef struct sol_recver_s {
    sol_recver_state_t state;
    unsigned char seq_no;
    unsigned char accept_cnt;
    int buf_avail_hndl;
} sol_recver_t;

/*
 * used config keys
 * -------------------
 */
const char* PP_BMC_SOL_CFG_IS_ENABLED     = "bmc.sol.is_enabled";
const char* PP_BMC_SOL_CFG_ACCU_INTERVAL  = "bmc.sol.accu_interval";
const char* PP_BMC_SOL_CFG_SEND_TRESHOLD  = "bmc.sol.send_threshold";
const char* PP_BMC_SOL_CFG_RETRY_CNT      = "bmc.sol.retry_count";
const char* PP_BMC_SOL_CFG_RETRY_INTERVAL = "bmc.sol.retry_interval";
const char* PP_BMC_SOL_CFG_TTY_SPEED      = "bmc.sol.tty_speed";

/*
 * private prototypes and variables
 * ----------------------------------
 */
static void smux_activate_sol_hndl(pp_bmc_serial_mux_state_t oldstate);
static void smux_deactivate_sol_hndl(pp_bmc_serial_mux_state_t newstate);

static void sol_deactivate(imsg_session_t* session);
static sol_config_t* sol_config_load(void);
static void sol_config_print(sol_config_t* cfg);
static unsigned char seq_no_inc(unsigned char seq_no);
static unsigned char seq_no_dec(unsigned char seq_no);

/* reader */
static int  sol_reader_handle_ser_input(const int item_id, const int fd, 
					const short event, void* ctx);
/* sender */
static inline void sol_sender_new_data(int flush);
static inline void sol_sender_ack_pkt(unsigned char pkt_seq_no,
				      unsigned char pkt_accept_cnt);
static inline void sol_sender_nack_pkt(unsigned char pkt_seq_no,
				      unsigned char pkt_accept_cnt);
static inline void sol_sender_pkt_acked(unsigned char pkt_seq_no,
					unsigned char pkt_accept_cnt);
static inline void sol_sender_pkt_nacked(unsigned char pkt_seq_no,
					 unsigned char pkt_accept_cnt);
static void sol_sender_process(sol_sender_trans_t action, int flush,
			       int ack_is_nack, unsigned char ack_seq_no,
			       unsigned char ack_accept_cnt);
static void sol_sender_sm_error(sol_sender_state_t s, sol_sender_trans_t t);
static inline void sol_sender_set_accu_timeout(void);
static inline void sol_sender_clear_accu_timeout(void);
static inline void sol_sender_set_retry_timeout(void);
static inline int sol_sender_check_send_tresh(int flush);
static void sol_sender_resume(int flush_pending);
static int  sol_sender_handle_accumulate_to(const int item_id, void* ctx);
static int  sol_sender_handle_retry_to(const int item_id, void* ctx);
static inline void sol_sender_send_data_pkt(void);
static void sol_sender_send_ackdata_pkt(int do_ack, int ack_is_nack,
					unsigned char ack_seq_no,
					unsigned char ack_accept_cnt);
static void sol_sender_send_ack_only_pkt(int ack_is_nack,
					 unsigned char ack_seq_no,
					 unsigned char ack_accept_cnt);
static size_t sol_sender_prepare_data(sol_payload_bmc2rc_t* pkt);
static inline size_t sol_sender_prepare_data_empty(sol_payload_bmc2rc_t* pkt);
static void sol_sender_prepare_ack(sol_payload_bmc2rc_t* pkt, int ack_is_nack,
				   unsigned char ack_seq_no,
				   unsigned char ackpkt_accept_cnt);
static void sol_sender_prepare_ack_empty(sol_payload_bmc2rc_t* pkt);
static void sol_sender_prepare_ctrl(sol_payload_bmc2rc_t* pkt);
static void sol_sender_append_pkt_data(void);
static inline void sol_sender_send_off_data_pkt(void);
static inline void sol_sender_send_off_ack_only_pkt(void);
static inline void sol_sender_send_off_pkt(sol_payload_bmc2rc_t* pkt,
					   size_t pkt_len);

/* receiver */
static int sol_recver_handle_lan_input(unsigned char* buf, size_t len,
				       imsg_session_t* session, 
				       lan_addr_t* addr);
static void sol_recv_sm_error(sol_recver_state_t s, unsigned char seq_no);
static int sol_recver_handle_data(sol_payload_rc2bmc_t* pkt,
				  size_t pkt_data_len);
static int sol_recver_buf_avail_cb(const int item_id, const int fd, 
				   const short event, void* ctx);
static void sol_recver_handle_ctrl(sol_payload_rc2bmc_t* pkt);
static void sol_recver_handle_acks(sol_payload_rc2bmc_t* pkt);
static size_t sol_writer_new_data(char* buf, size_t len);

#ifdef DEBUG
static const char* send_state_str(sol_sender_state_t s);
static const char* send_action_str(sol_sender_trans_t t);
#endif

/*
 * lan_sol has four states:
 * - activating: smux switch initiated, waiting for activate()
 * - activated: sol session up and running
 * - deactivating: smux switch initiated (currently not possible
 *   because deactivation is one continuous call)
 * - deactivated: no active sol session, no pending sol session
 *   (the smux should be switched to another position in this state)
 */ 
typedef enum {
    PP_SOL_ACTIVATING,
    PP_SOL_ACTIVATED,
    PP_SOL_DEACTIVATING,
    PP_SOL_DEACTIVATED
} lan_sol_state_t;
static lan_sol_state_t lan_sol_state = PP_SOL_DEACTIVATED;

/* variables valid for PP_SOL_ACTIVATING */
static imsg_session_t* activate_param_session;

/* variables valid for PP_SOL_ACTIVATED, PP_SOL_DEACTIVATING */
static sol_config_t* sol_config;
static int sol_volatile_tty_speed = -1;
static pp_rbuf_t* in_buf;
static int break_cond = 0; // BREAK from host detected
// static pp_rbuf_t* out_buf; // currently we use ttySx's buf, 1024 bytes
static sol_sender_t* sol_sender = NULL;
static sol_recver_t* sol_recver = NULL;
static int sol_tty_fd;
static int sol_tty_selector_hndl;
static pp_bmc_serial_mux_state_t smux_backup = PP_SMUX_HOST_CHASSIS;

/* no valid variables for or PP_SOL_DEACTIVATED */


/*
 * general functions
 * ------------------
 */
 
int pp_sol_init() {
    pp_bmc_serial_mux_register_handler(PP_SMUX_HOST_BMC,
                                       smux_activate_sol_hndl,
                                       smux_deactivate_sol_hndl);
    pp_bmc_log_info("[SOL] SOL serial handler initialized");
    return PP_SUC;
}

void pp_sol_cleanup() {
    // FIXME: deactivate active sessions ?
    pp_bmc_serial_mux_clear_handler(PP_SMUX_HOST_BMC);
    pp_bmc_log_info("[SOL] SOL serial handler cleaned up");
}

int pp_bmc_sol_state_is_active() {
    if (lan_sol_state == PP_SOL_ACTIVATED) {
	assert(sol_sender);
	return 1;
    }
    assert(lan_sol_state == PP_SOL_DEACTIVATED);
    assert(sol_sender == NULL);
    return 0;
}

int pp_bmc_sol_get_session_id() {
    if (sol_sender != NULL) {
	assert(sol_sender->session != NULL);
	return sol_sender->session->session_id;
    }
    return 0;
}

short pp_bmc_sol_get_remote_port() {
    struct sockaddr_in * ip_address;
    if (sol_sender != NULL) {
	assert(sol_sender->session != NULL);
	assert(sol_sender->session->addr != NULL);

	ip_address = (struct sockaddr_in *) &(sol_sender->session->addr->addr);
	return ip_address->sin_port;
    }
    return 0;
}

int pp_bmc_sol_activate(unsigned char* ipmi_rs_code,
			unsigned short* max_inbound_size,
			unsigned short* max_outbound_size,
			imsg_session_t* session)
{
    // check whether already active
    if (lan_sol_state != PP_SOL_DEACTIVATED) {
        *ipmi_rs_code = IPMI_ERR_PAYLOAD_ACT_ALREADY_ACTIVE;
        return PP_ERR;
    }
    
    // check whether enabled
    int enabled;
    pp_cfg_is_enabled(&enabled, PP_BMC_SOL_CFG_IS_ENABLED);
    if (!enabled) {
        *ipmi_rs_code = IPMI_ERR_PAYLOAD_ACT_DISABLED;
        return PP_ERR;
    }

    *max_inbound_size = *max_outbound_size = 
        sizeof(sol_payload_bmc2rc_t) + SOL_MAX_DATA_LEN;
    
    /* ok, initiate switch to sol */
    lan_sol_state = PP_SOL_ACTIVATING;
    activate_param_session = session;
    
    if (pp_bmc_serial_mux_get_state() != PP_SMUX_HOST_BMC) {
        smux_backup = pp_bmc_serial_mux_get_state();
    }
    if (pp_bmc_serial_mux_switch(PP_SMUX_HOST_BMC) == PP_ERR) {
        /* cannot switch, maybe another switch is in progress */
        pp_bmc_log_debug("[SOL] cannot active SOL, mux switch was rejected");
        *ipmi_rs_code = IPMI_ERR_UNSPECIFIED;
        lan_sol_state = PP_SOL_DEACTIVATED;
        activate_param_session = NULL;
        return PP_ERR;
    }
    
    *ipmi_rs_code = IPMI_ERR_SUCCESS;
    return PP_SUC;
}

int pp_bmc_sol_deactivate(unsigned char* ipmi_rs_code,
                    imsg_session_t* session)
{
    if ((lan_sol_state != PP_SOL_ACTIVATED) && 
        (lan_sol_state != PP_SOL_ACTIVATING))
    {
        /* we are not active and cannot be deactivated (yet) */
        pp_bmc_log_debug("[SOL] deactivation request blocked - no active sol session");
        *ipmi_rs_code = IPMI_ERR_PAYLOAD_DEACT_NOT_ACTIVE;
        return PP_ERR;
    }
    
    /* activation must be elementary so this state cannot happen - check here before you change that */
    assert (lan_sol_state == PP_SOL_ACTIVATED);
    
    if (sol_sender == NULL ||
        sol_sender->session->session_id != session->session_id) {
        *ipmi_rs_code = IPMI_ERR_PAYLOAD_DEACT_NOT_ACTIVE;
        pp_bmc_log_debug("[SOL] Received invalid SOL deactivation request");
        return PP_ERR;
    }
    
    pp_bmc_log_debug("[SOL] initiating SOL deactivation to %d", smux_backup);

    assert(smux_backup != PP_SMUX_HOST_BMC);
    pp_bmc_serial_mux_switch(smux_backup);
    
    *ipmi_rs_code = IPMI_ERR_SUCCESS;
    return PP_SUC;
}

int pp_bmc_sol_get_tty_speed() {
    if (sol_volatile_tty_speed < 0) {
        pp_cfg_get_int(&sol_volatile_tty_speed, PP_BMC_SOL_CFG_TTY_SPEED);
    }
    return sol_volatile_tty_speed;
}

int pp_bmc_sol_set_tty_speed(int ttyspeed) {
#if PP_BMC_DEBUG_LEVEL>=1
    const char* m = "[SOL] set volatile tty speed";
#endif
    int r = PP_ERR;
    if (ttyspeed == 9600 || ttyspeed == 19200 || ttyspeed == 38400 ||
        ttyspeed == 57600 || ttyspeed == 115200) {
        sol_volatile_tty_speed = ttyspeed;
        if (sol_sender != NULL) {
            if (0 > pp_base_set_tty_params_ex(sol_tty_fd, ttyspeed,
                                           "", 8, 0, 0, 0, 1, TTY_BREAK_AS_NULL_CHAR)) {
                pp_bmc_log_perror("%s, set_tty_params (%d) failed",
                                  m, ttyspeed);
            } else {
                pp_bmc_log_notice("%s to %d", m, ttyspeed);
            }
        }
        r = PP_SUC;
    }

    return r;
}


/**
 * Internal functions
 */ 


/**
 * Activate SOL smux handler for serial_lowlevel.c. This function will be
 * called by serial_lowlevel.c after the mux has been switched to indicate
 * that we are now responsible for handling serial communication.
 */
static void smux_activate_sol_hndl(pp_bmc_serial_mux_state_t oldstate UNUSED) {
    imsg_session_t* session;

    /* we must be in 'ACTIVATING' state or someone messed something up */
    assert((lan_sol_state == PP_SOL_ACTIVATING) ||
           (lan_sol_state == PP_SOL_DEACTIVATED));
    
    if (lan_sol_state == PP_SOL_DEACTIVATED) {
        /* someone switched the mux without configuring SOL   *
         * print error warning and stay deactivated           */
        pp_bmc_log_warn("[SOL] sol activated without configuration by smux switch");
        return;
    }
    
    session = activate_param_session;
    // FIXME: check if the session is still active.
    // session could be deactivated during smux switch time
    // FIXME: how can we do that ? - assume it works for now
    activate_param_session = NULL;  // reset this parameter to distinguish between
                                    // requested/unrequested smux switches
    
    if (session == NULL) {
        pp_bmc_log_info("[SOL] block SOL activation with NULL session");
        lan_sol_state = PP_SOL_DEACTIVATED;
        pp_bmc_serial_mux_switch(smux_backup);
        return;
    }
    
    sol_tty_fd = open(PP_BMC_SERIAL_TTY, O_RDWR | O_NONBLOCK);
    if (0 > sol_tty_fd)
    {
	pp_bmc_log_perror("[SOL] failed: get open %s", PP_BMC_SERIAL_TTY);
        lan_sol_state = PP_SOL_DEACTIVATED;
        pp_bmc_serial_mux_switch(smux_backup);
	return;
    }
    if (0 > pp_base_set_tty_params_ex(sol_tty_fd,
                                   pp_bmc_sol_get_tty_speed(),
                                   "", 8, 0, 0, 0, 1, TTY_BREAK_AS_NULL_CHAR))
    {
        pp_bmc_log_perror("[SOL] failed to set tty params");
        // ignore this and try to go without correct tty params
    }

    // create config
    sol_config = sol_config_load();
    sol_config_print(sol_config);

    // register for LAN channel's SOL payload in session
    session->sol_handler = sol_recver_handle_lan_input;
    session->sol_deactivate = sol_deactivate;
    
    // create buffers and state
    in_buf  = pp_rbuf_create(IN_RBUF_SIZE, TRUE);

    assert(sol_sender == NULL);
    sol_sender = malloc(sizeof(sol_sender_t));
    sol_sender->state = SOLS_STATE_ACCU;
    sol_sender->seq_no = 0;
    sol_sender->accu_to_hndl = 0;
    sol_sender->retry_to_hndl = 0;
    sol_sender->send_pending = 0;
    sol_sender->flush_pending = 0;
    // allocate space for the last packet send, max space
    // is limitied by the accepted char count param of an ACk
    sol_sender->pkt = malloc(sizeof(sol_payload_bmc2rc_t) + SOL_MAX_DATA_LEN);
    // allocate space for ack only packets
    sol_sender->ackpkt = malloc(sizeof(sol_payload_bmc2rc_t));
    // session is ref cnt, but we are referenced from sesseion as well
    // session will deactivate us before it gets destroyed
    sol_sender->session = session; 

    assert(sol_recver == NULL);
    sol_recver = malloc(sizeof(sol_recver_t));
    sol_recver->state = SOL_RECV_RECEIVING;
    sol_recver->seq_no = 0;
    sol_recver->accept_cnt = 0;

    // register for serial line reads
    sol_tty_selector_hndl = pp_select_add_fd(sol_tty_fd, POLLIN,
					     sol_reader_handle_ser_input,
					     NULL);
    
    pp_bmc_log_notice("[SOL] activated on %s", PP_BMC_SERIAL_TTY);
    
    lan_sol_state = PP_SOL_ACTIVATED;
}

/**
 * Deactivate the serial mux SOL handler. This function is called by
 * serial_lowlevel to force session deactivation. As a result, we
 * must end the SOL session asap and call
 * pp_bmc_serial_mux_deactivate_finished().
 * 
 * Note: this function cannot distinguish (yet) if the SOL
 * session ended regularly or was forced away by a mux switch.
 */
static void smux_deactivate_sol_hndl(pp_bmc_serial_mux_state_t newstate UNUSED)
{
    imsg_session_t* session;
    
    if (lan_sol_state != PP_SOL_ACTIVATED) {
        /* something is wrong - we could be 
         * - activating - should not happen as activation is elementary
         * - deactivated (due to activation failure) - not a problem
         * - deactivating  - we deactivate anyway. Ignore and wait for regular deactivation to end 
         */
         
        /* activation is not elementary ! stop activation to deactivate */
        assert(lan_sol_state != PP_SOL_ACTIVATING);
        
        if (lan_sol_state != PP_SOL_DEACTIVATING) {
            lan_sol_state = PP_SOL_DEACTIVATED;
            pp_bmc_serial_mux_deactivate_finished();
        }
        
        pp_bmc_log_debug("[SOL] deactivating SOL in non-active state(%i) due to smux switch", lan_sol_state);
        return;
    }
    
    if (sol_sender == NULL) {
        /* we must have sol_sender if we are activated */
        assert(0);
    }
    session = sol_sender->session;

    // FIXME: TODO do we have to send a finish msg ?
    
    // deregister for LAN channel reads
    session->sol_handler = NULL;
    session->sol_deactivate = NULL;
    
    // cancel all pending timeout
    sol_sender_clear_accu_timeout();
    if (sol_sender->state == SOLS_STATE_ACKPENDING) {
	pp_select_remove_to(sol_sender->retry_to_hndl);
    }
    
    // deregister for serial line reads
    pp_select_remove_fd(sol_tty_selector_hndl);
    
    close(sol_tty_fd);
    sol_tty_fd = -1;

    // cleanup memory
    free(sol_recver);
    sol_recver = NULL;
    free(sol_sender->ackpkt);
    free(sol_sender->pkt);
    free(sol_sender);
    sol_sender = NULL;
    pp_rbuf_destroy(in_buf);
    free(sol_config);
    
    pp_bmc_log_notice("[SOL] deactivated on %s", PP_BMC_SERIAL_TTY);
    
    lan_sol_state = PP_SOL_DEACTIVATED;
    pp_bmc_serial_mux_deactivate_finished();
}

static void sol_deactivate(imsg_session_t* session) {
    unsigned char rs_code;
    pp_bmc_sol_deactivate(&rs_code, session);
}

static sol_config_t* sol_config_load() {
    int send_thresh, accu_to, retry_cnt, retry_to, tty_speed;
    sol_config_t* cfg = malloc (sizeof(sol_config_t));
    pp_cfg_get_int(&send_thresh, PP_BMC_SOL_CFG_SEND_TRESHOLD);
    pp_cfg_get_int(&accu_to, PP_BMC_SOL_CFG_ACCU_INTERVAL);
    pp_cfg_get_int(&retry_cnt, PP_BMC_SOL_CFG_RETRY_CNT);
    pp_cfg_get_int(&retry_to, PP_BMC_SOL_CFG_RETRY_INTERVAL);
    pp_cfg_get_int(&tty_speed, PP_BMC_SOL_CFG_TTY_SPEED);
    
    cfg->send_threshold = send_thresh;
    cfg->char_accumulate = accu_to;
    cfg->retry_cnt = retry_cnt;
    cfg->retry_to  = retry_to;
    // TODO: tty_speed = 0 is not implemented, we'll take a default
    // in that case, but is has to be that of the serial channel, actually
    if (tty_speed == 0) cfg->tty_speed = 115200;
    else cfg->tty_speed = tty_speed;
    return cfg;
}

static void sol_config_print(sol_config_t* cfg UNUSED) {
    pp_bmc_log_notice("[SOL] activating with config: "
		      "send_threshold=%hhd accu_timeout=%hhdms "
		      "retry_count=%hhd retry_timeout=%hhdms "
		      "tty_speed=%dkbps non_volatile_tty_speed=%dkps",
		      cfg->send_threshold, cfg->char_accumulate * 5,
		      cfg->retry_cnt, cfg->retry_to * 10,
		      cfg->tty_speed, pp_bmc_sol_get_tty_speed());
}


static unsigned char seq_no_inc(unsigned char seq_no) {
    unsigned char ns;
    if ((ns = seq_no + 1) > 15)
	ns = 1;
    return ns;
}

static unsigned char seq_no_dec(unsigned char seq_no) {
    unsigned char ns;
    if ((ns = seq_no - 1) < 1)
	ns = 15;
    return ns;
}

/*
 * Reader:
 * - selects seriell fd and fills in-ringbuf
 * - notifies sender of new data
 * ------------------------------------------
 */
static int sol_reader_handle_ser_input(const int item_id UNUSED, const int fd, 
				       const short event UNUSED,
				       void* ctx UNUSED) {
    char buf[256];
    int n;

    if (0 > (n = read(fd, buf, sizeof(buf)))) {
	// read went wrong
	if (errno == EINTR || errno == EAGAIN) {
	    return PP_SUC;
	} else {
	    pp_bmc_log_perror("[SOL] read(fd=%d)", fd);
	    return PP_ERR;
	}
    } else {
	// read went ok

	// check for BREAK from host
	int i;
	for (i = n - 1; i >= 0; i--) {
	    if (buf[i] == '\0') {
		// BREAK detected, remove it from stream and set flag
		break_cond = 1;
		pp_bmc_log_debug("[SOL] BREAK from host detected");
		n--;
		memmove(buf + i, buf + i + 1, n - i);
	    }
	}

	pp_rbuf_append(in_buf, buf, n);
	D("Reader_handle_ser_input n=%d\n", n);
	sol_sender_new_data(break_cond ? TRUE : FALSE);
    }
    return PP_SUC;
}

/*
 * Sender:
 * - stati
 *   - IDLE
 *   - ACCUMULATING
 *   - ACKPENDING
 *   - SUSPENDED
 * - gets trigger from Reader to send data
 * - gets trigger from Receiver to send ack
 * - observs character accumulate interval timeout
 * - observs character send threshold
 * - builds packet
 *   - prepare_payload
 *   - prepare_ack
 *   - prepare_control
 * - observes ack timeout and does resend
 * - gives packet to LAN channel by len + data
 * ------------------------------------------
 */

/* must be called if new data has arrived, i.e. by reader */
static inline void sol_sender_new_data(int flush) {
    sol_sender_process(SOLS_TRANS_SEND_DATA, flush, 0, 0, 0);
}

/* must be called if received data packet needs to be acked by receiver */
static inline void sol_sender_ack_pkt(unsigned char pkt_seq_no,
				      unsigned char pkt_accept_cnt) {
    sol_sender_process(SOLS_TRANS_SEND_ACK, 0, FALSE /* ack */,
		       pkt_seq_no, pkt_accept_cnt);
}

static inline void sol_sender_nack_pkt(unsigned char pkt_seq_no,
				      unsigned char pkt_accept_cnt) {
    sol_sender_process(SOLS_TRANS_SEND_ACK, 0, TRUE /* is nack */,
		       pkt_seq_no, pkt_accept_cnt);
}

/* must be called if ack for pkt has arrived, i.e. by receiver */
static inline void sol_sender_pkt_acked(unsigned char pkt_seq_no,
					unsigned char pkt_accept_cnt) {
    sol_sender_process(SOLS_TRANS_ACK_DATA, 0, 0, pkt_seq_no, pkt_accept_cnt);
}

static inline void sol_sender_pkt_nacked(unsigned char pkt_seq_no,
					 unsigned char pkt_accept_cnt) {
    sol_sender_process(SOLS_TRANS_NACK_DATA, 0, 0, pkt_seq_no, pkt_accept_cnt);
}

/*
 * used params depend on action
 * SOLS_TRANS_SEND_DATA: flush: forces out_bound flush
 * SOLS_TRANS_SEND_ACK:  ack_is_nack: sends nack instead of ack
 * SOLS_TRANS_ACK_DATA
 * SOLS_TRANS_NACK_DATA: ack_seq_no: sequence number to ack/nack
 *                       ack_accept_cnt: char cnt to ack/nack
 */
static void sol_sender_process(sol_sender_trans_t action, int flush,
			       int ack_is_nack, unsigned char ack_seq_no,
			       unsigned char ack_accept_cnt) {
    D("Sender_process s=%s a=%s\n", send_state_str(sol_sender->state),
      send_action_str(action));

    switch (sol_sender->state) {
	
      case SOLS_STATE_ACCU:
	  switch (action) {
	    case SOLS_TRANS_ACCU_TIMEOUT:
		sol_sender_send_data_pkt(); // -> SOLS_STATE_ACKPENDING
		break;
	    case SOLS_TRANS_SEND_DATA:
		if (sol_sender_check_send_tresh(flush)) {
		    sol_sender_send_data_pkt(); // -> SOLS_STATE_ACKPENDING
		    sol_sender_clear_accu_timeout();
		} else { // no state change
		    sol_sender_set_accu_timeout();
		}
		break;
	    case SOLS_TRANS_SEND_ACK:
		if (pp_rbuf_size(in_buf) > 0) { // -> SOLS_STATE_ACKPENDING
		    sol_sender_send_ackdata_pkt(TRUE, ack_is_nack,
						ack_seq_no, ack_accept_cnt); 
		    sol_sender_clear_accu_timeout();
		} else { // no state change
		    sol_sender_send_ack_only_pkt(ack_is_nack, ack_seq_no,
						 ack_accept_cnt);
		}
		break;
	    case SOLS_TRANS_ACK_DATA:
	    case SOLS_TRANS_NACK_DATA:
		// ignore any ack/nack that is not expected
		// SUSPEND NACKs work only for non-acked pkts (15.11)
		break;
	    default:
		sol_sender_sm_error(sol_sender->state, action);
	  }
	  break;
	  
      case SOLS_STATE_ACKPENDING:
      case SOLS_STATE_SUSPENDED:
	  switch (action) {
	    case SOLS_TRANS_SEND_DATA: // no state change
		if (!sol_sender->send_pending)
		    sol_sender_set_accu_timeout();
		if (flush) {
		    if (sol_sender->state == SOLS_STATE_SUSPENDED) {
			// get out of suspend state if flush
			sol_sender_resume(flush); // -> SOLS_STATE_ACKPENDING
		    } else {
			sol_sender->flush_pending = TRUE;
		    }
		}
		break;
	    case SOLS_TRANS_ACCU_TIMEOUT: // no state change
		sol_sender->send_pending = TRUE;
		break;
	    case SOLS_TRANS_SEND_ACK:
		sol_sender_send_ack_only_pkt(ack_is_nack,   // no state change
					     ack_seq_no, ack_accept_cnt);
		break;
	    case SOLS_TRANS_ACK_DATA:
		// ignore ack/nacks for sequence numbers we don't expect
		D("Sender_process ACK_DATA asn=%d sn=%d acc=%d dl=%d\n",
		  ack_seq_no, sol_sender->seq_no, ack_accept_cnt,
		  sol_sender->pkt_data_len);
		if (ack_seq_no == sol_sender->seq_no) {
		    if (ack_accept_cnt == sol_sender->pkt_data_len){
			// Completion Ack
			pp_select_remove_to(sol_sender->retry_to_hndl);
			// check data -> ACKPENDING or ACCU
			sol_sender_resume(sol_sender->flush_pending);
		    } else { // Partial Ack, Resume Ack
			sol_sender_send_off_data_pkt();
			sol_sender->state = SOLS_STATE_ACKPENDING;
		    }
		} // else no state change
		break;
	    case SOLS_TRANS_NACK_DATA:
		// ignore unexpected nacks and "Partial Nacks"
		// (Partial Nacks are not defined as per spec ch. 15.11)
		if (ack_seq_no == sol_sender->seq_no && ack_accept_cnt == 0) {
		    // Suspend Nack
		    pp_select_remove_to(sol_sender->retry_to_hndl);
		    sol_sender->state = SOLS_STATE_SUSPENDED;
		} // else no state change
		break;
	    case SOLS_TRANS_RETRY_TIMEOUT: // no state change
		assert(sol_sender->state == SOLS_STATE_ACKPENDING);
		if (++sol_sender->retry_cnt <= sol_config->retry_cnt) {
		    sol_sender_append_pkt_data(); // appends newly arrived data
		    sol_sender_send_off_data_pkt();  // resends last packet
		} else {
		    pp_bmc_log_warn("[SOL] retries (%d) exceeded for pkt[%d]",
			    sol_config->retry_cnt, sol_sender->pkt->seq_no);
		    pp_select_remove_to(sol_sender->retry_to_hndl);
		    // check data -> SOLS_STATE_ACKPENDING or SOLS_STATE_ACCU
		    sol_sender_resume(sol_sender->flush_pending);
		}
		break;
	    default:
		sol_sender_sm_error(sol_sender->state, action);
	  }
	  break;
	  
      default:
	  sol_sender_sm_error(sol_sender->state, action);
    }
    D("Sender_process exit s=%s\n", send_state_str(sol_sender->state));
}
	  
static void sol_sender_sm_error(sol_sender_state_t s UNUSED,
				sol_sender_trans_t t UNUSED) {
    pp_bmc_log_debug("[SOL] SSM Error: state=%d transition=%d", s, t);
    assert(0);
}

static inline void sol_sender_set_accu_timeout() {
    if (sol_sender->accu_to_hndl == 0) {
	sol_sender->accu_to_hndl = pp_select_add_to(
		    SOL_ACCU_TO_2_MS(sol_config->char_accumulate), FALSE,
		    sol_sender_handle_accumulate_to, NULL);
    }
}

static inline void sol_sender_clear_accu_timeout() {
    if (sol_sender->accu_to_hndl > 0) {
	pp_select_remove_to(sol_sender->accu_to_hndl);
	sol_sender->accu_to_hndl = 0;
    }
}

static inline void sol_sender_set_retry_timeout() {
    sol_sender->retry_to_hndl = pp_select_add_to(
	SOL_RETRY_TO_2_MS(sol_config->retry_to), TRUE,
	sol_sender_handle_retry_to, NULL);
}

static inline int sol_sender_check_send_tresh(int flush_pending) {
    return flush_pending ||
	pp_rbuf_size(in_buf) >= sol_config->send_threshold;
}

static void sol_sender_resume(int flush_pending) {
    if (sol_sender_check_send_tresh(flush_pending) ||
	sol_sender->send_pending) {
	sol_sender_send_data_pkt();
	sol_sender->send_pending = 0;
	sol_sender->flush_pending = 0;
	sol_sender_clear_accu_timeout();
	sol_sender->state = SOLS_STATE_ACKPENDING;
    } else {
	sol_sender->state = SOLS_STATE_ACCU;
    }
}

static int sol_sender_handle_accumulate_to(const int item_id UNUSED,
					   void* ctx UNUSED) {
    sol_sender->accu_to_hndl = 0;
    sol_sender_process(SOLS_TRANS_ACCU_TIMEOUT, 0, 0, 0, 0);
    return PP_SUC;
}

static int sol_sender_handle_retry_to(const int item_id UNUSED,
				      void* ctx UNUSED) {
    sol_sender_process(SOLS_TRANS_RETRY_TIMEOUT, 0, 0, 0, 0);
    return PP_SUC;
}

static inline void sol_sender_send_data_pkt() {
    sol_sender_send_ackdata_pkt(FALSE, FALSE, 0, 0);
}

static void sol_sender_send_ackdata_pkt(int do_ack, int ack_is_nack,
					unsigned char ack_seq_no,
					unsigned char ack_accept_cnt) {
    sol_payload_bmc2rc_t* pkt = sol_sender->pkt;
#ifdef DEBUG
    size_t n =
#endif
	
    sol_sender_prepare_data(pkt);
    sol_sender_prepare_ctrl(pkt);

    if (do_ack) sol_sender_prepare_ack(pkt, ack_is_nack, ack_seq_no,
				       ack_accept_cnt);
    else sol_sender_prepare_ack_empty(pkt);
    
    sol_sender_set_retry_timeout();
    sol_sender->retry_cnt = 0;
    sol_sender->state = SOLS_STATE_ACKPENDING;

#ifdef DEBUG
    if (do_ack) {
	D("Sender_send_ackdata_pkt n=%zd an=%d aseq=%d aac=%d\n", 
	  n, ack_is_nack, ack_seq_no, ack_accept_cnt);
    } else {
	D("Sender_send_ackdata_pkt n=%zd\n", n);
    }
#endif
    sol_sender_send_off_data_pkt();
}

static void sol_sender_send_ack_only_pkt(int ack_is_nack,
					 unsigned char ack_seq_no,
					 unsigned char ack_accept_cnt) {
    sol_payload_bmc2rc_t* pkt = sol_sender->ackpkt;

    sol_sender_prepare_data_empty(pkt);
    sol_sender_prepare_ack(pkt, ack_is_nack, ack_seq_no, ack_accept_cnt);
    sol_sender_prepare_ctrl(pkt);

    D("Sender_send_ack_only_pkt an=%d aseq=%d aac=%d\n", 
      ack_is_nack, ack_seq_no, ack_accept_cnt);
    sol_sender_send_off_ack_only_pkt();
}

static size_t sol_sender_prepare_data(sol_payload_bmc2rc_t* pkt) {
    size_t n;
    int overflow;
    
    n = pp_rbuf_remove(in_buf, &overflow, pkt->data, SOL_MAX_DATA_LEN);
    sol_sender->seq_no = seq_no_inc(sol_sender->seq_no);
    sol_sender->pkt_data_len = (unsigned short)n;
    pkt->overflow = overflow;
    pkt->seq_no = sol_sender->seq_no;
    return n;
}

static inline size_t sol_sender_prepare_data_empty(sol_payload_bmc2rc_t* pkt) {
    pkt->seq_no = 0;
    return 0;
}
    
static void sol_sender_prepare_ack(sol_payload_bmc2rc_t* pkt, int ack_is_nack,
				   unsigned char ack_seq_no,
				   unsigned char ack_accept_cnt) {
    assert(ack_seq_no > 0);
    pkt->ack_seq_no = ack_seq_no;
    pkt->is_nack = ack_is_nack;
    pkt->accept_char_cnt = ack_accept_cnt;
}

static void sol_sender_prepare_ack_empty(sol_payload_bmc2rc_t* pkt) {
    pkt->ack_seq_no = 0;
    pkt->accept_char_cnt = 0;
    pkt->is_nack = 0;
}

static void sol_sender_prepare_ctrl(sol_payload_bmc2rc_t* pkt) {
    pkt->reserved0 = 0;
    pkt->reserved1 = 0;
    pkt->break_cond = break_cond; // BREAK from host detected
    break_cond = 0; // reset flag
    pkt->sol_deactive = 0; // TODO: needs to be set!
    pkt->sol_unavailable = 0; // TODO: needs to be set!
}

static void sol_sender_append_pkt_data() {
    size_t n;
    unsigned short clen = sol_sender->pkt_data_len;
    sol_payload_bmc2rc_t* pkt = sol_sender->pkt;

    if (!pp_rbuf_next_is_ovfl(in_buf)) {
	if ((n = pp_rbuf_remove(in_buf, NULL, &pkt->data[clen],
				SOL_MAX_DATA_LEN - clen)) > 0) {
	    sol_sender->pkt_data_len += n;
	    sol_sender->send_pending = 0;
	    sol_sender->flush_pending = 0;
	    sol_sender_clear_accu_timeout();
	}
	D("Sender_append_pkt_data n=%zd\n", n);
    }
}

static inline void sol_sender_send_off_data_pkt() {
    D("Sender_send_off_data_pkt seq=%d datalen=%d\n",
      sol_sender->pkt->seq_no, sol_sender->pkt_data_len);
    sol_sender_send_off_pkt(sol_sender->pkt, sol_sender->pkt_data_len);
}

static inline void sol_sender_send_off_ack_only_pkt() {
    D("Sender_send_off_ack_only_pkt seq=%d\n", sol_sender->ackpkt->seq_no);
    sol_sender_send_off_pkt(sol_sender->ackpkt, 0);
}

static inline void sol_sender_send_off_pkt(sol_payload_bmc2rc_t* pkt,
					   size_t pkt_data_len) {
    pp_ipmi20_send_payload((unsigned char*)pkt,
			   sizeof(sol_payload_bmc2rc_t) + pkt_data_len,
			   IPMI_PAYLOAD_SOL,
			   sol_sender->session,
			   sol_sender->session->addr);
}
	
/*
 * Receiver:
 * - stati
 *   - RECEIVING
 *   - SUSPENDING
 * - gets packet from LAN channel by len + data
 * - notifies sender of ack + seq#
 * - process packet
 *   - handle payload: give it to writer
 *     if writer's out-ringbuf is full, suspend SOL-client
 *   - handle control
 * ------------------------------------------
 */
static int sol_recver_handle_lan_input(unsigned char* buf, size_t len,
				       imsg_session_t* session UNUSED, 
				       lan_addr_t* addr UNUSED) {
    sol_payload_rc2bmc_t* pkt = (sol_payload_rc2bmc_t*)buf;
    size_t pkt_data_len = len - sizeof(sol_payload_rc2bmc_t);

    /* when we get the very first seqence number in this session     *
     * we need to adjust our stored last sequence number accordingly */
    if (sol_recver->seq_no == 0 && pkt->seq_no != 0) {
	sol_recver->seq_no = seq_no_dec(pkt->seq_no);
    }

    D("Recver_handle_lan_input seq=%d dlen=%zd an=%d aseq=%d acc=%d\n",
      pkt->seq_no, pkt_data_len, pkt->is_nack, pkt->ack_seq_no,
      pkt->accept_char_cnt);

    if (pkt->seq_no == 0) {	     // ack only pkt
	sol_recver_handle_acks(pkt);
	sol_recver_handle_ctrl(pkt);
    } else if (pkt->seq_no == sol_recver->seq_no) {
	switch (sol_recver->state) { // resend ack/nack
	  case SOL_RECV_RECEIVING:
	      sol_sender_ack_pkt(sol_recver->seq_no, sol_recver->accept_cnt);
	      break;
	  case SOL_RECV_SUSPENDING:
	      sol_sender_nack_pkt(sol_recver->seq_no, 0);
	      break;
	  default:
	      sol_recv_sm_error(sol_recver->state, pkt->seq_no);
	}
    } else if (pkt->seq_no == seq_no_inc(sol_recver->seq_no)) {
	switch (sol_recver->state) { // expected data pkt, handle and ack
	  case SOL_RECV_RECEIVING:
	      sol_recver_handle_acks(pkt);
	      sol_recver_handle_ctrl(pkt);
	      sol_recver->state = sol_recver_handle_data(pkt, pkt_data_len);
	      break;
	  case SOL_RECV_SUSPENDING:                         // ignore unwanted
	      pp_bmc_log_warn("[SOL] recveived data-pkt[%d] during SUSPEND",
			      pkt->seq_no);
	      break;
	  default:
	      sol_recv_sm_error(sol_recver->state, pkt->seq_no);
	}
    } else if (pkt->seq_no == seq_no_inc(seq_no_inc(sol_recver->seq_no))) {
	pp_bmc_log_warn("[SOL] Receiver: pkt[%d] lost, continue with [%d]",
			seq_no_inc(sol_recver->seq_no), pkt->seq_no);
	switch (sol_recver->state) { // expected data pkt, handle and ack
	  case SOL_RECV_RECEIVING:
	      sol_recver_handle_acks(pkt);
	      sol_recver_handle_ctrl(pkt);
	      sol_recver->state = sol_recver_handle_data(pkt, pkt_data_len);
	      break;
	  case SOL_RECV_SUSPENDING:                         // ignore unwanted
	      pp_bmc_log_warn("[SOL] recveived data-pkt[%d] during SUSPEND",
			      pkt->seq_no);
	      break;
	  default:
	      sol_recv_sm_error(sol_recver->state, pkt->seq_no);
	}
    } else {
	pp_bmc_log_warn("[SOL] Receiver: unexpected pkt[%d], expected [%d]",
			pkt->seq_no, seq_no_inc(sol_recver->seq_no));
	sol_recver->seq_no = pkt->seq_no;
    }
    return PP_SUC;
}

static void sol_recv_sm_error(sol_recver_state_t s UNUSED,
			      unsigned char seq_no UNUSED) {
    pp_bmc_log_debug("[SOL] RSM Error: state=%d pkt[%d]", s, seq_no);
    assert(0);
}

static int sol_recver_handle_data(sol_payload_rc2bmc_t* pkt,
				  size_t pkt_data_len) {
    size_t accept_cnt;
    sol_recver_state_t ns;

    if (pkt_data_len == 0) {
	pp_bmc_log_warn("[SOL] data pkt[%d] with no data received",
			pkt->seq_no);
    }
    
    accept_cnt = sol_writer_new_data(pkt->data, pkt_data_len);

    if (pkt_data_len == 0 || accept_cnt > 0) {
	if (accept_cnt == pkt_data_len) {
	    sol_sender_ack_pkt(pkt->seq_no, accept_cnt);   // complete ack
	} else {
	    sol_sender_nack_pkt(pkt->seq_no, accept_cnt);  // partial ack
	}
	sol_recver->seq_no = pkt->seq_no;
	sol_recver->accept_cnt = accept_cnt;
	ns = SOL_RECV_RECEIVING;
    } else { // accept_cnt == 0                            // need suspend
	sol_sender_nack_pkt(pkt->seq_no, accept_cnt);      // suspend nack
	ns = SOL_RECV_SUSPENDING;
	sol_recver->buf_avail_hndl = pp_select_add_fd(sol_tty_fd, POLLOUT,
						      sol_recver_buf_avail_cb,
						      NULL);
    }
    return ns;
}

static int sol_recver_buf_avail_cb(const int item_id UNUSED,
				   const int fd UNUSED, 
				   const short event UNUSED,
				   void* ctx UNUSED) {
    sol_sender_ack_pkt(seq_no_inc(sol_recver->seq_no), 0); // resume nack
    sol_recver->state = SOL_RECV_RECEIVING;
    // remove POLLOUT handle, real write will happen, once next packet arrives
    pp_select_remove_fd(sol_recver->buf_avail_hndl);
    return PP_SUC;
}

static void sol_recver_handle_ctrl(sol_payload_rc2bmc_t* pkt) {
    // ignore WOR
    // ignore CTS
    // ignore DCD/DSR
    
    if (pkt->flush_inbound) {  // clear (discard) out_buf
	tcflush(sol_tty_fd, TCOFLUSH);
    }
    if (pkt->flush_outbound) { // force in_buf to network
	if (pp_rbuf_size(in_buf) > 0)
	    sol_sender_new_data(TRUE); // TRUE means 'do flush'
    }
    if (pkt->gen_break) {
        tcsendbreak(sol_tty_fd, 0); // 0 means duration of 250-500ms
        // (should roughly fulfill the requirements of IPMI spec (300ms))
    }
}
		       
static void sol_recver_handle_acks(sol_payload_rc2bmc_t* pkt) {
    if (pkt->ack_seq_no != 0) { 
	if (pkt->is_nack) {
	    sol_sender_pkt_nacked(pkt->ack_seq_no, pkt->accept_char_cnt);
	} else {
	    sol_sender_pkt_acked(pkt->ack_seq_no, pkt->accept_char_cnt);
	}
    } // else: control only packet, no action
}

/*
 * Writer:
 * - get data from receiver
 * - writes data to seriell fd, non-blocking
 * - if write would block response to receiver, so it can suspend
 * ---------------------------------------------------------------
 */
static size_t sol_writer_new_data(char* buf, size_t len) {
    ssize_t n, sent = 0, rmdr = len;

    while (rmdr > 0) {
	if (0 > (n = write(sol_tty_fd, &buf[sent], rmdr))) {
	    if (errno == EINTR) {
		continue;
	    } else if (errno == EAGAIN) {
		break;
	    } else {
		pp_bmc_log_perror("[SOL] write(fd=%d)", sol_tty_fd);
		break;
	    }
	}
	rmdr -= n;
	sent += n;
    }
    return sent;
}

#ifdef DEBUG
static const char* send_state_str(sol_sender_state_t s) {
    switch (s) {
    case SOLS_STATE_ACCU: return "SOLS_STATE_ACCU";
    case SOLS_STATE_ACKPENDING: return "SOLS_STATE_ACKPENDING";
    case  SOLS_STATE_SUSPENDED: return "SOLS_STATE_SUSPENDED";
    default: return "INVALID!!!";
    }
}

static const char* send_action_str(sol_sender_trans_t t) {
    switch(t) {
    case SOLS_TRANS_SEND_DATA: return "SOLS_TRANS_SEND_DATA";
    case SOLS_TRANS_SEND_ACK: return "SOLS_TRANS_SEND_ACK";
    case SOLS_TRANS_ACK_DATA: return "SOLS_TRANS_ACK_DATA";
    case SOLS_TRANS_NACK_DATA: return "SOLS_TRANS_NACK_DATA";
    case SOLS_TRANS_RETRY_TIMEOUT: return "SOLS_TRANS_RETRY_TIMEOUT";
    case SOLS_TRANS_ACCU_TIMEOUT: return "SOLS_TRANS_ACCU_TIMEOUT";
    default: return "INVALID!!!";
    }
}
#endif
