/**
 * lan_sol_cmd.c
 *
 * commands dedicated to SOL (spec 2.0 chapter 26)
 * 
 * (c) 2005 Peppercon AG, 2005/05/08, tbr@peppecon.de
 * 
 * FIXME need to register handler to react on cfg changes while activated
 */
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/xdefs.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/utils.h>
#include <pp/bmc/topo_classes.h>

#include <pp/bmc/lan_serv.h>
#include "lan_sol_cmds.h"
#include "lan_sol.h"

const char *bmc_lan_sol_min_priv_level_str = "bmc.sol.min_priv_level";

/*
 * private definitions
 * ----------------------------------
 */

struct sol_set_config_rq_s {
    BITFIELD2(unsigned char,
	      channel_no : 4,
	      rsvd : 4);
    unsigned char param;
    unsigned char data[0];
} __attribute__ ((packed));
typedef struct sol_set_config_rq_s sol_set_config_rq_t;

/* sol_set_config_rs has no params, just completion code */

struct sol_get_config_rq_s {
    BITFIELD3(unsigned char,
	      channel_no : 4,
	      rsvd : 3,
	      param_rev_only : 1);
    unsigned char param;
    unsigned char selector;
    unsigned char block_selector;
} __attribute__ ((packed));
typedef struct sol_get_config_rq_s sol_get_config_rq_t;

struct sol_get_config_rs_s {
    unsigned char param_rev;
    unsigned char data[0];
} __attribute__ ((packed));
typedef struct sol_get_config_rs_s sol_get_config_rs_t;

/* SOL cfg parameters */
#define SOL_CFG_SET_IN_PROGRESS   0
#define SOL_CFG_ENABLE            1
#define SOL_CFG_AUTHENTICATION    2
#define SOL_CFG_ACCU_TRESH        3
#define SOL_CFG_RETRY             4
#define SOL_CFG_NV_BITRATE        5
#define SOL_CFG_BITRATE           6
#define SOL_CFG_PAYLOAD_CHAN      7
#define SOL_CFG_PORT_NUMBER       8
#define SOL_CFG_MAX_PARAM_NO      8


/* SOL_CFG_SET_IN_PROGRESS : 1 byte */
/* SOL_CFG_ENABLE          : 1 byte */

/* SOL_CFG_AUTHENTICATION  */
struct sol_cfg_auth_s {
    BITFIELD4(unsigned char,
	      priv_level : 4,
	      rsvd : 2,
	      force_auth : 1,
	      force_encrypt : 1
	      );
} __attribute__ ((packed));
typedef struct sol_cfg_auth_s sol_cfg_auth_t;			  

/* SOL_CFG_ACCU_TRESH      */
struct sol_cfg_accu_thresh_s {
    unsigned char accu_interval;
    unsigned char send_threshold;
} __attribute__ ((packed));
typedef struct sol_cfg_accu_thresh_s sol_cfg_accu_thresh_t;

/* SOL_CFG_RETRY           */
struct sol_cfg_retry_s {
    unsigned char retry_cnt;      // only bits [2:0]
    unsigned char retry_interval;
} __attribute__ ((packed));
typedef struct sol_cfg_retry_s sol_cfg_retry_t;

/* SOL_CFG_NV_BITRATE      : 1 byte */
/* SOL_CFG_BITRATE         : 1 byte */
/* SOL_CFG_PAYLOAD_CHAN    : 1 byte */
/* SOL_CFG_PORT_NUMBER     : 1 byte */


/* 
 * private methods 
 * -----------------
 */
static int sol_set_config(imsg_t* imsg);
static int sol_get_config(imsg_t* imsg);
static int sol_tty_speed_ipmi2cfg(unsigned char ipmispeed);
static unsigned char sol_tty_speed_cfg2ipmi(int cfgspeed);

static unsigned char set_in_progress_state = 0;

/*
 * sol channel cmds
 * ------------------  
 */
static const dev_cmd_entry_t sol_cmd_tab[] = {
    /* TODO: this is actually send from the BMC, maybe not need here
    { netfn:          IPMI_NETFN_TRANSPORT,
      cmd:            IPMI_CMD_SOL_ACTIVATING,
      cmd_hndlr:      ,
      min_data_size:  ,
      min_priv_level: ,
    },
     */
    { netfn:          IPMI_NETFN_TRANSPORT,
      cmd:            IPMI_CMD_SOL_SET_SOL_CFG,
      cmd_hndlr:      sol_set_config,
      min_data_size:  sizeof(sol_set_config_rq_t),
      min_priv_level: IPMI_PRIV_ADMIN,
    },
    { netfn:          IPMI_NETFN_TRANSPORT,
      cmd:            IPMI_CMD_SOL_GET_SOL_CFG,
      cmd_hndlr:      sol_get_config,
      min_data_size:  sizeof(sol_get_config_rq_t),
      min_priv_level: IPMI_PRIV_USER,
    },
    { cmd_hndlr:      NULL,
    }
};

#define GET_CFG(c) \
    if (PP_FAILED(c)) { \
        pp_bmc_log_error("[SOL-CFG] can't read param #%d from cfg", rq->param); \
        ierr = IPMI_ERR_UNSPECIFIED; \
        break; \
    }

/*
 * implementations
 * ----------------
 */
static int sol_get_config(imsg_t* imsg) {
    sol_get_config_rq_t* rq = (void*)imsg->data;
    sol_get_config_rs_t* rs;
    unsigned char rs_data[256];
    unsigned char rs_dsz = 1;
    int ierr = IPMI_ERR_SUCCESS;

    // FIXME: rq->channel_no ,ignored currently

    if (rq->param_rev_only) {
	// return param rev only
	rs_dsz = 0;
    } else {
	pp_bmc_log_info("[SOL-CFG] param GET #%d", rq->param);
	switch (rq->param) {
	  case SOL_CFG_SET_IN_PROGRESS:
	      rs_data[0] = set_in_progress_state;
	      break;
	  case SOL_CFG_ENABLE: {
	      int enabled;
	      GET_CFG(pp_cfg_is_enabled(&enabled, PP_BMC_SOL_CFG_IS_ENABLED));
	      rs_data[0] = enabled; }
	      break;
	  case SOL_CFG_AUTHENTICATION: {
	      char *val;
	      int priv_level;
	      rs_data[0] = 0;
	      
              GET_CFG(pp_cfg_get(&val, bmc_lan_sol_min_priv_level_str));
              priv_level = pp_bmc_get_privlevel_value(val);
              if (priv_level < IPMI_PRIV_USER) {
		  pp_bmc_log_debug("[SOL-CFG] reading minimum privilege level <%s> from configuration. Ignoring and using \"USER\" instead.", val);
		  priv_level =  IPMI_PRIV_USER;
	      }
             
	      ((sol_cfg_auth_t*)&rs_data[0])->force_auth = 1;
	      ((sol_cfg_auth_t*)&rs_data[0])->force_encrypt = 1;
	      ((sol_cfg_auth_t*)&rs_data[0])->priv_level = priv_level; 
	      free(val); }
	      break;
	  case SOL_CFG_ACCU_TRESH: {
	      sol_cfg_accu_thresh_t* rsv = (sol_cfg_accu_thresh_t*)rs_data;
	      int accu, thresh;
	      rs_dsz = sizeof(sol_cfg_accu_thresh_t);
	      GET_CFG(pp_cfg_get_int(&accu, PP_BMC_SOL_CFG_ACCU_INTERVAL));
	      rsv->accu_interval = accu;
	      GET_CFG(pp_cfg_get_int(&thresh, PP_BMC_SOL_CFG_SEND_TRESHOLD));
	      rsv->send_threshold = thresh; }
	      break;
	  case SOL_CFG_RETRY: {
	      sol_cfg_retry_t* rsv = (sol_cfg_retry_t*)rs_data;
	      int retry_cnt, retry_interval;
	      rs_dsz = sizeof(sol_cfg_retry_t);
	      GET_CFG(pp_cfg_get_int(&retry_cnt, PP_BMC_SOL_CFG_RETRY_CNT));
	      rsv->retry_cnt = retry_cnt;
	      GET_CFG(pp_cfg_get_int(&retry_interval, PP_BMC_SOL_CFG_RETRY_INTERVAL));
	      rsv->retry_interval = retry_interval; }
	      break;
	  case SOL_CFG_NV_BITRATE: {
	      int bitrate;
	      GET_CFG(pp_cfg_get_int(&bitrate, PP_BMC_SOL_CFG_TTY_SPEED));
	      rs_data[0] = sol_tty_speed_cfg2ipmi(bitrate); }
	      break;
	  case SOL_CFG_BITRATE:
	      rs_data[0] = sol_tty_speed_cfg2ipmi(pp_bmc_sol_get_tty_speed());
	      break;
	  case SOL_CFG_PAYLOAD_CHAN:
	      rs_data[0] = rq->channel_no; // mirror request channel back
	      break;
	  case SOL_CFG_PORT_NUMBER: {
	      rs_dsz = 2;
	      unsigned short port;
	      if (PP_FAILED(pp_cfg_get_short_nodflt(&port, "bmc.lan.port"))) {
	          port = IPMI_LAN_PORT;
	      }
	      *(short*)rs_data = cpu_to_le16(port);
	      break; }
	  default:
	      ierr = IPMI_ERR_SOL_CFG_PARAM_NOT_SUPPORTED;
	}

    }

    if (ierr != IPMI_ERR_SUCCESS) {
        return pp_bmc_router_resp_err(imsg, ierr);
    }

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
                            sizeof(sol_get_config_rs_t) + rs_dsz);
    rs->param_rev = 0x11;
    memcpy(rs->data, rs_data, rs_dsz);

    return pp_bmc_router_send_msg(imsg);
}

#define CHECK_PDATA_SIZE(s) \
    if (psize < s) { ierr = IPMI_ERR_REQ_DATA_LEN_INVALID; break; }

#define SET_CFG(c) \
    if (PP_FAILED(c)) { \
        pp_bmc_log_error("[SOL-CFG] can't write param #%d to cfg", rq->param); \
        ierr = IPMI_ERR_UNSPECIFIED; \
        break; \
    }

static int sol_set_config(imsg_t* imsg) {
    sol_set_config_rq_t* rq = (void*)imsg->data;
    // FIXME: rq->channel_no ,ignored currently
    int psize = imsg->data_size - sizeof(sol_set_config_rq_t);
    unsigned char* pdata = rq->data;

    unsigned char ierr = IPMI_ERR_SUCCESS;
    int dirty = 0;
    pp_cfg_tx_t* cfg_trans;

    cfg_trans = pp_cfg_tx_begin(1);
    if (!cfg_trans) {
        pp_bmc_log_error("[SOL-CFG] can't create cfg transaction");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }

    pp_bmc_log_info("[SOL-CFG] param SET #%d", rq->param);
    switch (rq->param) {
        case SOL_CFG_SET_IN_PROGRESS:
            CHECK_PDATA_SIZE(1);
            switch (pdata[0] & 0x3) {
                case 0: /* set complete */
                    set_in_progress_state = 0;
                    break;
                case 1: /* set in progress */
                    /* no error if already in progress here (see spec) */
                    set_in_progress_state = 1;
                    break;
                default:
                    ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
            }
            break;
        case SOL_CFG_AUTHENTICATION: {
            sol_cfg_auth_t* val = (sol_cfg_auth_t*)pdata; 
            CHECK_PDATA_SIZE((unsigned char)sizeof(sol_cfg_auth_t));
            if  ( !val->force_auth || !val->force_encrypt) {
                pp_bmc_log_warn("[SOL-CFG] param #2 (SOL Authentication): Force bits not set");
                // Ignore these bits but accept the
                // minimum privilege level
            }

            if (val->priv_level < IPMI_PRIV_USER || val->priv_level > IPMI_PRIV_OEM) {
                pp_bmc_log_warn("[SOL-CFG] param #%d (SOL Authentication): Unsupported privilege level %d",
                                 rq->param, val->priv_level);
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
                break;
            }
            SET_CFG(pp_cfg_set(pp_bmc_get_privlevel_string(val->priv_level), bmc_lan_sol_min_priv_level_str));
            dirty = 1;
            break; }
        case SOL_CFG_PORT_NUMBER:
        case SOL_CFG_PAYLOAD_CHAN:
            ierr = IPMI_ERR_SOL_CFG_READ_ONLY;
            break;
        case SOL_CFG_ENABLE:
            CHECK_PDATA_SIZE(1);
            pdata[0] &= 0x01;
            SET_CFG(pp_cfg_set_enabled(pdata[0], PP_BMC_SOL_CFG_IS_ENABLED));
            dirty = 1;
            break;
        case SOL_CFG_ACCU_TRESH: {
            sol_cfg_accu_thresh_t* val = (sol_cfg_accu_thresh_t*)pdata;
            CHECK_PDATA_SIZE((unsigned char)sizeof(sol_cfg_accu_thresh_t));
            if (val->accu_interval > 0) {
                SET_CFG(pp_cfg_set_int(val->accu_interval, PP_BMC_SOL_CFG_ACCU_INTERVAL));
                dirty = 1;
            }
            if (val->send_threshold == 0) {
                SET_CFG(pp_cfg_set_int(val->send_threshold, PP_BMC_SOL_CFG_SEND_TRESHOLD));
                dirty = 1;
            }
            break; }
        case SOL_CFG_RETRY: {
            sol_cfg_retry_t* val = (sol_cfg_retry_t*)pdata;
            CHECK_PDATA_SIZE((unsigned char)sizeof(sol_cfg_retry_t));
            SET_CFG(pp_cfg_set_int(val->retry_cnt & 0x7, PP_BMC_SOL_CFG_RETRY_CNT));
            SET_CFG(pp_cfg_set_int(val->retry_interval, PP_BMC_SOL_CFG_RETRY_INTERVAL));
            dirty = 1;
            break; }
        case SOL_CFG_NV_BITRATE:
            CHECK_PDATA_SIZE(1);
            pdata[0] &= 0x0f;
            if (sol_tty_speed_ipmi2cfg(pdata[0]) == 0) {
                pp_bmc_log_warn("[SOL-CFG] param #%d (NV Bitrate): Unsupported bitrate %d",
                                 rq->param, pdata[0]);
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
                break;
            }
            SET_CFG(pp_cfg_set_int(sol_tty_speed_ipmi2cfg(pdata[0]), PP_BMC_SOL_CFG_TTY_SPEED));
            dirty = 1;
            break;
        case SOL_CFG_BITRATE:
            CHECK_PDATA_SIZE(1);
            pdata[0] &= 0x0f;
            if (sol_tty_speed_ipmi2cfg(pdata[0]) == 0) {
                pp_bmc_log_warn("[SOL-CFG] param #%d (Bitrate): Unsupported bitrate %d",
                                 rq->param, pdata[0]);
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
            } else if (PP_FAILED(pp_bmc_sol_set_tty_speed(sol_tty_speed_ipmi2cfg(pdata[0])))) {
                ierr = IPMI_ERR_UNSPECIFIED;
            }
            break;
        default:
            ierr = IPMI_ERR_SOL_CFG_PARAM_NOT_SUPPORTED;
    }

    if (dirty && ierr == IPMI_ERR_SUCCESS) {
        if (PP_FAILED(pp_cfg_tx_commit(cfg_trans))) {
            pp_bmc_log_error("[SOL-CFG] can't commit cfg transaction");
            ierr = IPMI_ERR_UNSPECIFIED;
        }
        pp_cfg_save(DO_FLUSH);
    } else {
        pp_cfg_tx_rollback(cfg_trans);
    }

    return pp_bmc_router_resp_err(imsg, ierr);
}

/* c'tor / d'tor */
int pp_sol_cmd_init() {
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(sol_cmd_tab))
	return PP_ERR;
    return PP_SUC;
}

void pp_sol_cmd_cleanup() {
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(sol_cmd_tab);
}

/*
 * supporting methods
 * -------------------
 */
static int sol_tty_speed_ipmi2cfg(unsigned char ipmispeed) {
    int cfgspeed;
    switch (ipmispeed) {
      case 0x6: cfgspeed = 9600; break;
      case 0x7: cfgspeed = 19200; break;
      case 0x8: cfgspeed = 38400; break;
      case 0x9: cfgspeed = 57600; break;
      case 0xA: cfgspeed = 115200; break;
      default: cfgspeed = 0;
    }
    return cfgspeed;
}

static unsigned char sol_tty_speed_cfg2ipmi(int cfgspeed) {
    unsigned char ipmispeed;
    switch (cfgspeed) {
      case 9600: ipmispeed = 0x6; break;
      case 19200: ipmispeed = 0x7; break;
      case 38400: ipmispeed = 0x8; break;
      case 57600: ipmispeed = 0x9; break;
      case 115200: ipmispeed = 0xA; break;
      default: ipmispeed = 0;
    }
    return ipmispeed;
}
