/**
 * scsi_ipmi.c
 *
 * (c) 2006 Raritan Computer Inc.
 * Ingo van Lil <ingo.lil@raritan.com>
 *
 * Proprietary IPMI-over-SCSI channel adapter
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <linux/ioctl.h>
#include <sys/ioctl.h>
#include <termios.h>

#include <pp/base.h>
#include <pp/ipc.h>
#include <pp/termios.h>
#include <lara.h>

#include <pp/bmc/scsi_ipmi.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/bmc.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/utils.h>
#include <pp/bmc/session_manager.h>
#include <pp/bmc/debug.h>

static pp_bmc_router_chan_adapter_config_t *scsi_channel_config = NULL;

static const pp_bmc_router_chan_adapter_info_t scsi_chan_info = {
    .medium    = IPMI_CHAN_MED_SCSI,
    .proto     = IPMI_CHAN_PROT_SCSI,
    .sess_supp = IPMI_CHAN_SESS_SINGLE
};

struct scsi_imsg_s {
    uint32_t session_id_le32;
    uint32_t session_secret; // random data, no endianness issues
    uint8_t rs_addr;
    uint8_t rs_lun;
    uint8_t rq_addr;
    uint8_t rq_lun;
    uint8_t netfn;
    uint8_t cmd;
    uint8_t data[0];
} __attribute__((packed));
typedef struct scsi_imsg_s scsi_imsg_t;

static int handle_msg_callback(void *ctx)
{
    return pp_bmc_core_handle_msg(ctx);
}

static void handle_ipc_message(pp_ipc_req_ctx_t *ctx,
	pp_ipc_req_type_t type UNUSED, int length, unsigned char *data)
{
    scsi_imsg_t *scsi_imsg = (scsi_imsg_t *)data;
    int data_len = length - sizeof(scsi_imsg_t);
    imsg_t *imsg;

    if (data_len < 0) {
	pp_bmc_log_debug("[SCSI] short imsg received.");
	return;
    }

    imsg = pp_bmc_imsg_new(data_len);
    imsg->chan = IPMI_CHAN_SCSI;
    imsg->rs_addr = scsi_imsg->rs_addr;
    imsg->rs_lun = scsi_imsg->rs_lun;
    imsg->rq_addr = scsi_imsg->rq_addr;
    imsg->rq_lun = scsi_imsg->rq_lun;
    imsg->netfn = scsi_imsg->netfn;
    imsg->cmd = scsi_imsg->cmd;
    imsg->session_secret = scsi_imsg->session_secret;
    if (scsi_imsg->session_id_le32) {
	uint32_t session_id = le32_to_cpu(scsi_imsg->session_id_le32);
	imsg->session = pp_bmc_get_session(session_id);
	if (!imsg->session) {
	    pp_bmc_log_debug("[SCSI] Invalid session ID, message ignnored.");
	    goto bail;
	} else if (imsg->session_secret != imsg->session->session_secret) {
	    printf("imsg secret = %d, session secret = %d\n", imsg->session_secret, imsg->session->session_secret);
	    pp_bmc_log_debug("[SCSI] Invalid session secret, message ignnored.");
	    goto bail;
	}
	imsg->priv_level = imsg->session->cur_priv_level;
    } else {
	imsg->priv_level = IPMI_PRIV_UNSPEC;
    }
    memcpy(imsg->data, scsi_imsg->data, data_len);

    // addressing information; automatically freed by pp_bmc_imsg_delete()
    imsg->buf = (unsigned char *)pp_ipc_ctx_dup(ctx);

    pp_bmc_run_fkt(handle_msg_callback, imsg);
    return;

bail:
    pp_bmc_imsg_delete(imsg);
}

static int scsi_send_imsg(imsg_t *imsg)
{
    pp_ipc_req_ctx_t *ctx = (pp_ipc_req_ctx_t *)imsg->buf;
    scsi_imsg_t *scsi_imsg;

    scsi_imsg = malloc(sizeof(scsi_imsg_t) + imsg->data_size);
    scsi_imsg->rs_addr = imsg->rs_addr;
    scsi_imsg->rs_lun = imsg->rs_lun;
    scsi_imsg->rq_addr = imsg->rq_addr;
    scsi_imsg->rq_lun = imsg->rq_lun;
    scsi_imsg->netfn = imsg->netfn;
    scsi_imsg->cmd = imsg->cmd;
    scsi_imsg->session_secret = imsg->session_secret;
    memcpy(scsi_imsg->data, imsg->data, imsg->data_size);

    pp_ipc_send_response(ctx, PP_IPC_RSP_SUCCESS,
	    sizeof(scsi_imsg_t) + imsg->data_size, (unsigned char *)scsi_imsg);

    free(scsi_imsg);
    pp_bmc_imsg_delete(imsg);
    return PP_SUC;
}

int pp_bmc_scsi_ipmi_init(void)
{
    assert(scsi_channel_config == NULL);

    // register the channel
    scsi_channel_config = pp_bmc_router_reg_chan(
                                IPMI_CHAN_SCSI,
                                scsi_send_imsg,
                                NULL,
                                NULL,
                                &scsi_chan_info);
    if (scsi_channel_config == NULL) {
	return PP_ERR;
    }

    if (PP_FAILED( pp_ipc_register_handler(PP_IPC_REQ_IPMI, handle_ipc_message) )) {
	return PP_ERR;
    }

    pp_bmc_log_info("[SCSI] channel initialized");
    return PP_SUC;
}

void pp_bmc_scsi_ipmi_cleanup(void)
{
    pp_ipc_unregister_handler(PP_IPC_REQ_IPMI);
    pp_bmc_router_unreg_chan(IPMI_CHAN_SCSI);
    pp_bmc_log_info("[SCSI] channel shut down");
}

