/**
 * adm1024.c
 *
 * Implements the ADM1024 Monitor / Fan controller chip
 * Includes the i2c chip object as well as the sensor object
 * 
 * (c) 2005 Peppercon AG, 2005/11/22, thomas@peppercon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_chip.h>
#include <pp/bmc/tp_scan_sensdev.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/tp_gpio_dev.h>

#include "drivers/adm1024.h"

/*
 * The chip, may contain global initialization code
 ****************************************************/

typedef struct pp_tp_i2c_chip_t {
    pp_tp_i2c_chip_t base;
    unsigned char fan1_ticks_revol;
    unsigned char fan2_ticks_revol;
} adm1024_chip_t;

static void adm1024_dtor(pp_tp_obj_t* o);
static void adm1024_power_up(pp_tp_i2c_chip_t* this);
static inline int adm1024_fan_ticks_valid(unsigned char ticks);
static unsigned char adm1024_fan_ticks_to_cfg(unsigned char ticks);
static unsigned int adm1024_fan_cfg_to_prescale(unsigned char cfg);

pp_tp_obj_t* pp_sensor_adm1024_ctor(const char* id, vector_t* args) {
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    adm1024_chip_t* o = NULL;
    pp_tp_i2c_comdev_t* i2cdev;
    int i2caddr;
    pp_tp_cond_t* icond;
    unsigned char fan1_ticks_revol;
    unsigned char fan2_ticks_revol;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>do<sc>d<c>d<c>",
			&i2cdev, &i2caddr, &icond,
			&fan1_ticks_revol, &fan2_ticks_revol) != 5) {
	pp_bmc_log_perror("%s(): '%s' failed: %s", ___F, id,
			  pp_strstream_buf(&err));
    } else if (!adm1024_fan_ticks_valid(fan1_ticks_revol) ||
	       !adm1024_fan_ticks_valid(fan2_ticks_revol)) {
	errno = EINVAL;
	pp_bmc_log_perror("%s(): '%s' failed: ticks can be 1, 2 or 4",
			  ___F, id);
    } else {
	o = malloc(sizeof(adm1024_chip_t));
	pp_tp_i2c_chip_init(&o->base, PP_TP_I2C_CHIP, id, adm1024_dtor,
			    ADM1024_MODEL_STRING, i2cdev, i2caddr, icond,
			    adm1024_power_up, NULL);
	o->fan1_ticks_revol = fan1_ticks_revol;
	o->fan2_ticks_revol = fan2_ticks_revol;
    }
    
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)o;
}

static void adm1024_dtor(pp_tp_obj_t* o) {
    pp_tp_i2c_chip_t* this = (pp_tp_i2c_chip_t*)o;
    assert(this != NULL);
    pp_tp_i2c_chip_cleanup(this);
    free(this);
}

static void adm1024_power_up(pp_tp_i2c_chip_t* chip) {
    adm1024_chip_t* this = (adm1024_chip_t*)chip;
    unsigned char r;
    
    if (PP_ERR == pp_tp_i2c_chip_pre_com(chip)) return;
    
    /* setup tachos, configure fan devisor */
    r = (adm1024_fan_ticks_to_cfg(this->fan1_ticks_revol) << 4) 
	| (adm1024_fan_ticks_to_cfg(this->fan2_ticks_revol) << 6);
    if (PP_ERR == pp_tp_i2c_chip_tx_byte_data(chip, ADM1024_REG_FAN_DEVISOR,
					      r)) {
	goto finally;
    }

    /* enable scanning */
    if (PP_ERR == pp_tp_i2c_chip_tx_byte_data(chip, ADM1024_REG_CONFIG_REG_1,
					      0x80 /* default config */) ||
	PP_ERR == pp_tp_i2c_chip_tx_byte_data(chip, ADM1024_REG_CONFIG_REG_1,
					      0x01 /* start */)) {
	goto finally;
    }
 finally:
    pp_tp_i2c_chip_post_com(chip);
}

static inline int adm1024_fan_ticks_valid(unsigned char ticks) {
    return (ticks == 1 || ticks == 2 || ticks == 4);
}

static unsigned char adm1024_fan_ticks_to_cfg(unsigned char ticks) {
    unsigned char ret;
    switch(ticks) {
      default:	assert(0);
      case 1: ret = 0x03; break; // prescale 8
      case 2: ret = 0x02; break; // prescale 4
      case 4: ret = 0x01; break; // prescale 2
    }
    return ret;
}

static unsigned int adm1024_fan_cfg_to_prescale(unsigned char cfg) {
    unsigned int ret;
    switch(cfg) {
      default: assert(0);
      case 0x00: ret = 1; break;
      case 0x01: ret = 2; break;
      case 0x02: ret = 4; break;
      case 0x03: ret = 8; break;
    }
    return ret;
}

/*
 * sensors
 ************************/
typedef struct adm1024_sens_type_cfg_s {
    unsigned char sens_type;
    unsigned char reading_type;
    unsigned short analog_type;
    unsigned short unit_type;
    char sdr_rexp;
    void (*update)(pp_sensor_scannable_t*);
} adm1024_sens_type_cfg_t;

typedef struct adm1024_sens_cfg_s {
    unsigned char reg;
    unsigned char fan_prescale_shift;
    adm1024_sens_type_cfg_t* type;
    unsigned short sdr_mtol;
} adm1024_sens_cfg_t;

typedef struct adm1024_sens_s {
    pp_tp_scan_sensdev_t base;
    pp_tp_i2c_chip_t* chip;
    adm1024_sens_cfg_t* cfg;
} adm1024_sens_t;

static void adm1024_sens_destroy(pp_tp_obj_t* o);
static ipmi_sdr_header_t* adm1024_sens_default_sdr(pp_tp_sensdev_t* s);
static void adm1024_update_temp_reading(pp_sensor_scannable_t*);
static void amd1024_update_tach_reading(pp_sensor_scannable_t*);
static void amd1024_update_voltage_reading(pp_sensor_scannable_t*);

adm1024_sens_type_cfg_t adm1024_sens_type_cfg_temp = {
    .sens_type    = IPMI_SENSOR_TYPE_TEMPERATURE,
    .reading_type = IPMI_EVENT_READING_TYPE_THRESHOLD,
    .analog_type  = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
    .unit_type    = IPMI_UNIT_TYPE_DEGREES_C,
    .sdr_rexp     = 0,
    .update       = adm1024_update_temp_reading,
};

adm1024_sens_type_cfg_t adm1024_sens_type_cfg_tacho = {
    .sens_type    = IPMI_SENSOR_TYPE_FAN,
    .reading_type = IPMI_EVENT_READING_TYPE_THRESHOLD,
    .analog_type  = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
    .unit_type    = IPMI_UNIT_TYPE_RPM,
    .sdr_rexp     = 0,
    .update       = amd1024_update_tach_reading,
};

adm1024_sens_type_cfg_t adm1024_sens_type_cfg_voltage = {
    .sens_type    = IPMI_SENSOR_TYPE_VOLTAGE,
    .reading_type = IPMI_EVENT_READING_TYPE_THRESHOLD,
    .analog_type  = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
    .unit_type    = IPMI_UNIT_TYPE_VOLTS,
    .sdr_rexp     = -3,
    .update       = amd1024_update_voltage_reading,
};

adm1024_sens_cfg_t adm1024_sens_cfg[9] = {
    { .reg = ADM1024_REG_VALUE_FAN1_AIN1,
      .fan_prescale_shift = 4,
      .type = &adm1024_sens_type_cfg_tacho,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(70, 1) },
    { .reg = ADM1024_REG_VALUE_FAN2_AIN2,
      .fan_prescale_shift = 6,
      .type = &adm1024_sens_type_cfg_tacho,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(70, 1) },
    { .reg = ADM1024_REG_VALUE_VCCP1,
      .type = &adm1024_sens_type_cfg_voltage,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(14, 1) },
    { .reg = ADM1024_REG_VALUE_VCCP2,
      .type = &adm1024_sens_type_cfg_voltage,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(14, 1) },
    { .reg = ADM1024_REG_VALUE_25V,
      .type = &adm1024_sens_type_cfg_voltage,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(13, 1) },
    { .reg = ADM1024_REG_VALUE_5V,
      .type = &adm1024_sens_type_cfg_voltage,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 1) },
    { .reg = ADM1024_REG_VALUE_12V,
      .type = &adm1024_sens_type_cfg_voltage,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(62, 1) },
    { .reg = ADM1024_REG_VALUE_INTERNAL_TEMP,
      .type = &adm1024_sens_type_cfg_temp,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0) },
    { .reg = ADM1024_REG_VALUE_EXT_TEMP1,
      .type = &adm1024_sens_type_cfg_temp,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0) },
};	
      
pp_tp_obj_t* pp_sensor_adm1024_sens_ctor(const char* id, vector_t* args) {
    const char* errmsg = "%s(): '%s' failed: %s";
    adm1024_sens_t* this = NULL;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_i2c_chip_t* chip;
    unsigned int sens_no;

    if (pp_tp_arg_scanf(args, 0, &err, "o<h>d", &chip, &sens_no) != 2) {
	pp_bmc_log_perror(errmsg, ___F, id, pp_strstream_buf(&err));
    } else if (strcmp(chip->model, ADM1024_MODEL_STRING)) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "incorrect chip model");
    } else if (sens_no > 8) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "invalid sensor number");
    } else {
	adm1024_sens_cfg_t* cfg = &adm1024_sens_cfg[sens_no];
	this = malloc(sizeof(adm1024_sens_t));
	pp_tp_scan_sensdev_init(&this->base, PP_TP_SCAN_SENS_DEV, id,
				(void(*)(pp_tp_obj_t*))adm1024_sens_destroy,
				adm1024_sens_default_sdr, cfg->type->update,
				chip->init_cond);
	this->chip = pp_tp_i2c_chip_duplicate(chip);
	this->cfg = cfg;

	/* register for scanning  */
	pp_tp_scan_sensdev_register(&this->base, 0);
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void adm1024_sens_destroy(pp_tp_obj_t* o) {
    adm1024_sens_t* this = (adm1024_sens_t*)o;
    assert(this != NULL);
    pp_tp_scan_sensdev_unregister(&this->base);
    pp_tp_i2c_chip_release(this->chip);
    pp_tp_scan_sensdev_cleanup(&this->base);
    free(this);
}

static ipmi_sdr_header_t* adm1024_sens_default_sdr(pp_tp_sensdev_t* s) {
    adm1024_sens_t* this = (adm1024_sens_t*)s;
    adm1024_sens_cfg_t* cfg = this->cfg;
    ipmi_sdr_full_sensor_t* sdr;

    sdr = ipmi_sdr_full_part_create(cfg->type->sens_type,
			       cfg->type->reading_type,
			       cfg->type->unit_type,
			       cfg->type->analog_type,
			       cfg->sdr_mtol, 
			       PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
			       0, cfg->type->sdr_rexp,
			       ipmi_sdr_max_by_form(cfg->type->analog_type),
			       ipmi_sdr_min_by_form(cfg->type->analog_type));
    return (ipmi_sdr_header_t*)sdr;
}

static void adm1024_update_temp_reading(pp_sensor_scannable_t* s) {
    adm1024_sens_t* this = PP_TP_SCANNABLE_2_OBJ_CAST(s, adm1024_sens_t);
    pp_tp_i2c_chip_t* chip = this->chip;
    adm1024_sens_cfg_t* cfg = this->cfg;
    int ret = -2;
    unsigned char value;
    
    if (pp_tp_i2c_chip_is_initialized(chip) &&
	pp_tp_scan_sensdev_do_scan(&this->base)) {
	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	    ret = pp_tp_i2c_chip_rx_byte_data(chip, cfg->reg, &value);
	    pp_tp_i2c_chip_post_com(chip);
	    if (ret == PP_SUC) ret = value;
	}
    } 
    pp_tp_scan_sensdev_update_reading(&this->base, ret);
}

static void amd1024_update_tach_reading(pp_sensor_scannable_t* s) {
    adm1024_sens_t* this = PP_TP_SCANNABLE_2_OBJ_CAST(s, adm1024_sens_t);
    pp_tp_i2c_chip_t* chip = this->chip;
    adm1024_sens_cfg_t* cfg = this->cfg;
    int ret = -2, prescale;
    unsigned char cnt, raw_prescale;

    if (pp_tp_i2c_chip_is_initialized(chip) &&
	pp_tp_scan_sensdev_do_scan(&this->base)) {
	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	    if (PP_ERR == pp_tp_i2c_chip_rx_byte_data(chip, cfg->reg, &cnt) ||
		PP_ERR == pp_tp_i2c_chip_rx_byte_data(chip,
						      ADM1024_REG_FAN_DEVISOR,
						      &raw_prescale)) {
		goto finally;
	    }
	    
	    prescale = adm1024_fan_cfg_to_prescale((raw_prescale >>
						    cfg->fan_prescale_shift)
						   & 0x03);
	    ret = 20000 / cnt / prescale;
	    if (ret > 0xff) ret = 0xff; /* limit to 8 bit */
	    
	finally:
	    pp_tp_i2c_chip_post_com(chip);
	}
    } 
    pp_tp_scan_sensdev_update_reading(&this->base, ret);
}

static void amd1024_update_voltage_reading(pp_sensor_scannable_t* s) {
    adm1024_sens_t* this = PP_TP_SCANNABLE_2_OBJ_CAST(s, adm1024_sens_t);
    pp_tp_i2c_chip_t* chip = this->chip;
    adm1024_sens_cfg_t* cfg = this->cfg;
    int ret = -2;
    unsigned char value;
    
    if (pp_tp_i2c_chip_is_initialized(chip) &&
	pp_tp_scan_sensdev_do_scan(&this->base)) {
	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	    ret = pp_tp_i2c_chip_rx_byte_data(chip, cfg->reg, &value);
	    pp_tp_i2c_chip_post_com(chip);
	    if (ret == PP_SUC) ret = value;
	}
    } 
    pp_tp_scan_sensdev_update_reading(&this->base, ret);
}
