/**
 * adm1029.c
 *
 * implements the ADM1029 Monitor / Fan controller chip
 * This file includes all related objects (chip, sensors, pwm, gpio)
 * 
 * (c) 2005 Peppercon AG, 2005/10/28, thomas@peppercon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_chip.h>
#include <pp/bmc/tp_scan_sensdev.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/tp_gpio_dev.h>

#include "drivers/adm1029.h"

/*
 * The chip, may contain global initialization code
 ****************************************************/

typedef struct pp_tp_i2c_chip_t {
    pp_tp_i2c_chip_t base;
    unsigned char fan1_ticks_revol;
    unsigned char fan2_ticks_revol;
    unsigned char gpio_mask;
    unsigned char gpio_io_mask;
} adm1029_chip_t;

static void adm1029_dtor(pp_tp_obj_t* o);
static void adm1029_power_up(pp_tp_i2c_chip_t* this);
static inline int adm1029_fan_ticks_valid(unsigned char ticks);
static unsigned char adm1029_fan_ticks_to_cfg(unsigned char ticks);
#if 0
static unsigned int adm1029_fan_ticks_to_freq(unsigned char ticks);
#endif

pp_tp_obj_t* pp_sensor_adm1029_ctor(const char* id, vector_t* args) {
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    adm1029_chip_t* o = NULL;
    pp_tp_i2c_comdev_t* i2cdev;
    int i2caddr;
    pp_tp_cond_t* icond;
    unsigned char fan1_ticks_revol;
    unsigned char fan2_ticks_revol;
    unsigned char gpio_mask;
    unsigned char gpio_io_mask;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>do<sc>d<c>d<c>d<c>d<c>",
			&i2cdev, &i2caddr, &icond,
			&fan1_ticks_revol, &fan2_ticks_revol,
			&gpio_mask, &gpio_io_mask) != 7) {
	pp_bmc_log_perror("%s(): '%s' failed: %s", ___F, id,
			  pp_strstream_buf(&err));
    } else if (!adm1029_fan_ticks_valid(fan1_ticks_revol) ||
	       !adm1029_fan_ticks_valid(fan2_ticks_revol)) {
	errno = EINVAL;
	pp_bmc_log_perror("%s(): '%s' failed: ticks can be 1, 2 or 4",
			  ___F, id);
    } else {
	o = malloc(sizeof(adm1029_chip_t));
	pp_tp_i2c_chip_init(&o->base, PP_TP_I2C_CHIP, id, adm1029_dtor,
			    ADM1029_MODEL_STRING, i2cdev, i2caddr, icond,
			    adm1029_power_up, NULL);
	o->fan1_ticks_revol = fan1_ticks_revol;
	o->fan2_ticks_revol = fan2_ticks_revol;
	o->gpio_mask = gpio_mask;
	o->gpio_io_mask = gpio_io_mask;
    }
    
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)o;
}

static void adm1029_dtor(pp_tp_obj_t* o) {
    pp_tp_i2c_chip_t* this = (pp_tp_i2c_chip_t*)o;
    assert(this != NULL);
    pp_tp_i2c_chip_cleanup(this);
    free(this);
}

/* some convience i2c communication functions */
#define I2C_RX_BYTE(reg, data)				          \
    if (PP_ERR == pp_tp_i2c_chip_rx_byte_data(chip, reg, data)) { \
        goto finally;                                             \
    }

#define I2C_TX_BYTE(reg, data)                                    \
    if (PP_ERR == pp_tp_i2c_chip_tx_byte_data(chip, reg, data)) { \
	goto finally;                                             \
    }

static void adm1029_power_up(pp_tp_i2c_chip_t* chip) {
    adm1029_chip_t* this = (adm1029_chip_t*)chip;
    unsigned char b, b1, b2;
    unsigned char r, m, i;
    
    if (PP_ERR == pp_tp_i2c_chip_pre_com(chip)) return;

    /* enable both fans */
    I2C_RX_BYTE(ADM1029_CONFIG_REGISTER, &b);
    I2C_TX_BYTE(ADM1029_CONFIG_REGISTER, b | 0x1);
    
    /* setup tachos, configure Oszillator frequency */
    I2C_RX_BYTE(ADM1029_FAN_1_CONFIGURATION, &b);
    b1 = adm1029_fan_ticks_to_cfg(this->fan1_ticks_revol);
    I2C_TX_BYTE(ADM1029_FAN_1_CONFIGURATION, (b & 0x3F) | b1 );
    I2C_RX_BYTE(ADM1029_FAN_2_CONFIGURATION, &b);
    b2 = adm1029_fan_ticks_to_cfg(this->fan2_ticks_revol);
    I2C_TX_BYTE(ADM1029_FAN_2_CONFIGURATION, (b & 0x3F) | b2 );
    
    /* setup pwms for manual mode */
    I2C_TX_BYTE(ADM1029_LOCAL_TEMP_COOLING_ACTION, 0);
    I2C_TX_BYTE(ADM1029_REMOTE_1_TEMP_COOLING_ACTION, 0);
    I2C_TX_BYTE(ADM1029_REMOTE_2_TEMP_COOLING_ACTION, 0);
    
    /* setup gpios, gpio io mask, gpio mask */
    for (i = 0, r = ADM1029_GPIO0_BEHAVIOR; r <= ADM1029_GPIO6_BEHAVIOR;
	 ++i, ++r) {
	m = ((~this->gpio_io_mask >> i) & 0x01) | 0x02 /* active high */;
	I2C_TX_BYTE(r, m);
    }
    /* check whether termal diode was detected, if yes and user   *
     * requested to enable that pin as gpio, issue warning and skip */
    I2C_RX_BYTE(ADM1029_GPIO_PRESENT_AIN, &b);
    for (i = 0; i < 7; ++i) {
	m = 0x01 << i;
	if (i > 2 && (this->gpio_mask & m) != 0 && (b & m) == 0) {
	    pp_bmc_log_warn("%s(): '%s' skipped configuration of GPIO %d, "
			    "thermal diode detected", ___F,
			    pp_tp_obj_to_string(&chip->base), i);
	} else {
	    b = b | (this->gpio_mask & m);
	}
    }
    I2C_TX_BYTE(ADM1029_GPIO_PRESENT_AIN, b);

    /* enable scanning */
    I2C_RX_BYTE(ADM1029_CONFIG_REGISTER, &b);
    I2C_TX_BYTE(ADM1029_CONFIG_REGISTER, b | 0x10);
    
 finally:
    pp_tp_i2c_chip_post_com(chip);
}

static inline int adm1029_fan_ticks_valid(unsigned char ticks) {
    return (ticks == 1 || ticks == 2 || ticks == 4);
}

static unsigned char adm1029_fan_ticks_to_cfg(unsigned char ticks) {
    unsigned char ret;
    switch(ticks) {
      default:	assert(0);
      case 1: ret = 0x40; break;
      case 2: ret = 0x80; break;
      case 4: ret = 0xC0; break;
    }
    return ret;
}

#if 0 // not needed, currently
static unsigned int adm1029_fan_ticks_to_freq(unsigned char ticks) {
    unsigned int ret;
    switch(ticks) {
      default:	assert(0);
      case 1: ret = 470; break;
      case 2: ret = 940; break;
      case 4: ret = 1880; break;
    }
    return ret;
}
#endif

/*
 * sensors
 ************************/
typedef struct adm1029_sens_type_cfg_s {
    unsigned char sens_type;
    unsigned char reading_type;
    unsigned short sdr_mtol;
    unsigned short analog_type;
    unsigned short unit_type;
    void (*update)(pp_sensor_scannable_t*);
} adm1029_sens_type_cfg_t;

typedef struct adm1029_sens_cfg_s {
    unsigned char reg;
    adm1029_sens_type_cfg_t* type;
} adm1029_sens_cfg_t;

typedef struct adm1029_sens_s {
    pp_tp_scan_sensdev_t base;
    pp_tp_i2c_chip_t* chip;
    adm1029_sens_cfg_t* cfg;
    unsigned char ticks_per_revol;
} adm1029_sens_t;

static void adm1029_sens_destroy(pp_tp_obj_t* o);
static ipmi_sdr_header_t* adm1029_sens_default_sdr(pp_tp_sensdev_t* s);
static void adm1029_update_temp_reading(pp_sensor_scannable_t*);
static void amd1029_update_tach_reading(pp_sensor_scannable_t*);

adm1029_sens_type_cfg_t adm1029_sens_type_cfg_temp = {
    .sens_type    = IPMI_SENSOR_TYPE_TEMPERATURE,
    .reading_type = IPMI_EVENT_READING_TYPE_THRESHOLD,
    .analog_type  = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
    .unit_type    = IPMI_UNIT_TYPE_DEGREES_C,
    .sdr_mtol     = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
    .update       = adm1029_update_temp_reading,
};

adm1029_sens_type_cfg_t adm1029_sens_type_cfg_tach = {
    .sens_type    = IPMI_SENSOR_TYPE_FAN,
    .reading_type = IPMI_EVENT_READING_TYPE_THRESHOLD,
    .analog_type  = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
    .unit_type    = IPMI_UNIT_TYPE_RPM,
    .sdr_mtol     = PP_BMC_SDR_M_TOL_2_MTOL(120, 1),
    .update       = amd1029_update_tach_reading,
};
    
adm1029_sens_cfg_t adm1029_sens_cfg[5] = {
    { .reg  = ADM1029_LOCAL_TEMP_VALUE,
      .type = &adm1029_sens_type_cfg_temp, },
    { .reg  = ADM1029_REMOTE_1_TEMP_VALUE,
      .type = &adm1029_sens_type_cfg_temp, },
    { .reg  = ADM1029_REMOTE_2_TEMP_VALUE,
      .type = &adm1029_sens_type_cfg_temp, },
    { .reg  = ADM1029_FAN_1_TACH_VALUE,
      .type = &adm1029_sens_type_cfg_tach, },
    { .reg  = ADM1029_FAN_2_TACH_VALUE,
      .type = &adm1029_sens_type_cfg_tach, }
};	
      
pp_tp_obj_t* pp_sensor_adm1029_sens_ctor(const char* id, vector_t* args) {
    const char* errmsg = "%s(): '%s' failed: %s";
    adm1029_sens_t* this = NULL;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_i2c_chip_t* chip;
    unsigned int sens_no;

    if (pp_tp_arg_scanf(args, 0, &err, "o<h>d", &chip, &sens_no) != 2) {
	pp_bmc_log_perror(errmsg, ___F, id, pp_strstream_buf(&err));
    } else if (strcmp(chip->model, ADM1029_MODEL_STRING)) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "incorrect chip model");
    } else if (sens_no > 4) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "invalid sensor number");
    } else {
	adm1029_sens_cfg_t* cfg = &adm1029_sens_cfg[sens_no];
	this = malloc(sizeof(adm1029_sens_t));
	pp_tp_scan_sensdev_init(&this->base, PP_TP_SCAN_SENS_DEV, id,
				(void(*)(pp_tp_obj_t*))adm1029_sens_destroy,
				adm1029_sens_default_sdr, cfg->type->update,
				chip->init_cond);
	this->chip = pp_tp_i2c_chip_duplicate(chip);
	this->cfg = cfg;

	/* register for scanning  */
	pp_tp_scan_sensdev_register(&this->base, 0);
    }
    
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void adm1029_sens_destroy(pp_tp_obj_t* o) {
    adm1029_sens_t* this = (adm1029_sens_t*)o;
    assert(this != NULL);
    pp_tp_scan_sensdev_unregister(&this->base);
    pp_tp_i2c_chip_release(this->chip);
    pp_tp_scan_sensdev_cleanup(&this->base);
    free(this);
}

static ipmi_sdr_header_t* adm1029_sens_default_sdr(pp_tp_sensdev_t* s) {
    adm1029_sens_t* this = (adm1029_sens_t*)s;
    adm1029_sens_type_cfg_t* cfg = this->cfg->type;
    ipmi_sdr_full_sensor_t* sdr;

    sdr = ipmi_sdr_full_part_create(cfg->sens_type, cfg->reading_type,
			       cfg->unit_type, cfg->analog_type,
			       cfg->sdr_mtol, 
			       PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 0, 0,
			       ipmi_sdr_max_by_form(cfg->analog_type),
			       ipmi_sdr_min_by_form(cfg->analog_type));

    return (ipmi_sdr_header_t*)sdr;
}

static void adm1029_update_temp_reading(pp_sensor_scannable_t* s) {
    adm1029_sens_t* this = PP_TP_SCANNABLE_2_OBJ_CAST(s, adm1029_sens_t);
    pp_tp_i2c_chip_t* chip = this->chip;
    adm1029_sens_cfg_t* cfg = this->cfg;
    int ret = -2;
    unsigned char value;
    
    if (pp_tp_i2c_chip_is_initialized(chip) &&
	pp_tp_scan_sensdev_do_scan(&this->base)) {
	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	    ret = pp_tp_i2c_chip_rx_byte_data(chip, cfg->reg, &value);
	    pp_tp_i2c_chip_post_com(chip);
	    if (ret == PP_SUC) ret = value;
	}
    } 
    pp_tp_scan_sensdev_update_reading(&this->base, ret);
}

/*
 * Note:
 * rotations per minute must be found with the following formula:
 *
 * R = (f * 4 * 60) / (cnt * N)
 *
 * where R   = Rotations per minute
 *       f   = oszilator freq
 *       cnt = ticks found in counter reg
 *       N   = number of ticks per revolution (1, 2 or 4)
 *
 * Since power up init code will configure f according to N
 * resulting f' will actually have a constant value of 470 Hz.
 * Moreover, for IPMI reading values beeing a 8 bit unsigned number,
 * the result will be scaled down by a factor of 120 (see m in sdr).
 *
 * So the actually used formula will be:
 *
 * f' = f / N = 470 Hz
 * r  = (2 * f') / cnt
 *    = 940 / cnt
 */
static void amd1029_update_tach_reading(pp_sensor_scannable_t* s) {
    adm1029_sens_t* this = PP_TP_SCANNABLE_2_OBJ_CAST(s, adm1029_sens_t);
    pp_tp_i2c_chip_t* chip = this->chip;
    adm1029_sens_cfg_t* cfg = this->cfg;
    int ret = -2;
    unsigned char cnt;

    if (pp_tp_i2c_chip_is_initialized(chip) &&
	pp_tp_scan_sensdev_do_scan(&this->base)) {
	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	    ret = pp_tp_i2c_chip_rx_byte_data(chip, cfg->reg, &cnt);
	    pp_tp_i2c_chip_post_com(chip);
	    if (ret == PP_SUC) {
		if (cnt > 0) {
		    ret = (60 * 4 * 470) / (120 * cnt);
		    //pp_log("cnt=%hhu, rpm=%u\n", cnt, (60 * 4 * 470) / cnt);
		} else {
		    ret = 0;
		}
		if (ret > 0xff) ret = 0xff; /* limit to 8 bit */
	    }
	}
    } 
    pp_tp_scan_sensdev_update_reading(&this->base, ret);
}

/*
 * pwms
 **************************/

typedef struct adm1029_pwm_cfg_s {
    unsigned char reg;
} adm1029_pwm_cfg_t;

typedef struct adm1029_pwm_s {
    pp_tp_pwm_act_t base;
    pp_tp_i2c_chip_t* chip;
    adm1029_pwm_cfg_t* cfg;
} adm1029_pwm_t;

adm1029_pwm_cfg_t adm1029_pwm_cfg[2] = {
    { 0x60 }, { 0x61 }
};

static void adm1029_pwm_dtor(pp_tp_obj_t* o);
static int adm1029_pwm_set_mode(pp_tp_pwm_act_t* o, pp_tp_pwm_mode_t mode);
static int adm1029_pwm_get_mode(pp_tp_pwm_act_t* o, pp_tp_pwm_mode_t*);
static int adm1029_pwm_set_duty_cycle(pp_tp_pwm_act_t* o, unsigned char );
static int adm1029_pwm_get_duty_cycle(pp_tp_pwm_act_t* o, unsigned char* );

pp_tp_obj_t* pp_sensor_adm1029_pwm_ctor(const char* id, vector_t* args) {
    const char* errmsg = "%s(): '%s' failed: %s";
    adm1029_pwm_t* this = NULL;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_i2c_chip_t* chip;
    unsigned int pwm_no;

    if (pp_tp_arg_scanf(args, 0, &err, "o<h>d", &chip, &pwm_no) != 2) {
	pp_bmc_log_perror(errmsg, ___F, id, pp_strstream_buf(&err));
    } else if (strcmp(chip->model, ADM1029_MODEL_STRING)) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "incorrect chip model");
    } else if (pwm_no > 1) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "invalid pwm number");
    } else {
	this = malloc(sizeof(adm1029_pwm_t));
	pp_tp_pwm_act_init(&this->base, PP_TP_PWM_ACT, id, adm1029_pwm_dtor,
			   adm1029_pwm_set_mode,
			   adm1029_pwm_get_mode,
			   adm1029_pwm_set_duty_cycle,
			   adm1029_pwm_get_duty_cycle);
	this->chip    = pp_tp_i2c_chip_duplicate(chip);
	this->cfg     = &adm1029_pwm_cfg[pwm_no];
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void adm1029_pwm_dtor(pp_tp_obj_t* o) {
    adm1029_pwm_t* this = (adm1029_pwm_t*)o;
    assert(this != NULL);
    pp_tp_i2c_chip_release(this->chip);
    pp_tp_pwm_act_cleanup(&this->base);
    free(this);
}

/* set and got mode not supported for this chip, currently */
static int adm1029_pwm_set_mode(pp_tp_pwm_act_t* o UNUSED,
				pp_tp_pwm_mode_t mode UNUSED) {
    errno = EINVAL;
    return PP_ERR;
}

static int adm1029_pwm_get_mode(pp_tp_pwm_act_t* o UNUSED,
				pp_tp_pwm_mode_t* mode_ptr) {
    *mode_ptr = PP_TP_PWM_MANUAL;
    return PP_SUC;
}

/*
 * Note:
 * - lower nibble of Fan register codes duty cylce
 * - higher nibble codes max speed, will be set to max
 */
static int adm1029_pwm_set_duty_cycle(pp_tp_pwm_act_t* o,
				      unsigned char duty_cycle) {
    adm1029_pwm_t* this = (adm1029_pwm_t*)o;
    pp_tp_i2c_chip_t* chip = this->chip;
    adm1029_pwm_cfg_t* cfg = this->cfg;
    int ret;
    
    if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	/* HACK: For duty cycle >0 and <7 we use 7% */
	if (duty_cycle > 0 && duty_cycle < 7) duty_cycle = 7;
#ifdef PRODUCT_ICPMMD
	/* HACK: We have to control multiple fans; maybe we should do this using the topo file */
	ret = pp_tp_i2c_chip_tx_byte_data(chip, 0x60, 0xf0 | (15 * duty_cycle / 100));
	ret = pp_tp_i2c_chip_tx_byte_data(chip, 0x61, 0xf0 | (15 * duty_cycle / 100));
#else
	ret = pp_tp_i2c_chip_tx_byte_data(chip, cfg->reg, 0xf0 | (15 * duty_cycle / 100));
#endif
	pp_tp_i2c_chip_post_com(chip);
    }
    return ret;
}

static int adm1029_pwm_get_duty_cycle(pp_tp_pwm_act_t* o,
				      unsigned char* duty_cycle_ptr) {
    adm1029_pwm_t* this = (adm1029_pwm_t*)o;
    pp_tp_i2c_chip_t* chip = this->chip;
    adm1029_pwm_cfg_t* cfg = this->cfg;
    unsigned char dc;
    int ret;
    
    if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	ret = pp_tp_i2c_chip_rx_byte_data(chip, cfg->reg, &dc);
	*duty_cycle_ptr = 100 * (dc & 0xf) / 15;
	pp_tp_i2c_chip_post_com(chip);
    }
    return ret;
}

/*
 * gpio dev
 **************************/

typedef struct adm1029_gpio_s {
    pp_tp_gpio_dev_t base;
    pp_tp_i2c_chip_t* chip;
    u_char state[7]; /* saved output state */
    u_char state_valid[7]; /* output state validity */
    pp_tp_sensdev_subscriber_t init_cond_subs;
} adm1029_gpio_t;

static void adm1029_gpio_destroy(pp_tp_obj_t* o);
static int adm1029_gpio_get_val(pp_tp_gpio_dev_t* d, unsigned long mask);
static int adm1029_gpio_set_val(pp_tp_gpio_dev_t* d, unsigned long mask,
				unsigned long val, unsigned long tristate);
static void init_cond_recv_reading(pp_tp_sensdev_subscriber_t* subscriber, 
				   pp_tp_sensdev_t* source, int reading);

pp_tp_obj_t* pp_sensor_adm1029_gpio_ctor(const char* id, vector_t* args) {
    const char* errmsg = "%s(): '%s' failed: %s";
    adm1029_gpio_t* this = NULL;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_i2c_chip_t* chip;

    if (pp_tp_arg_scanf(args, 0, &err, "o<h>", &chip) != 1) {
	pp_bmc_log_perror(errmsg, ___F, id, pp_strstream_buf(&err));
    } else if (strcmp(chip->model, ADM1029_MODEL_STRING)) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "incorrect chip model");
    } else {
	this = malloc(sizeof(adm1029_gpio_t));
	pp_tp_gpio_dev_init(&this->base, PP_FPGA_GPIO_DEV,
			    id, ((adm1029_chip_t*)chip)->gpio_mask,
			    adm1029_gpio_destroy,
			    NULL,/* no init necessary */
			    adm1029_gpio_get_val,
			    adm1029_gpio_set_val,
			    NULL);
	this->chip    = pp_tp_i2c_chip_duplicate(chip);

	/* invalidate states */
	memset(&this->state_valid, 0, sizeof(this->state_valid));

	/*
	 * NOTE: If someone sets a GPIO when the chip is not powered up yet we
	 *       have to remember the value and delay the GPIO operation until
	 *       the chip is alive. We do this using the init-condition of the
	 *       i2c-chip we are bound to. We assume that the callback is
	 *       called after the power-up callback that is registered on the
	 *       same condition!
	 */

	/* register for init condition to call init code */
	if (this->chip->init_cond) {
	    this->init_cond_subs.recv_reading = init_cond_recv_reading;
	    pp_tp_cond_subscribe(this->chip->init_cond, &this->init_cond_subs);
	}
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void adm1029_gpio_destroy(pp_tp_obj_t* o) {
    adm1029_gpio_t* this = (adm1029_gpio_t*)o;
    assert(this != NULL);
    if (this->chip->init_cond) {
	pp_tp_cond_unsubscribe(this->chip->init_cond, &this->init_cond_subs);
    }
    pp_tp_i2c_chip_release(this->chip);
    pp_tp_gpio_dev_cleanup(&this->base);
    free(this);
}

static int adm1029_gpio_get_val(pp_tp_gpio_dev_t* d UNUSED,
				unsigned long mask UNUSED) {
    adm1029_gpio_t* this = (adm1029_gpio_t*)d;
    /* check if the chip is available */
    if (this->chip->init_cond && !pp_bmc_tp_cond_is_true(this->chip->init_cond)) {
	errno = ENODEV; /* device not available */
    } else {
	errno = ENOSYS; /* not implemented */
    }
    return -1;
}

static int adm1029_gpio_set_val(pp_tp_gpio_dev_t* d, unsigned long mask,
				unsigned long val,
				unsigned long tristate UNUSED) {
    adm1029_gpio_t* this = (adm1029_gpio_t*)d;
    pp_tp_i2c_chip_t* chip = this->chip;
    adm1029_chip_t* adm1029 = (adm1029_chip_t*)chip;
    unsigned char r, i, v;
    unsigned long m;
    int ret;
    assert((mask & ~d->valid_mask) == 0);

    /* first determine the output states and remeber it for possible later use*/
    for (i = 0; i < 7; ++i) {
	m = 0x01 << i;
	// if val bit valid and gpio configure as output
	if ((mask & m) != 0 && (adm1029->gpio_io_mask & m) != 0) {
	    // set val bit negated to bit 2 of behaviour reg
	    v = (~((val >> i) << 1)) & 0x02;
	    this->state[i] = v;
	    this->state_valid[i] = 1;
	} else if ((adm1029->gpio_io_mask & m) == 0) {
	    this->state_valid[i] = 0;
	}
    }

    /* check if the chip is available */
    if (!pp_bmc_tp_cond_is_true(this->chip->init_cond)) {
	return PP_ERR;
    }

    /* output state values */
    if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	for (r = ADM1029_GPIO0_BEHAVIOR, i = 0; r <= ADM1029_GPIO6_BEHAVIOR; ++r, ++i) {
	    if (this->state_valid[i]) {
		if (PP_ERR == (ret = pp_tp_i2c_chip_tx_byte_data(chip, r, this->state[i]))) {
		    break;
		}
	    }
	}
	pp_tp_i2c_chip_post_com(chip);
    }
    return ret;
}

static void
init_cond_recv_reading(pp_tp_sensdev_subscriber_t* subscriber, 
		       pp_tp_sensdev_t* source UNUSED,  int reading)
{
    adm1029_gpio_t* this;
    unsigned char r, i;

    this = PP_TP_INTF_2_OBJ_CAST(subscriber, adm1029_gpio_t, init_cond_subs);
    if (reading > 0) {
	/* chip is available now: output state values */
	if (PP_SUCCED(pp_tp_i2c_chip_pre_com(this->chip))) {
	    for (r = ADM1029_GPIO0_BEHAVIOR, i = 0; r <= ADM1029_GPIO6_BEHAVIOR; ++r, ++i) {
		if (this->state_valid[i]) {
		    if (PP_FAILED(pp_tp_i2c_chip_tx_byte_data(this->chip, r, this->state[i]))) {
			break;
		    }
		}
	    }
	    pp_tp_i2c_chip_post_com(this->chip);
	}
    }
}
