/**
 * ads7830_sensor.c
 *
 * implements the ADS7830 ADC sensors
 * 
 * (c) 2005 Peppercon AG, rwa@peppecon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_bits.h>
#include <pp/bmc/ipmi_cmd_sensor.h>

#include <pp/bmc/sensor.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_scan_sensdev.h>

#include "drivers/ads7830_sensor.h"


/**
 * private structures
 */

/**
 * The instance of a ADS7830 sensor.
 */
typedef struct {
    pp_tp_scan_sensdev_t base;	/* basic sensor description, must be first */
    pp_tp_i2c_comdev_t* i2cdev;	/* the I2C device */
    u_char i2caddr;		/* The I2C slave address */
    u_char v_channel;           /* voltage rail */
    u_short sdr_mtol;
    u_char sdr_rexp;
} ads7830_sensor_t;

#define ADS7830_SDR_MTOL  PP_BMC_SDR_M_TOL_2_MTOL(129, 1)


/**
 * private function prototypes
 */
static ads7830_sensor_t*
ads7830_sensor_create(const char* id, pp_tp_i2c_comdev_t* i2cdev,
		      u_char i2caddr, u_int sensor_idx, int corr_factor,
		      pp_tp_cond_t* scan_cond);
static ipmi_sdr_header_t* ads7830_default_sdr(pp_tp_sensdev_t* s);
static void ads7830_update_voltage_reading(pp_sensor_scannable_t* s);
static int ads7830_read_channel(pp_tp_i2c_comdev_t* i2cdev, u_char i2caddr, u_int channel, u_char* data);
static void ads7830_destroy(ads7830_sensor_t* sensor);

/**
 * public functions
 */

pp_tp_obj_t*
pp_sensor_ads7830_ctor(const char* id, vector_t* args)
{
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    ads7830_sensor_t* o;
    pp_tp_i2c_comdev_t* i2cdev;
    pp_tp_cond_t* scan;
    int i2caddr;
    u_int snum;
    u_int corr_factor;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>ddd|o<sc>",
			&i2cdev, &i2caddr, &snum, &corr_factor, &scan) != 5) {
        pp_bmc_log_error("%s(): '%s' failed: %s (%s)",
            ___F, id, strerror(errno), pp_strstream_buf(&err));
        o = NULL;
    } else if (snum > 7) { /* we have 8 sensors (0-7) */
        pp_bmc_log_error("%s(): '%s' failed: invalid sensor number: %d",
	       ___F, id, snum);
        o = NULL;
    } else if (abs(corr_factor) > 10) {
	/* only corr factors -10..10 are supported */
        pp_bmc_log_error("%s(): '%s' failed: correction factor %d exceeds allowed range (-10...10)",
            ___F, id, corr_factor);
        o = NULL;
    } else {
        o = ads7830_sensor_create(id, i2cdev, i2caddr, snum, corr_factor,
				  scan);
    }
    pp_strstream_free(&err);

    return (pp_tp_obj_t*)o;
}

/**
 * private functions
 */

static ads7830_sensor_t*
ads7830_sensor_create(const char* id, pp_tp_i2c_comdev_t* i2cdev,
		      u_char i2caddr, u_int sensor_idx, int corr_factor,
		      pp_tp_cond_t* scan_cond)
{
    ads7830_sensor_t* ads7830;
    u_int corr_factor_abs = abs(corr_factor);
    int m, tol;

    ads7830 = calloc(1, sizeof(ads7830_sensor_t));

    /* globals */
    pp_tp_scan_sensdev_init(&ads7830->base, PP_TP_SCAN_SENS_DEV, id,
			    (void(*)(pp_tp_obj_t*))ads7830_destroy,
			    ads7830_default_sdr, 
			    ads7830_update_voltage_reading, scan_cond);
    ads7830->i2cdev = pp_tp_i2c_comdev_duplicate(i2cdev);

    /* ads7830 configuration */
    ads7830->i2caddr = i2caddr;
    ads7830->v_channel = sensor_idx;
    ads7830->sdr_rexp = -4;

    m = PP_BMC_SDR_MTOL_2_M(ADS7830_SDR_MTOL);
    tol = PP_BMC_SDR_MTOL_2_TOL(ADS7830_SDR_MTOL);

    if (corr_factor_abs > 1 && corr_factor_abs <= 10) {
	m = (m * corr_factor_abs + 5) / 10;
	++ads7830->sdr_rexp;
    }
    if (corr_factor < 0) m = -m;

    ads7830->sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(m, tol);
    
    /* register for scanning  */
    pp_tp_scan_sensdev_register(&ads7830->base, 0);

    return ads7830;
}

static ipmi_sdr_header_t*
ads7830_default_sdr(pp_tp_sensdev_t* s)
{
    ads7830_sensor_t* this = (ads7830_sensor_t*)s;
    ipmi_sdr_full_sensor_t* sdr;

    sdr = ipmi_sdr_full_part_create(IPMI_SENSOR_TYPE_VOLTAGE,
			       IPMI_EVENT_READING_TYPE_THRESHOLD,
			       IPMI_UNIT_TYPE_VOLTS,
			       IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
			       this->sdr_mtol, 
                               PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
                               0, this->sdr_rexp,
			       ipmi_sdr_max_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED),
			       ipmi_sdr_min_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED));
    
    return (ipmi_sdr_header_t*)sdr;
}

static void
ads7830_update_voltage_reading(pp_sensor_scannable_t* s)
{
    ads7830_sensor_t* ads7830 = PP_TP_SCANNABLE_2_OBJ_CAST(s,ads7830_sensor_t);
    pp_tp_i2c_comdev_t* i2cdev = ads7830->i2cdev;
    u_char i2caddr = ads7830->i2caddr;
    u_char value;
    int ret = -2;

    if (pp_tp_scan_sensdev_do_scan(&ads7830->base)) {
	if (PP_SUCCED(ret = pp_tp_i2c_comdev_pre_com(i2cdev, i2caddr))) {
	    ret = ads7830_read_channel(i2cdev, i2caddr, ads7830->v_channel,
				       &value);
	    pp_tp_i2c_comdev_post_com(i2cdev, i2caddr);
	    if (PP_SUCCED(ret)) ret = value;
	}
    }
    pp_tp_scan_sensdev_update_reading(&ads7830->base, ret);
}

static int
ads7830_read_channel(pp_tp_i2c_comdev_t* i2cdev, u_char i2caddr,
		     u_int channel, u_char* data)
{
    u_int channel_hw = ((channel & 0x6) >> 1) | ((channel & 0x1) << 2);
    return i2cdev->rx_byte_data(i2cdev, i2caddr, 0x84 | (channel_hw << 4),
				data);
}


static void
ads7830_destroy(ads7830_sensor_t* sensor)
{
    pp_tp_scan_sensdev_unregister(&sensor->base);
    pp_tp_i2c_comdev_release(sensor->i2cdev);
    pp_tp_scan_sensdev_cleanup(&sensor->base);
    free(sensor);
}
