/**
 * adt7463_pwm.c
 *
 * constructor for all ADT7463 pwm controllers
 * 
 * (c) 2005 Peppercon AG, 2005/05/24 thomas@peppercon.de
 */

#include <pp/vector.h>
#include <pp/bmc/topo_classes.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/tp_i2c_chip.h>

#include "drivers/adt7463_pwm.h"

/*
 * ADT7463 pwm structure
 */
typedef struct {
    /* we are an actor */
    pp_tp_pwm_act_t base;

    /* our i2c chip */
    pp_tp_i2c_chip_t* i2cchip;

    /* the pwm controler number */
    int pwmno;
    
} adt7463_pwm_t;

// ADT7463 registers
#define PWM1_CONFIGURATION_REGISTER    0x5c
#define PWM2_CONFIGURATION_REGISTER    0x5d
#define PWM3_CONFIGURATION_REGISTER    0x5e
  // some bit positions for PWMx configration registers
  #define PWM_CFG_SPIN 0
  #define PWM_CFG_BHVR 5
#define PWM1_DUTY_CYCLE_REGISTER       0x30
#define PWM2_DUTY_CYCLE_REGISTER       0x31
#define PWM3_DUTY_CYCLE_REGISTER       0x32


/* internal prototypes */
static int adt7463_pwm_get_duty_cycle(pp_tp_pwm_act_t* o,
                                    unsigned char* duty_cycle_ptr);
static int adt7463_pwm_set_duty_cycle(pp_tp_pwm_act_t* o,
                                    unsigned char duty_cycle);
static int adt7463_pwm_get_mode(pp_tp_pwm_act_t* o,
                            pp_tp_pwm_mode_t* mode_ptr);
static int adt7463_pwm_set_mode(pp_tp_pwm_act_t* o,
                           pp_tp_pwm_mode_t mode);
static void adt7463_pwm_dtor(pp_tp_obj_t* o);


static const unsigned char pwm_cfg_addr[] = { PWM1_CONFIGURATION_REGISTER, PWM2_CONFIGURATION_REGISTER, PWM3_CONFIGURATION_REGISTER };
static const unsigned char pwm_dc_addr[] = { PWM1_DUTY_CYCLE_REGISTER, PWM2_DUTY_CYCLE_REGISTER, PWM3_DUTY_CYCLE_REGISTER };


pp_tp_obj_t* pp_adt7463_pwm_ctor(const char* id, vector_t* args) {
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    adt7463_pwm_t* o = NULL;

    pp_tp_i2c_chip_t* i2cchip;
    int pwmno;

    /** 
     * Note: initial state argument removed because initial state cannot
     * be set when host is powered off.
     */
    if (pp_tp_arg_scanf(args, 0, &err,"o<h>d",
                   &i2cchip,&pwmno) != 2)
    {
        pp_bmc_log_error("[ADT7463] pwm '%s' init: wrong arguments (%s)", id, pp_strstream_buf(&err));
    }
    else if (pwmno < 0 || pwmno > PP_PWM_ADT7463_MAX_PWM_NUM) {
        pp_bmc_log_error("[ADT7463] pwm '%s' init: wrong pwm No. referenced", id);
    }
    else if (strcmp((i2cchip)->model, "adt7463") != 0) {
        pp_bmc_log_error("[ADT7463] pwm '%s' needs adt7463 chip to initialize", id);
        o = NULL;
    } else {
        o = malloc(sizeof(adt7463_pwm_t));
        pp_tp_pwm_act_init(&o->base, PP_TP_PWM_ACT, id, adt7463_pwm_dtor,
                           adt7463_pwm_set_mode,
                           adt7463_pwm_get_mode,
                           adt7463_pwm_set_duty_cycle,
                           adt7463_pwm_get_duty_cycle);
        o->i2cchip = pp_tp_i2c_chip_duplicate(i2cchip);
        o->pwmno = pwmno;
    }
    return (pp_tp_obj_t*)o;
}                                             

static void adt7463_pwm_dtor(pp_tp_obj_t* o) {
    adt7463_pwm_t* this = (adt7463_pwm_t*)o;
    if (this != NULL) {
	pp_tp_i2c_chip_release(this->i2cchip);
	pp_tp_pwm_act_cleanup(&this->base);
	free(this);
    }
}

static int adt7463_pwm_set_mode(pp_tp_pwm_act_t* o,
                            pp_tp_pwm_mode_t mode)
{
    adt7463_pwm_t* this = (adt7463_pwm_t*)o;
    pp_tp_i2c_chip_t* i2cchip = this->i2cchip;
    unsigned m;
    int ret = PP_SUC;

    switch (mode) {
      case PP_TP_PWM_DISABLE:
	  m = 0x4 << PWM_CFG_BHVR | 0x2 << PWM_CFG_SPIN;
	  break;
      case PP_TP_PWM_MANUAL:
	  m = 0x7 << PWM_CFG_BHVR | 0x2 << PWM_CFG_SPIN;
	  break;
      case PP_TP_PWM_AUTO:
	  // this is full speed currently, hehe.. ;-)
	  m = 0x3 << PWM_CFG_BHVR | 0x2 << PWM_CFG_SPIN;
	  break;
      default:
	  errno = EINVAL;
	  return PP_ERR;
    }
    
    if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(i2cchip)))
    {
        // set mode
        ret = pp_tp_i2c_chip_tx_byte_data(i2cchip, pwm_cfg_addr[this->pwmno], m);

	pp_tp_i2c_chip_post_com(i2cchip);
    }
    return ret;
}

static int adt7463_pwm_get_mode(pp_tp_pwm_act_t* o,
                            pp_tp_pwm_mode_t* mode_ptr)
{
    adt7463_pwm_t* this = (adt7463_pwm_t*)o;
    pp_tp_i2c_chip_t* i2cchip = this->i2cchip;
    unsigned char m;
    int ret = PP_SUC;
    
    if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(i2cchip))) {
        ret == pp_tp_i2c_chip_rx_byte_data(i2cchip, pwm_cfg_addr[this->pwmno], &m);
        pp_tp_i2c_chip_post_com(i2cchip);
    }
    if (ret == PP_ERR) {
        return PP_ERR;
    }
    
    switch (m >> PWM_CFG_BHVR) {
      case 0x4: *mode_ptr = PP_TP_PWM_DISABLE; break;
      case 0x7: *mode_ptr = PP_TP_PWM_MANUAL; break;
      default: *mode_ptr = PP_TP_PWM_AUTO; break; // any auto will map to this
    }
    
    return ret;
}

static int adt7463_pwm_set_duty_cycle(pp_tp_pwm_act_t* o,
                                    unsigned char duty_cycle)
{
    adt7463_pwm_t* this = (adt7463_pwm_t*)o;
    pp_tp_i2c_chip_t* i2cchip = this->i2cchip;
    int ret;
    
    if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(i2cchip))) {
        // pp_bmc_log_debug("[ADT7463] pwm_set_duty_cycle: pwm_dc_addr[%d](%d) = %d", this->pwmno, pwm_dc_addr[this->pwmno], duty_cycle);
	ret = pp_tp_i2c_chip_tx_byte_data(i2cchip,
                                          pwm_dc_addr[this->pwmno],
                                          duty_cycle);
	pp_tp_i2c_chip_post_com(i2cchip);
    }
    return ret;
}

static int adt7463_pwm_get_duty_cycle(pp_tp_pwm_act_t* o,
				      unsigned char* duty_cycle_ptr)
{
    adt7463_pwm_t* this = (adt7463_pwm_t*)o;
    pp_tp_i2c_chip_t* i2cchip = this->i2cchip;
    int ret;

    if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(i2cchip))) {
	ret = pp_tp_i2c_chip_rx_byte_data(i2cchip,
                                          pwm_cfg_addr[this->pwmno],
                                          duty_cycle_ptr);
        pp_tp_i2c_chip_post_com(i2cchip);
    }
    
    return ret;
}

