/**
 * cheyenne_fan_ctrl.c
 *
 * An object controlling the fan speed on the AMD Cheyenne board.
 * 
 * (c) 2006 Peppercon AG, Mirko Klefker <mkl@peppercon.de>
 */

#include <math.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/tp_ipmi_sens.h>
#include "drivers/cheyenne_fan_ctrl.h"

static void cheyenne_fan_ctrl_gpio_recv_reading(pp_tp_sensdev_subscriber_t*, pp_tp_sensdev_t*, int);
static void cheyenne_fan_ctrl_temp_recv_reading(pp_tp_sensdev_subscriber_t*, pp_tp_sensdev_t*, int);
static void cheyenne_fan_ctrl_temp_recv_event(pp_tp_ipmi_sens_subscriber_t*, pp_tp_ipmi_sens_t*, int);
static void cheyenne_fan_ctrl_fan_recv_event(pp_tp_ipmi_sens_subscriber_t*, pp_tp_ipmi_sens_t*, int);
static void cheyenne_fan_ctrl_adjust_fans(pp_cheyenne_fan_ctrl_t* this);
static double cheyenne_convert_sensor_reading(pp_tp_ipmi_sens_t * sensor);

#define FORCE_FAN_SPEED_40	0x00
#define FORCE_FAN_SPEED_70	0x01
#define FORCE_FAN_SPEED_100	0x02
#define FORCE_FAN_SPEED_DEFAULT 0x03

pp_tp_obj_t*
pp_cheyenne_fan_ctrl_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[CheyenneFanCtrl] %s: '%s' failed (%s)";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_cheyenne_fan_ctrl_t * instance = NULL;
    pp_tp_gpio_multi_sens_t * fan_speed_multi_gpio = NULL;
    vector_t * pwm_actors = NULL;
    vector_t * temp_sensors = NULL;
    vector_t * fan_sensors = NULL;
    vector_t * mid_speed_sensors = NULL;
    int pwm_count, fan_count, temp_count;
    int i = 0;

    /* get fan speed multi gpio and pwm/temp/fan counts */
    if (pp_tp_arg_scanf(args, i, &err, "o<sm>ddd", &fan_speed_multi_gpio, &pwm_count,
			&temp_count, &fan_count) != 4) {
	pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	goto bail;
    }

    /* initialize vectors */
    pwm_actors = vector_new(NULL, pwm_count,
			    (vector_elem_del_func_simple)pp_tp_obj_release);
    temp_sensors = vector_new(NULL, fan_count,
			      (vector_elem_del_func_simple)pp_tp_obj_release);
    fan_sensors = vector_new(NULL, temp_count,
			     (vector_elem_del_func_simple)pp_tp_obj_release);
    mid_speed_sensors = vector_new(NULL, 0,
			     (vector_elem_del_func_simple)pp_tp_obj_release);

    for (i = 0; i < pwm_count; ++i) {
	pp_tp_pwm_act_t * pwm_act;
        if (pp_tp_arg_scanf(args, 4 + i, &err, "o<ap>", &pwm_act) != 1) {
	    pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
            goto bail;
        }
	vector_add(pwm_actors, pp_tp_pwm_act_duplicate(pwm_act));
    }

    for (i = 0; i < temp_count; ++i) {
	pp_tp_ipmi_sens_t * ipmi_sens;
        if (pp_tp_arg_scanf(args, 4 + pwm_count + i, &err, "o<p>", &ipmi_sens)
	    != 1) {
	    pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
            goto bail;
        }
	vector_add(temp_sensors, pp_tp_ipmi_sens_duplicate(ipmi_sens));
    }

    for (i = 0; i < fan_count; ++i) {
	pp_tp_ipmi_sens_t * ipmi_sens;
        if (pp_tp_arg_scanf(args, 4 + pwm_count + temp_count + i, &err, "o<p>",
			    &ipmi_sens) != 1) {
	    pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
            goto bail;
        }
	vector_add(fan_sensors, pp_tp_ipmi_sens_duplicate(ipmi_sens));
    }

    instance = calloc(1, sizeof(pp_cheyenne_fan_ctrl_t));
    if (PP_FAILED(pp_cheyenne_fan_ctrl_init(instance, PP_TP_FAN_CTRL, id,
					    pp_cheyenne_fan_ctrl_dtor,
					    pwm_actors, temp_sensors,
					    fan_sensors, mid_speed_sensors,
					    fan_speed_multi_gpio))) {
	pp_bmc_log_error(errmsg, ___F, id, "init");
	free(instance);
	instance = NULL;
    }

 bail:
    pp_strstream_free(&err);
    if (instance == NULL) {
	vector_delete(temp_sensors);
	vector_delete(fan_sensors);
    }
    return (pp_tp_obj_t*)instance;
}

void
pp_cheyenne_fan_ctrl_dtor(pp_tp_obj_t* this)
{
    pp_cheyenne_fan_ctrl_cleanup((pp_cheyenne_fan_ctrl_t*)this);
    free(this);
}

int
pp_cheyenne_fan_ctrl_init(pp_cheyenne_fan_ctrl_t* this, pp_tp_obj_type_t type,
			  const char* id, pp_tp_obj_dtor_func_t dtor,
			  vector_t * pwm_actors, vector_t * temp_sensors,
			  vector_t * fan_sensors, vector_t * mid_speed_sensors,
			  pp_tp_gpio_multi_sens_t * fan_speed_multi_gpio) {
    pp_tp_ipmi_sens_t* ipmi_sens;
    
    if (PP_SUCCED(pp_tp_ctrl_init(&this->base, type, id, dtor))) {
	u_int temp_sensor_cnt = vector_size(temp_sensors);
	u_int fan_sensor_cnt = vector_size(fan_sensors);
	u_int i;
	this->force_fan_speed = FORCE_FAN_SPEED_DEFAULT;
	this->crit_cnt = this->old_crit_cnt = 0;
	this->pwm_actors = pwm_actors;
	this->temp_sensors = temp_sensors;
	this->fan_sensors = fan_sensors;
	this->mid_speed_sensors = mid_speed_sensors;
	this->max_temp_sensor = NULL;
	this->temp_event_subscriber.recv_reading = cheyenne_fan_ctrl_temp_recv_event;
	this->temp_reading_subscriber.recv_reading = cheyenne_fan_ctrl_temp_recv_reading;
	this->fan_event_subscriber.recv_reading = cheyenne_fan_ctrl_fan_recv_event;
	this->fan_speed_multi_gpio =
	    pp_tp_gpio_multi_sens_duplicate(fan_speed_multi_gpio);
	this->gpio_subscriber.recv_reading = cheyenne_fan_ctrl_gpio_recv_reading;
	pp_bmc_tp_sensdev_subscribe(&this->fan_speed_multi_gpio->base,
				&this->gpio_subscriber);
	for (i = 0; i < temp_sensor_cnt; ++i) {
	    ipmi_sens = (pp_tp_ipmi_sens_t *)vector_get(temp_sensors, i);
	    pp_tp_ipmi_sens_subscribe(ipmi_sens, &this->temp_event_subscriber);
	    pp_bmc_tp_sensdev_subscribe(ipmi_sens->sensor, &this->temp_reading_subscriber);
	}
	for (i = 0; i < fan_sensor_cnt; ++i) {
	    ipmi_sens = (pp_tp_ipmi_sens_t *)vector_get(fan_sensors, i);
	    pp_tp_ipmi_sens_subscribe(ipmi_sens, &this->fan_event_subscriber);
	}
	return PP_SUC;
    }
    return PP_ERR;
}

void
pp_cheyenne_fan_ctrl_cleanup(pp_cheyenne_fan_ctrl_t* this)
{
    u_int temp_sensor_cnt = vector_size(this->temp_sensors);
    u_int fan_sensor_cnt = vector_size(this->fan_sensors);
    pp_tp_ipmi_sens_t * ipmi_sens;
    u_int i;
    for (i = 0; i < temp_sensor_cnt; ++i) {
	ipmi_sens = (pp_tp_ipmi_sens_t *)vector_get(this->temp_sensors, i);
	pp_tp_ipmi_sens_unsubscribe(ipmi_sens, &this->temp_event_subscriber);
	pp_bmc_tp_sensdev_unsubscribe(ipmi_sens->sensor, &this->temp_reading_subscriber);
    }
    for (i = 0; i < fan_sensor_cnt; ++i) {
	ipmi_sens = (pp_tp_ipmi_sens_t *)vector_get(this->fan_sensors, i);
	pp_tp_ipmi_sens_unsubscribe(ipmi_sens, &this->fan_event_subscriber);
    }
    pp_bmc_tp_sensdev_unsubscribe(&this->fan_speed_multi_gpio->base,
			          &this->gpio_subscriber);
    vector_delete(this->pwm_actors);
    vector_delete(this->temp_sensors);
    vector_delete(this->fan_sensors);
    vector_delete(this->mid_speed_sensors);
    pp_tp_ctrl_cleanup(&this->base);
}

static void cheyenne_fan_ctrl_gpio_recv_reading(pp_tp_sensdev_subscriber_t* subs UNUSED,
					       	pp_tp_sensdev_t* source UNUSED, int reading)
{
    pp_cheyenne_fan_ctrl_t* this =
	PP_TP_INTF_2_OBJ_CAST(subs, pp_cheyenne_fan_ctrl_t, gpio_subscriber);

    pp_bmc_log_info("Get new GPIO reading %x\n", reading);

    if ( this->force_fan_speed != (u_char) reading ) {
	this->force_fan_speed = (u_char) reading;
	cheyenne_fan_ctrl_adjust_fans(this);
    }
}

static void cheyenne_fan_ctrl_temp_recv_event(pp_tp_ipmi_sens_subscriber_t* subs,
					      pp_tp_ipmi_sens_t* source, int event)
{   
    pp_cheyenne_fan_ctrl_t* this =
	PP_TP_INTF_2_OBJ_CAST(subs, pp_cheyenne_fan_ctrl_t, temp_event_subscriber);
    
    assert(PP_TP_OBJ_IS_TYPE(PP_TP_IPMI_SENS, source));
    
    this->old_crit_cnt = this->crit_cnt;
    this->old_warn_cnt = this->warn_cnt;
    
    if (event & IPMI_SENSOR_EVENT_UPPER_NON_CRIT_GOING_HI) {
	if ( event & 0x80000000 ) {	    
	    // Warning Event deasserted
	    size_t i = 0;
	    pp_bmc_log_info("Remove Warning Sensor\n");
	    for ( i = 0; i < vector_size(this->mid_speed_sensors); i++ ) {		
		pp_tp_ipmi_sens_t * warn_sensor = (pp_tp_ipmi_sens_t *)vector_get(this->mid_speed_sensors, i);
		if ( source == (pp_tp_ipmi_sens_t *)vector_get(this->mid_speed_sensors, i)) {
		    pp_tp_ipmi_sens_release(warn_sensor);
		    vector_remove(this->mid_speed_sensors, i);
		    this->warn_cnt--;
		    break;
		}
	    }
	} else {
	    // Warning Event asserted
	    vector_add(this->mid_speed_sensors, pp_tp_ipmi_sens_duplicate(source));
	    pp_bmc_log_info("Add Warning Sensor\n");
	    if ( !this->max_temp_sensor || pp_tp_sensdev_get_reading(source->sensor) >
		 pp_tp_sensdev_get_reading(this->max_temp_sensor->sensor)) {
		pp_bmc_log_info("Replace Warning Sensor\n");
		if ( this->max_temp_sensor ) {		    
		    pp_tp_ipmi_sens_release(this->max_temp_sensor);
		}
		this->max_temp_sensor = pp_tp_ipmi_sens_duplicate(source);
	    }
	    this->warn_cnt++;
	}
    } else if (event & IPMI_SENSOR_EVENT_UPPER_CRIT_GOING_HI) {
	pp_bmc_log_info("Add critical event for temp\n");
	this->crit_cnt += event & 0x80000000 ? -1 : 1;
    }
    cheyenne_fan_ctrl_adjust_fans(this);
}

static void cheyenne_fan_ctrl_temp_recv_reading(pp_tp_sensdev_subscriber_t* subs UNUSED,
					        pp_tp_sensdev_t* source UNUSED, int reading UNUSED)
{
    u_int i;
    pp_tp_ipmi_sens_t * reading_source;
    pp_cheyenne_fan_ctrl_t* this =
	PP_TP_INTF_2_OBJ_CAST(subs, pp_cheyenne_fan_ctrl_t, temp_reading_subscriber);

    assert(PP_TP_OBJ_IS_TYPE(PP_TP_SENS_DEV, source));
    reading_source = (pp_tp_ipmi_sens_t *) source;
    
    for ( i = 0; i < vector_size(this->mid_speed_sensors); i++ ) {
	pp_tp_ipmi_sens_t * ipmi_sens = (pp_tp_ipmi_sens_t *)vector_get(this->mid_speed_sensors, i);	
	if (!this->max_temp_sensor) {
	    pp_bmc_log_info("Set Max sensor initial\n");
	    this->max_temp_sensor = pp_tp_ipmi_sens_duplicate(ipmi_sens);
	}
	if ( pp_tp_sensdev_get_reading(ipmi_sens->sensor) >
	     pp_tp_sensdev_get_reading(this->max_temp_sensor->sensor)) {
	    pp_bmc_log_info("Set New Max sensor\n");
	    pp_tp_ipmi_sens_release(this->max_temp_sensor);
	    this->max_temp_sensor = pp_tp_ipmi_sens_duplicate(ipmi_sens);
	}
    }
    cheyenne_fan_ctrl_adjust_fans(this);
    pp_bmc_log_info("Temp Reading received %x", reading);
}

static void cheyenne_fan_ctrl_fan_recv_event(pp_tp_ipmi_sens_subscriber_t* subs,
					     pp_tp_ipmi_sens_t* source UNUSED, int event)
{
    pp_cheyenne_fan_ctrl_t* this =
	PP_TP_INTF_2_OBJ_CAST(subs, pp_cheyenne_fan_ctrl_t, fan_event_subscriber);

    this->old_crit_cnt = this->crit_cnt;
    
    if (event & IPMI_SENSOR_EVENT_LOWER_CRIT_GOING_LO) {
	pp_bmc_log_info("Add critical event for fans\n");
	this->crit_cnt += event & 0x80000000 ? -1 : 1;
    }
    cheyenne_fan_ctrl_adjust_fans(this);
}

static void cheyenne_fan_ctrl_adjust_fans(pp_cheyenne_fan_ctrl_t* this)
{
#if PP_BMC_DEBUG_LEVEL>=7
    const char* msg = "[CheyenneFanCtrl] %s";
#endif
    int duty_cycle = -1;
    static int initialized = 0;    

    if ( this->force_fan_speed != FORCE_FAN_SPEED_DEFAULT ) {
	switch ( this->force_fan_speed ) {
	  case FORCE_FAN_SPEED_40:
	      duty_cycle = 40;
	      pp_bmc_log_info(msg, "Force fans to 40% speed");
	      break;
	  case FORCE_FAN_SPEED_70:
	      pp_bmc_log_info(msg, "Force fans to 70% speed");
	      duty_cycle = 70;
	      break;
	  case FORCE_FAN_SPEED_100:
	      pp_bmc_log_info(msg, "Force fans to 100% speed");
	      duty_cycle = 100;
	      break;
	  default:
	      break;
	}
    } else {
	if (this->crit_cnt < 0) {
	    this->crit_cnt = 0;
	}
	if (this->warn_cnt < 0) {
	    this->warn_cnt = 0;
	}
	if (this->crit_cnt > 0) {
	    if (this->old_crit_cnt == 0) {
		pp_bmc_log_info(msg, "setting fans to high speed");
		duty_cycle = 100;
	    }
	} else if (this->warn_cnt > 0) {
	    double result = cheyenne_convert_sensor_reading(this->max_temp_sensor);
	    pp_bmc_log_info("Converted Reading %f", result);

	    if ( result < 0 ) {
		pp_bmc_log_info("Reading %f, setting fans to high speed", result);
		duty_cycle = 100;
	    }

	    duty_cycle = (int) (5.7778 * result - 304.44);

	    pp_bmc_log_info("Setting fans to %d speed\n", duty_cycle);
	    	   
	} else {
	    if (!initialized || this->old_crit_cnt > 0 || this->old_warn_cnt > 0) {
		pp_bmc_log_info(msg, "setting fans to low speed\n");
		duty_cycle = 10;
	    }
	}
    }
    if (duty_cycle >= 0) {
	u_int pwm_actor_cnt = vector_size(this->pwm_actors);
	u_int i;
	for (i = 0; i < pwm_actor_cnt; ++i) {
	    pp_tp_pwm_act_t * pwm_act =
		(pp_tp_pwm_act_t *)vector_get(this->pwm_actors, i);
	    pp_tp_pwm_set_duty_cycle(pwm_act, duty_cycle);
	}
	initialized = 1;
    }
}

static double
cheyenne_convert_sensor_reading(pp_tp_ipmi_sens_t * ipmi_sens) {
    ipmi_sdr_full_sensor_t* thresh_sdr = NULL;
    int m, b, k1, k2, val;
    double result;

    if (*ipmi_sdr_get_event_type_ptr(ipmi_sens->sdr) != IPMI_EVENT_READING_TYPE_THRESHOLD) {
	return 0.0;
    }

    thresh_sdr = (ipmi_sdr_full_sensor_t*) ipmi_sens->sdr;

    val = pp_tp_sensdev_get_reading(ipmi_sens->sensor);

    m  = PP_BMC_SDR_MTOL_2_M(thresh_sdr->mtol);
    b  = PP_BMC_SDR_BACC_2_B(thresh_sdr->bacc);
    k1 = thresh_sdr->bexp;
    k2 = thresh_sdr->rexp;    
    
    switch (thresh_sdr->unit.analog)
	{
	  case 0:	      
	      result = (double)(((m * val) +
				 (b * pow(10, k1))) * pow(10, k2));
	      break;
	  case 1:
	      if (val & 0x80) val ++;
	      /* Deliberately fall through to case 2. */
	  case 2:
	      result = (double)(((m * (int8_t)val) +
				 (b * pow(10, k1))) * pow(10, k2));
	      break;
	  default:
	      /* Oops! This isn't an analog sensor. */
	      return 0.0;
	}
    
    switch (thresh_sdr->linearization & 0x7f) {
      default:
      case IPMI_LINEARIZATION_LINEAR:
	  break;
      case IPMI_LINEARIZATION_LN:
	  result = log(result);
	  break;
      case IPMI_LINEARIZATION_LOG10:
	  result = log10(result);
	  break;
      case IPMI_LINEARIZATION_LOG2:
	  result = (double)(log(result) / log(2.0));
	  break;
      case IPMI_LINEARIZATION_E:
	  result = exp(result);
	  break;
      case IPMI_LINEARIZATION_EXP10:
	  result = pow(10.0, result);
	  break;
      case IPMI_LINEARIZATION_EXP2:
	  result = pow(2.0, result);
	  break;
      case IPMI_LINEARIZATION_1_OVER_X:
	  result = pow(result, -1.0); /*1/x w/o exception*/
	  break;
      case IPMI_LINEARIZATION_SQR:
	  result = pow(result, 2.0);
	  break;
      case IPMI_LINEARIZATION_CUBE:
	  result = pow(result, 3.0);
	  break;
      case IPMI_LINEARIZATION_SQRT:
	  result = sqrt(result);
	  break;
      case IPMI_LINEARIZATION_1_OVER_CUBE:
	  result = cbrt(result);
	  break;
    }
    return result;
}
