/**
 * linear_fan_ctrl.h
 *
 * Configurable Linear fan controller for one temperature sensor
 * and several pwms.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/tp_ipmi_sens.h>
#include "drivers/linear_fan_ctrl.h"


/* internal prototypes */
static void recv_reading(pp_tp_sensdev_subscriber_t* subscriber, 
			 pp_tp_sensdev_t* source, int reading);


pp_tp_obj_t* pp_linear_fan_ctrl_ctor(const char* id, vector_t* args)
{
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;  

    pp_linear_fan_ctrl_t*  fc = NULL;
    pp_tp_sensdev_t*       temp_sens = NULL;
    pp_tp_pwm_act_t*       pwm_act = NULL;
    vector_t *             pwm_actors = NULL;
    int i;
    
    int a;
    int b;
    int c;
    int min;
    int max;
    
    if (pp_tp_arg_scanf(args, 0, &err, "o<s>ddddd", &temp_sens, &a, &b, &c, &min, &max) != 6) {
        pp_bmc_log_error("[LinFanCtrl] invalid ctor params: %s", pp_strstream_buf(&err));
        goto bail;
    }
    
    pwm_actors = vector_new(NULL, 2, (vector_elem_del_func_simple)pp_tp_obj_release);
    // Note: vector is deleted in object destructor
    i = 6;
    while (pp_tp_arg_scanf(args, i, &err, "o<ap>", &pwm_act) == 1) {
        vector_add(pwm_actors, pwm_act);
        i++;
    }
    if (i == 0) {
        pp_bmc_log_warn("[LinFanCtrl] no fan actors registered");
    }

    fc = malloc(sizeof(pp_linear_fan_ctrl_t));
    if (pp_linear_fan_ctrl_init(fc, PP_TP_FAN_CTRL, id,
                             pp_linear_fan_ctrl_dtor,
                             temp_sens, a, b, c, min, max, pwm_actors))
    {
        pp_bmc_log_error("[LinFanCtrl] initialization failed");
        free(fc);
        fc = NULL;
    }

    if (fc != NULL) {
        pp_bmc_log_info("[LinFanCtrl] initialized with %d PWMs", vector_size(pwm_actors));
    }
    
 bail:
    pp_strstream_free(&err);

    return (pp_tp_obj_t*)fc;
}

void pp_linear_fan_ctrl_dtor(pp_tp_obj_t* this)
{
    pp_linear_fan_ctrl_cleanup((pp_linear_fan_ctrl_t*)this);
    free(this);
}

int pp_linear_fan_ctrl_init(pp_linear_fan_ctrl_t* this,
                        pp_tp_obj_type_t type, const char* id,
                        pp_tp_obj_dtor_func_t dtor,
                        pp_tp_sensdev_t* temp_sens,
                        int a, int b, int c, int min, int max, 
                        vector_t * pwm_actors)
{
    unsigned int i;
    
    if (pp_tp_ctrl_init(&this->base, type, id, dtor)) {
        return PP_ERR;
    }
    
    // duplicate
    this->temp_sensor = pp_tp_sensdev_duplicate(temp_sens);
    this->a = a;
    this->b = b;
    this->c = c;
    this->min = min;
    this->max = max;
    this->pwm_actors = pwm_actors;
    for (i=0; i<vector_size(pwm_actors); i++) {
        pp_tp_pwm_act_duplicate(vector_get(pwm_actors, i));
    }

    // subscribe
    this->temp_subscriber.recv_reading = recv_reading;
    pp_bmc_tp_sensdev_subscribe(temp_sens, &this->temp_subscriber);

    return PP_SUC;
}

void pp_linear_fan_ctrl_cleanup(pp_linear_fan_ctrl_t* this)
{
    // unsubscribe
    pp_bmc_tp_sensdev_unsubscribe(this->temp_sensor, &this->temp_subscriber);

    // release
    pp_tp_sensdev_release(this->temp_sensor);
    vector_delete(this->pwm_actors);     // members are released by vector->free handler
    
    pp_tp_ctrl_cleanup(&this->base);
}

static void recv_reading(pp_tp_sensdev_subscriber_t* subscriber, 
			 pp_tp_sensdev_t* source UNUSED, int reading)
{
    unsigned int i;
    int pwm_percent;
    int abs_pwm;
    pp_tp_pwm_act_t* pwm_act;
    
    pp_linear_fan_ctrl_t* this =
	PP_TP_INTF_2_OBJ_CAST(subscriber, pp_linear_fan_ctrl_t,
			      temp_subscriber);
    
    //pp_bmc_log_debug("[LinFanCtrl] received new reading %d", reading);
    
    if (reading >= 0) {
        // new reading received, update pwm
        // calculate pwm_percent, abs pwm
        pwm_percent = ((this->a * reading * 100) / (this->b * 100) ) + this->c;
        if (pwm_percent < this->min) pwm_percent = this->min;
        if (pwm_percent > this->max) pwm_percent = this->max;
    
        abs_pwm = ((pwm_percent * 255) / 100);
        if (abs_pwm < 0) abs_pwm = 0;
        if (abs_pwm > 255) abs_pwm = 255;

        pp_bmc_log_debug("[LinFanCtrl] setting new duty cycle %x (%d)", abs_pwm, pwm_percent);
        for (i = 0; i < vector_size(this->pwm_actors); i++) {
            pwm_act = (pp_tp_pwm_act_t*)vector_get(this->pwm_actors, i);
            pp_tp_pwm_set_duty_cycle(pwm_act, abs_pwm);
        }
    }
}
