/**
 * lm94_sensor.c
 *
 * implements the lm94 sensors
 * 
 * (c) 2006 Peppercon AG, 2006/03/24, rgue@peppercon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_bits.h>

#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_scan_sensdev.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/debug.h>

#include "drivers/lm94_sensor.h"
#include "drivers/lm94.h"


/*
 * LM94 sensor structure
 */
typedef struct sensor_config_s {
    /* the sensor type */
    unsigned char sensor_type;
    unsigned char unit_type;
    unsigned char analog_format;
    /* update function */
    void (*update)(pp_sensor_scannable_t*);
    /* configuration */
    union {
        struct {
            unsigned char reg;
        } volt;
        struct {
            unsigned char reg;
            unsigned char idx; // fan index in pulse per revolution array
        } fan;
        struct {
            unsigned char val_reg;
            unsigned char adj_reg;
        } temp;
    } config; 
    /* sdr configuration */
    unsigned short sdr_mtol;
    unsigned short sdr_bacc;
    char sdr_bexp;
    char sdr_rexp;
} sensor_config_t;

typedef struct {
    /* the basic sensor description, must be first */
    pp_tp_scan_sensdev_t base;

    /* which adt chip to use */
    pp_tp_i2c_chip_t* i2cchip;

    /* sensor configuration */
    sensor_config_t* cfg;

} lm94_sensor_t;

/* internal prototypes */
static lm94_sensor_t* lm94_sensor_create(/* globals */
					 const char* id,
					 pp_tp_i2c_chip_t* i2cchip,
					 int sensor_idx,
					 pp_tp_cond_t* scan);
//static char* lm94_to_string(lm94_sensor_t* sensor);
static ipmi_sdr_header_t* lm94_default_sdr(pp_tp_sensdev_t* s);
static void lm94_update_voltage_reading(pp_sensor_scannable_t* s);
static void lm94_update_tacho_reading(pp_sensor_scannable_t* s);
static void lm94_update_temp_reading(pp_sensor_scannable_t* s);
static void lm94_destroy(lm94_sensor_t* sensor);


/*
 * private structures
 ************************/
static sensor_config_t sensor_config[] = {
    { /* 0 - ad_in1 */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x56, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 1 - ad_in2 */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x57, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 2 - ad_in3 (+12V) */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x58, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(63, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 3 - ad_in4 */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x59, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 4 - ad_in5 */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x5a, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 5 - ad_in6 */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x5b, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 6 - ad_in7 (CPU1 core) */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x5c, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(63, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -4,
    }, { /* 7 - ad_in8 (CPU2 core) */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x5d, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(63, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -4,
    }, { /* 8 - ad_in9 (+3.3V) */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x5e, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(172, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -4,
    }, { /* 9 - ad_in10 (+5V) */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x5f, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 10 - ad_in11 */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x60, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 11 - ad_in12 */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x61, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 12 - ad_in13 */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x62, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 13 - ad_in14 */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x63, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 14 - ad_in15 (-12V) */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_voltage_reading,
        .config = { .volt = { .reg = 0x64, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(25, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(-136, 0),
	.sdr_bexp = 2, .sdr_rexp = -3,
    }, { /* 15 - fan speed 1 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_tacho_reading,
        .config = { .fan = { .reg = 0x6e, .idx = 0, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(15, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 16 - fan speed 2 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_tacho_reading,
        .config = { .fan = { .reg = 0x70, .idx = 1, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(15, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 17 - fan speed 3 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_tacho_reading,
        .config = { .fan = { .reg = 0x72, .idx = 2, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(15, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 18 - fan speed 4 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = lm94_update_tacho_reading,
        .config = { .fan = { .reg = 0x74, .idx = 3, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(15, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 19 - cpu1_tempA */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.update = lm94_update_temp_reading,
        .config = { .temp = { .val_reg = 0x50, .adj_reg = 0xee, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 20 - cpu2_tempA */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.update = lm94_update_temp_reading,
        .config = { .temp = { .val_reg = 0x51, .adj_reg = 0xef, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 21 - cpu1_tempB */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.update = lm94_update_temp_reading,
        .config = { .temp = { .val_reg = 0x06, .adj_reg = 0x8e, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 22 - cpu2_tempB */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.update = lm94_update_temp_reading,
        .config = { .temp = { .val_reg = 0x07, .adj_reg = 0x8f, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0, .sdr_rexp = 0,
    }
};



/*
 * implementations
 */

pp_tp_obj_t* pp_lm94_sensor_ctor(const char* id, vector_t* args) {
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    lm94_sensor_t* o;

    pp_tp_i2c_chip_t* i2cchip;
    int snum;
    pp_tp_cond_t* scan_cond;
    
    if (pp_tp_arg_scanf(args, 0, &err, "o<h>d|o<sc>",
			&i2cchip, &snum, &scan_cond) != 3)
    {
	pp_bmc_log_error("[LM94] sensor '%s' initialization failed: %s",
			 id, pp_strstream_buf(&err));
	o = NULL;
    } else if (snum > PP_SENSOR_LM94_MAX_SENSOR_NUM) {
	pp_bmc_log_error("[LM94] sensor '%s' initialization: invalid sensor number: %d", id, snum);
	o = NULL;
    } else if (strcmp((i2cchip)->model, "lm94") != 0) {
        pp_bmc_log_error("[LM94] sensor '%s' needs lm94 chip to initialize", id);
        o = NULL;
    } else{
        o = lm94_sensor_create(id, i2cchip, snum, scan_cond);
    }
    return (pp_tp_obj_t*)o;
}						

/*
 * private functions
 ********************************************************************/

static lm94_sensor_t* lm94_sensor_create(/* globals */
					      const char* id,
					      pp_tp_i2c_chip_t* i2cchip,
					      int sensor_idx,
					      pp_tp_cond_t* scan)
{
    sensor_config_t *s = &sensor_config[sensor_idx];
    lm94_sensor_t* adts = malloc(sizeof(lm94_sensor_t));

    /* globals */
    pp_tp_scan_sensdev_init(&adts->base, PP_TP_SCAN_SENS_DEV, id,
			    (void(*)(pp_tp_obj_t*))lm94_destroy,
			    lm94_default_sdr, s->update, scan);
    adts->i2cchip = pp_tp_i2c_chip_duplicate(i2cchip);
    adts->cfg = s;
	
    /* register for scanning  */
    pp_tp_scan_sensdev_register(&adts->base, 0);

    return adts;
}

static ipmi_sdr_header_t* lm94_default_sdr(pp_tp_sensdev_t* s) {
    sensor_config_t* cfg = ((lm94_sensor_t*)s)->cfg;
    ipmi_sdr_full_sensor_t* mysdr;

    mysdr = ipmi_sdr_full_part_create(cfg->sensor_type,
			       IPMI_EVENT_READING_TYPE_THRESHOLD,
			       cfg->unit_type,
			       cfg->analog_format,
			       cfg->sdr_mtol, cfg->sdr_bacc,
			       cfg->sdr_bexp, cfg->sdr_rexp,
			       ipmi_sdr_max_by_form(cfg->analog_format),
			       ipmi_sdr_min_by_form(cfg->analog_format));
				      
    return (ipmi_sdr_header_t*)mysdr;
}

static void lm94_update_voltage_reading(pp_sensor_scannable_t* s) {
    lm94_sensor_t* adts = PP_TP_SCANNABLE_2_OBJ_CAST(s, lm94_sensor_t);
    pp_tp_i2c_chip_t* i2cchip = adts->i2cchip;
    sensor_config_t* cfg = adts->cfg;
    int ret = -2;
    unsigned char value;

    if (pp_tp_scan_sensdev_do_scan(&adts->base)) {
        // we should read (scan condition on) and we can read (lm94 ready)
        
	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(i2cchip))) {
            ret = pp_tp_i2c_chip_rx_byte_data(i2cchip, cfg->config.volt.reg,
					      &value);
	    pp_tp_i2c_chip_post_com(i2cchip);
           
            /* return value if read successful*/
	    if (ret == PP_SUC) ret = value;
	}
    }
    
    pp_tp_scan_sensdev_update_reading(&adts->base, ret);
}

static void lm94_update_tacho_reading(pp_sensor_scannable_t* s) {
    lm94_sensor_t* adts = PP_TP_SCANNABLE_2_OBJ_CAST(s, lm94_sensor_t);
    sensor_config_t* cfg = adts->cfg;
    pp_tp_i2c_chip_t* i2cchip = adts->i2cchip;
    int ret = -2;
    u_char low, high;
    unsigned char value;
    unsigned int cycles;

    if (pp_tp_scan_sensdev_do_scan(&adts->base)) {
	// we should read (scan condition on) and we can read (lm94 ready)

	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(i2cchip))) {

	    // read all relevant registers
	    (   (PP_SUC == (ret = pp_tp_i2c_chip_rx_byte_data(i2cchip,
					 cfg->config.fan.reg, &low)))
             && (PP_SUC == (ret = pp_tp_i2c_chip_rx_byte_data(i2cchip,
					 cfg->config.fan.reg + 1, &high))) );

	    pp_tp_i2c_chip_post_com(i2cchip);

	    /* return value if read successful*/
	    if (ret == PP_SUC) {
		
		// how many clock cycles have been counted?
		cycles = (high << 6) | (low >> 2);

		if (cycles == 0x3fff || cycles == 0) {
		    // fan is not moving, set value to 0 and return
		    value = 0;
		} else {
		    // convert the count into a byte value
		    value = min((u_int)255, 22500 * 60 * 2 / cycles
		            / ((pp_tp_lm94_t*)i2cchip)->fan_pulse_per_revol[cfg->config.fan.idx]);
		}
		ret = value;
	    }
	}
    }

    pp_tp_scan_sensdev_update_reading(&adts->base, ret);
}

static void lm94_update_temp_reading(pp_sensor_scannable_t* s) {
    lm94_sensor_t* adts = PP_TP_SCANNABLE_2_OBJ_CAST(s, lm94_sensor_t);
    sensor_config_t* cfg = adts->cfg;
    pp_tp_i2c_chip_t* i2cchip = adts->i2cchip;
    int ret = -2;
    unsigned char val, adj;

    if (pp_tp_scan_sensdev_do_scan(&adts->base)) {
        // we should read (scan condition on) and we can read (lm94 ready)

	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(i2cchip))) {
	    ret = pp_tp_i2c_chip_rx_byte_data(i2cchip, cfg->config.temp.val_reg, &val);
	    if (ret == PP_SUC) {
		ret = pp_tp_i2c_chip_rx_byte_data(i2cchip, cfg->config.temp.adj_reg, &adj);
	    }
	    pp_tp_i2c_chip_post_com(i2cchip);

	    if (ret == PP_SUC) {
		// Supermicro: to check, if the value is valid,
		//             we have first to undo the adjustment added by hardware
		if (adj & 0x20) adj |= 0xc0; // sign extension
		if ((unsigned char)(val - adj) >= 0x7f) {
		    errno = ENODATA;
		    ret = PP_ERR;
		} else {
		    // use the origin value here (already including adjustment)
		    ret = val;
		}
	    } else errno = EIO;
	} else errno = EIO;
    }

    pp_tp_scan_sensdev_update_reading(&adts->base, ret);
}

static void lm94_destroy(lm94_sensor_t* sensor) {
    /* unregister from scanning  */
    pp_tp_scan_sensdev_unregister(&sensor->base);
    
    pp_tp_i2c_chip_release(sensor->i2cchip);
    pp_tp_scan_sensdev_cleanup(&sensor->base);
    free(sensor);
}

