/**
 * max6681_sensor.c
 *
 * implements the MAX6681 sensors
 * 
 * (c) 2006 Peppercon AG, miba@peppercon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_bits.h>
#include <pp/bmc/ipmi_cmd_sensor.h>

#include <pp/bmc/sensor.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_scan_sensdev.h>

#include "drivers/max6681_sensor.h"

/* MAX6685 Registers */
#define MAX_REG_RLTS	0x00
#define MAX_REG_RRTE	0x01
#define MAX_REG_RSL	0x02
#define MAX_REG_RCL	0x03
#define MAX_REG_RCRA	0x04
#define MAX_REG_RIH	0x05
#define MAX_REG_RIL	0x06
#define MAX_REG_REH	0x07
#define MAX_REG_REL	0x08
#define MAX_REG_WCL	0x09
#define MAX_REG_WCRA	0x0A
#define MAX_REG_WIH	0x0B
#define MAX_REG_WIL	0x0C
#define MAX_REG_WEH	0x0D
#define MAX_REG_WEL	0x0E
#define MAX_REG_OSHT	0x0F
#define MAX_REG_REET	0x10
#define MAX_REG_RWOH	0x11
#define MAX_REG_RWOL	0x12
/* ... */
#define MAX_REG_RWOE	0x19
#define MAX_REG_RWOI	0x20
#define MAX_REG_HYST	0x21

#define MAX_REG_RDID	0xFE
#define MAX_REG_REV	0xFF

/* Register bits */

// 0x01 reserved
#define REG_CFG_RESET			0x02
#define REG_CFG_SMB_TOUT_DISABLE	0x04
#define REG_CFG_EXTENDED_RANGE		0x08
#define REG_CFG_EXTENDED_RES		0x10
#define REG_CFG_SPNP			0x20
#define REG_CFG_STOP			0x40
#define REG_CFG_MASK			0x80

/* some magic values */
#define DIODE_FAULT			0x80

/**
 * private structures
 */

/**
 * The instance of a sensor.
 */
typedef struct {
    u_char reg;	/* temperature register to read (internal or external) */
} sensor_config_t;

typedef struct {
    pp_tp_scan_sensdev_t base;	/* basic sensor description, must be first */
    pp_tp_i2c_comdev_t* i2cdev;	/* the I2C device */
    u_char i2caddr;		/* The I2C slave address */
    sensor_config_t* cfg;
} max6681_sensor_t;


/**
 * Initialize sensor templates.
 */

static sensor_config_t sensor_config[] = {
    { /* 0 - local temperature */
	.reg = MAX_REG_RLTS,
    }, { /* 1 - remote temperature */
	.reg = MAX_REG_RRTE,
    },
};

static max6681_sensor_t*
max6681_sensor_create(const char* id, pp_tp_i2c_comdev_t* i2cdev,
		      u_char i2caddr, u_int sensor_idx, pp_tp_cond_t* scan);
static ipmi_sdr_header_t* max6681_default_sdr(pp_tp_sensdev_t* s);
static void max6681_update_temperature_reading(pp_sensor_scannable_t* s);
static void max6681_destroy(max6681_sensor_t* sensor);

/**
 * read a register
 * @return PP_SUC on success
 *         PP_ERR on failure
 */
static inline int
read_reg(pp_tp_i2c_comdev_t* i2cdev, u_char i2caddr,
	 u_char reg, u_char* data)
{
    return i2cdev->rx_byte_data(i2cdev, i2caddr, reg, data);
}

/**
 * write a register
 * @return PP_SUC on success
 *         PP_ERR on failure
 */
static inline int
write_reg(pp_tp_i2c_comdev_t* i2cdev, u_char i2caddr,
	  u_char reg, u_char data)
{
    return i2cdev->tx_byte_data(i2cdev, i2caddr, reg, data);
}

/**
 * public functions
 */

pp_tp_obj_t*
pp_sensor_max6681_ctor(const char* id, vector_t* args)
{
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    max6681_sensor_t* o;
    pp_tp_i2c_comdev_t* i2cdev;
    pp_tp_cond_t* scan;
    int i2caddr;
    u_int snum;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>dd|o<sc>",
			&i2cdev, &i2caddr, &snum, &scan) != 4) {
        pp_bmc_log_error("%s(): '%s' failed: %s (%s)",
	       ___F, id, strerror(errno), pp_strstream_buf(&err));
        o = NULL;
    } else if (snum > 2) { /* we have 3 temperature sensors (0-2) */
        pp_bmc_log_error("%s(): '%s' failed: invalid sensor number: %d",
	       ___F, id, snum);
        o = NULL;
    } else {
        o = max6681_sensor_create(id, i2cdev, i2caddr, snum, scan);
    }
    pp_strstream_free(&err);

    return (pp_tp_obj_t*)o;
}

/**
 * private functions
 */

static max6681_sensor_t*
max6681_sensor_create(const char* id, pp_tp_i2c_comdev_t* i2cdev,
		      u_char i2caddr, u_int sensor_idx, pp_tp_cond_t* scan)
{
    sensor_config_t *sc = &sensor_config[sensor_idx];
    max6681_sensor_t* max6681;

    max6681 = calloc(1, sizeof(max6681_sensor_t));

    /* globals */
    pp_tp_scan_sensdev_init(&max6681->base, PP_TP_SCAN_SENS_DEV, id,
			    (void(*)(pp_tp_obj_t*))max6681_destroy,
			    max6681_default_sdr,
			    max6681_update_temperature_reading, scan);
    max6681->i2cdev = pp_tp_i2c_comdev_duplicate(i2cdev);

    /* max6681 configuration */
    max6681->i2caddr = i2caddr;
    max6681->cfg = sc;

    /* initialize the config reg */
    write_reg(i2cdev, i2caddr, MAX_REG_WCL, REG_CFG_EXTENDED_RANGE);
    
    /* register for scanning  */
    pp_tp_scan_sensdev_register(&max6681->base, 0);
    
    return max6681;
}


static ipmi_sdr_header_t*
max6681_default_sdr(pp_tp_sensdev_t* s UNUSED)
{
    ipmi_sdr_full_sensor_t* mysdr;
    
    mysdr = ipmi_sdr_full_part_create(IPMI_SENSOR_TYPE_TEMPERATURE,
				      IPMI_EVENT_READING_TYPE_THRESHOLD,
				      IPMI_UNIT_TYPE_DEGREES_C,
				      IPMI_ANALOG_DATA_FORMAT_2_COMPL,
				      PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
				      PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
                                      0, 0,
		  ipmi_sdr_max_by_form(IPMI_ANALOG_DATA_FORMAT_2_COMPL),
   		  ipmi_sdr_min_by_form(IPMI_ANALOG_DATA_FORMAT_2_COMPL));
        
    return (ipmi_sdr_header_t*)mysdr;
}

static void
max6681_update_temperature_reading(pp_sensor_scannable_t* s)
{
    max6681_sensor_t* max6681 = PP_TP_SCANNABLE_2_OBJ_CAST(s,max6681_sensor_t);
    pp_tp_i2c_comdev_t* i2cdev = max6681->i2cdev;
    u_char i2caddr = max6681->i2caddr;
    u_char value;
    int ret = -2;

    if (pp_tp_scan_sensdev_do_scan(&max6681->base)) {
	if (PP_SUCCED(ret = pp_tp_i2c_comdev_pre_com(i2cdev, i2caddr))) {	    
	    if (PP_FAILED(ret = read_reg(i2cdev, i2caddr, max6681->cfg->reg,
					 &value))) {
		goto bail;
	    }

	    if (value == DIODE_FAULT) {
		ret = PP_ERR;
		goto bail;
	    }
	bail:
	    pp_tp_i2c_comdev_post_com(i2cdev, i2caddr);
	}
	if (PP_SUCCED(ret)) ret = value;
    }

    pp_tp_scan_sensdev_update_reading(&max6681->base, ret);
}

static void
max6681_destroy(max6681_sensor_t* sensor)
{
    pp_tp_scan_sensdev_unregister(&sensor->base);
    pp_tp_i2c_comdev_release(sensor->i2cdev);
    pp_tp_scan_sensdev_cleanup(&sensor->base);
    free(sensor);
}
