/**
 * max6695_sensor.c
 *
 * implements the MAX6695 sensors
 * 
 * (c) 2005 Peppercon AG, miba@peppecon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_bits.h>
#include <pp/bmc/ipmi_cmd_sensor.h>

#include <pp/bmc/sensor.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_scan_sensdev.h>

#include "drivers/max6695_sensor.h"

/* MAX6695 Registers */
#define MAX_REG_RLTS	0x00
#define MAX_REG_RRTE	0x01
#define MAX_REG_RSL1	0x02
#define MAX_REG_RCL	0x03
#define MAX_REG_RCRA	0x04
#define MAX_REG_RLHN	0x05
#define MAX_REG_RLLI	0x06
#define MAX_REG_RRHI	0x07
#define MAX_REG_RRLS	0x08
#define MAX_REG_WCA	0x09
#define MAX_REG_WCRW	0x0A
#define MAX_REG_WLHO	0x0B
#define MAX_REG_WLLM	0x0C
#define MAX_REG_WRHA	0x0D
#define MAX_REG_WRLN	0x0E
#define MAX_REG_OSHT	0x0F
#define MAX_REG_REET	0x10
#define MAX_REG_RIET	0x11
#define MAX_REG_RSL2	0x12

#define MAX_REG_RWO2E	0x16
#define MAX_REG_RWO2I	0x17

#define MAX_REG_RWO1E	0x19
#define MAX_REG_RWO1I	0x20
#define MAX_REG_HYST	0x21

#define MAX_REG_RDID	0xFE

/* Register bits */
#define REG_WCA_ALERT1			0x01
#define REG_WCA_ALERT2			0x02
#define REG_WCA_SMB_TOUT_DISABLE	0x04
#define REG_WCA_REMOTE2_SELECT		0x08
#define REG_WCA_RFU			0x10
#define REG_WCA_FAULT_QUEUE		0x20
#define REG_WCA_STOP			0x40
#define REG_WCA_MASK1			0x80

/* some magic values */
#define MAX6695_DIODE_FAULT		0x80

/**
 * private structures
 */

/**
 * The instance of a sensor.
 */
typedef struct {
    u_char reg;
    int channel_select; /* selected remote channel bit, -1 to not select */
} sensor_config_t;

typedef struct {
    pp_tp_scan_sensdev_t base;	/* basic sensor description, must be first */
    pp_tp_i2c_comdev_t* i2cdev;	/* the I2C device */
    u_char i2caddr;		/* The I2C slave address */
    sensor_config_t* cfg;
} max6695_sensor_t;


/**
 * Initialize sensor templates.
 */

static sensor_config_t sensor_config[] = {
    { /* 0 - local temperature */
	.reg = 0x00,
	.channel_select = -1,
    }, { /* 1 - remote temperature 1 */
	.reg = 0x01,
	.channel_select = 0,
    }, { /* 2 - remote temperature 2 */
	.reg = 0x01, .
	channel_select = 1,
    }
};

static max6695_sensor_t*
max6695_sensor_create(const char* id, pp_tp_i2c_comdev_t* i2cdev,
		      u_char i2caddr, u_int sensor_idx, pp_tp_cond_t* scan);
static ipmi_sdr_header_t* max6695_default_sdr(pp_tp_sensdev_t* s);
static void max6695_update_temperature_reading(pp_sensor_scannable_t* s);
static void max6695_destroy(max6695_sensor_t* sensor);

/**
 * read a register
 * @return PP_SUC on success
 *         PP_ERR on failure
 */
static inline int
max6695_read_register(pp_tp_i2c_comdev_t* i2cdev, u_char i2caddr,
		      u_char reg, u_char* data)
{
    return i2cdev->rx_byte_data(i2cdev, i2caddr, reg, data);
}

/**
 * write a register
 * @return PP_SUC on success
 *         PP_ERR on failure
 */
static inline int
max6695_write_register(pp_tp_i2c_comdev_t* i2cdev, u_char i2caddr,
		       u_char reg, u_char data)
{
    return i2cdev->tx_byte_data(i2cdev, i2caddr, reg, data);
}

/**
 * public functions
 */

pp_tp_obj_t*
pp_sensor_max6695_ctor(const char* id, vector_t* args)
{
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    max6695_sensor_t* o;
    pp_tp_i2c_comdev_t* i2cdev;
    pp_tp_cond_t* scan;
    int i2caddr;
    u_int snum;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>dd|o<sc>",
			&i2cdev, &i2caddr, &snum, &scan) != 4) {
        pp_bmc_log_error("%s(): '%s' failed: %s (%s)",
	       ___F, id, strerror(errno), pp_strstream_buf(&err));
        o = NULL;
    } else if (snum > 2) { /* we have 3 temperature sensors (0-2) */
        pp_bmc_log_error("%s(): '%s' failed: invalid sensor number: %d",
	       ___F, id, snum);
        o = NULL;
    } else {
        o = max6695_sensor_create(id, i2cdev, i2caddr, snum, scan);
    }
    pp_strstream_free(&err);

    return (pp_tp_obj_t*)o;
}

/**
 * private functions
 */

static max6695_sensor_t*
max6695_sensor_create(const char* id, pp_tp_i2c_comdev_t* i2cdev,
		      u_char i2caddr, u_int sensor_idx, pp_tp_cond_t* scan)
{
    sensor_config_t *sc = &sensor_config[sensor_idx];
    max6695_sensor_t* max6695;

    max6695 = calloc(1, sizeof(max6695_sensor_t));

    /* globals */
    pp_tp_scan_sensdev_init(&max6695->base, PP_TP_SCAN_SENS_DEV, id,
			    (void(*)(pp_tp_obj_t*))max6695_destroy,
			    max6695_default_sdr,
			    max6695_update_temperature_reading, scan);
    max6695->i2cdev = pp_tp_i2c_comdev_duplicate(i2cdev);

    /* max6695 configuration */
    max6695->i2caddr = i2caddr;
    max6695->cfg = sc;

    /* initialize the config reg of MAX6695 */
    max6695_write_register(i2cdev, i2caddr, MAX_REG_WCA, 0x00);
    
    /* register for scanning  */
    pp_tp_scan_sensdev_register(&max6695->base, 0);
    
    return max6695;
}


static ipmi_sdr_header_t*
max6695_default_sdr(pp_tp_sensdev_t* s UNUSED)
{
    ipmi_sdr_full_sensor_t* mysdr;
    
    mysdr = ipmi_sdr_full_part_create(IPMI_SENSOR_TYPE_TEMPERATURE,
				      IPMI_EVENT_READING_TYPE_THRESHOLD,
				      IPMI_UNIT_TYPE_DEGREES_C,
				      IPMI_ANALOG_DATA_FORMAT_2_COMPL,
				      PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
				      PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
                                      0, 0,
		  ipmi_sdr_max_by_form(IPMI_ANALOG_DATA_FORMAT_2_COMPL),
   		  ipmi_sdr_min_by_form(IPMI_ANALOG_DATA_FORMAT_2_COMPL));
        
    return (ipmi_sdr_header_t*)mysdr;
}

static void
max6695_update_temperature_reading(pp_sensor_scannable_t* s)
{
    max6695_sensor_t* max6695 = PP_TP_SCANNABLE_2_OBJ_CAST(s,max6695_sensor_t);
    pp_tp_i2c_comdev_t* i2cdev = max6695->i2cdev;
    u_char i2caddr = max6695->i2caddr;
    u_char value;
    int ret = -2;

    if (pp_tp_scan_sensdev_do_scan(&max6695->base)) {
	if (PP_SUCCED(ret = pp_tp_i2c_comdev_pre_com(i2cdev, i2caddr))) {

	    /* if we are reading a remote diode, switch to it */
	    if (max6695->cfg->channel_select != -1) {
		u_char conf_reg;
		
		if (PP_FAILED(ret = max6695_read_register(i2cdev, i2caddr,
						    MAX_REG_RCL, &conf_reg))) {
		    goto bail;
		}

		conf_reg &= ~REG_WCA_REMOTE2_SELECT;
		if (max6695->cfg->channel_select == 1) {
		    conf_reg |= REG_WCA_REMOTE2_SELECT;
		}

		if (PP_FAILED(ret = max6695_write_register(i2cdev, i2caddr,
							   MAX_REG_WCA,
							   conf_reg))) {
		    goto bail;
		}	
	    }
    
	    if (PP_FAILED(ret = max6695_read_register(i2cdev, i2caddr,
						      max6695->cfg->reg,
						      &value))) {
		goto bail;
	    }

	    if (value == MAX6695_DIODE_FAULT) {
		pp_bmc_log_warn("%s: diode fault for channel=%d", ___F,
				max6695->cfg->channel_select);
		ret = PP_ERR;
		goto bail;
	    }
	bail:
	    pp_tp_i2c_comdev_post_com(i2cdev, i2caddr);
	}
	if (PP_SUCCED(ret)) ret = value;
    }

    pp_tp_scan_sensdev_update_reading(&max6695->base, ret);
}

static void
max6695_destroy(max6695_sensor_t* sensor)
{
    pp_tp_scan_sensdev_unregister(&sensor->base);
    pp_tp_i2c_comdev_release(sensor->i2cdev);
    pp_tp_scan_sensdev_cleanup(&sensor->base);
    free(sensor);
}
