/**
 * max7312_gpio_dev.c
 *
 * Device driver base class for the MAXIM MAX7312 I2C GPIO Extender
 * 
 * (c) 2006 Peppercon AG, Mirko Klefker <mkl@peppercon.de>
 */

#include <unistd.h>
#include <fcntl.h>

#include <pp/base.h>
#include <pp/selector.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>

#include "max7312_gpio_dev.h"

#define GPIO_VALID_MASK	0x0000FFFF

static int
max7312_get_val(pp_max7312_gpio_dev_t* d)
{
    int tmpv, value = -1;
    u_char data;
    
    /* read upper 8 bit */
    if (PP_FAILED(d->i2cdev->rx_byte_data(d->i2cdev, d->i2caddr,
					  REG_OUTPUT + 1, &data))) {
	goto bail;
    }
    tmpv = data << 8;

    /* read lower 8 bit */
    if (PP_FAILED(d->i2cdev->rx_byte_data(d->i2cdev, d->i2caddr,
					  REG_OUTPUT, &data))) {
	goto bail;
    }
    
    tmpv |= data;
    
    value = tmpv & GPIO_VALID_MASK;
    
 bail:
    return value;
}

static int
max7312_set_reg(pp_max7312_gpio_dev_t* d, u_char reg, u_long mask, u_long val)
{
    int ret = PP_ERR;
    u_char data;

    mask &= GPIO_VALID_MASK;
    val  &= GPIO_VALID_MASK;
    
    /* set upper 8 bit */
    if (PP_FAILED(d->i2cdev->rx_byte_data(d->i2cdev, d->i2caddr,
					  reg * 2 + 1, &data))) {

	goto bail;
    }

    data &= ~(mask >> 8);
    data |= (val & mask) >> 8;

    //pp_bmc_log_debug("max7312: Set Data to high byte: %x", data);
    
    if (PP_FAILED(d->i2cdev->tx_byte_data(d->i2cdev, d->i2caddr,
					  reg * 2 + 1, data))) {

	goto bail;
    }

    /* set lower 8 bit */
    if (PP_FAILED(d->i2cdev->rx_byte_data(d->i2cdev, d->i2caddr,
					  reg * 2, &data))) {

	goto bail;
    }

    data &= ~(mask & 0xFF);
    data |= val & mask & 0xFF;
    
    if (PP_FAILED(d->i2cdev->tx_byte_data(d->i2cdev, d->i2caddr,
					  reg * 2, data))) {

	goto bail;
    }
    
    ret = PP_SUC;
    
 bail:
    return ret;   
}

static int
pp_max7312_gpio_get_val(pp_tp_gpio_dev_t* s, u_long mask) {
    int value;
    pp_max7312_gpio_dev_t *dev = (pp_max7312_gpio_dev_t*) s;
    assert((mask & ~s->valid_mask) == 0);

    if ((value = dev->value) >= 0) {
	value &= mask;
    }
    return value;
}

static int
pp_max7312_gpio_set_val(pp_tp_gpio_dev_t* d, u_long mask, u_long val,
			u_long tristate)
{
    pp_max7312_gpio_dev_t *dev = (pp_max7312_gpio_dev_t*) d;
    int ret = PP_ERR;
    
    // check whether access allowed per presence condition
    if (dev->pres_cond != NULL && !pp_bmc_tp_cond_is_true(dev->pres_cond)) {
	goto bail;
    }
    
    if (PP_FAILED(pp_tp_i2c_comdev_pre_com(dev->i2cdev, dev->i2caddr))) {
	goto bail;
    }
    
    /* set output value for !tristate pins */
    if (PP_FAILED(dev->set_reg(dev, REG_OUTPUT, mask & ~tristate, val))) {
	goto bail_post;
    }

    /* handle tristate */
    if (PP_FAILED(dev->set_reg(dev, REG_CONFIG, mask, tristate))) {
	goto bail_post;
    }
    
    ret = PP_SUC;

 bail_post:
    pp_tp_i2c_comdev_post_com(dev->i2cdev, dev->i2caddr);
    
 bail:
    return ret;
}

/** Call to (re)read the sensor and notify subscribers */
static void read_sensor(pp_max7312_gpio_dev_t* d)
{
    int new_value = -1;
    int old_value = d->value;
    unsigned long change_mask;

    // check whether access allowed per presence condition
    if (d->pres_cond != NULL && !pp_bmc_tp_cond_is_true(d->pres_cond)) {
	goto bail;
    }
    
    if (PP_FAILED(pp_tp_i2c_comdev_pre_com(d->i2cdev, d->i2caddr))) {
	goto bail;
    }

    new_value = d->get_val(d);
    
    pp_tp_i2c_comdev_post_com(d->i2cdev, d->i2caddr);

 bail:
    if (new_value != old_value) {
	d->value = new_value;
	if (new_value < 0) change_mask = 0;
	else change_mask = old_value < 0 ? d->base.valid_mask : 
		(unsigned long)(old_value ^ new_value);
	pp_tp_gpio_dev_notify_subscribers(&d->base, (unsigned long)new_value, change_mask, new_value < 0 ? 0 : 1);
    }
}

/** Currently we poll the gpio device FIXME: replace by interrupt */
static int read_sensor_to(const int item_id UNUSED, void* ctx) {
    read_sensor((pp_max7312_gpio_dev_t*)ctx);
    return PP_SUC;
}


void pp_max7312_gpio_dev_cleanup(pp_tp_obj_t* d) {
    pp_max7312_gpio_dev_t *this = (pp_max7312_gpio_dev_t*)d;
    
    pp_bmc_log_debug("[%s] cleanup", d->id);
    
    pp_select_remove_to(this->to_hndl);
    
    if (this->pres_cond) pp_tp_cond_release(this->pres_cond);
    pp_tp_obj_release((pp_tp_obj_t*)(this->i2cdev));
    pp_tp_gpio_dev_cleanup((pp_tp_gpio_dev_t*)this);
}


int pp_max7312_gpio_dev_init(pp_max7312_gpio_dev_t* d,
			     pp_tp_i2c_comdev_t* i2cdev,
			     u_char i2caddr, pp_tp_cond_t* pres_cond) {
    
    pp_tp_gpio_dev_init((pp_tp_gpio_dev_t*)d,
			PP_MAX7312_GPIO_DEV, "MAX7312 I2C GPIO extender",
                        GPIO_VALID_MASK,
			pp_max7312_gpio_dev_cleanup,
			NULL, /* no init necessary */
			pp_max7312_gpio_get_val,
			pp_max7312_gpio_set_val,
                        NULL);
    
    d->i2cdev = (pp_tp_i2c_comdev_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)i2cdev);
    d->i2caddr = i2caddr;
    d->pres_cond = pres_cond == NULL ? NULL : pp_tp_cond_duplicate(pres_cond);
    d->set_reg = max7312_set_reg;
    d->get_val = max7312_get_val;
    d->value = -1;
    d->to_hndl = pp_select_add_to(500, 1, read_sensor_to, d);
			     
    return PP_SUC;
}

pp_tp_obj_t* pp_max7312_gpio_dev_ctor(const char* id, vector_t* args) {
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;  
    pp_max7312_gpio_dev_t* o = NULL;

    pp_tp_i2c_comdev_t* i2cdev;
    pp_tp_cond_t* pres_cond;
    u_char i2caddr;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>d<c>|o<sc>",
			&i2cdev, &i2caddr, &pres_cond) != 3) {
        pp_bmc_log_error("%s: '%s' failed: %s (%s)", ___F, id,
			 strerror(errno), pp_strstream_buf(&err));
	goto bail;
    }
    
    o = malloc(sizeof(pp_max7312_gpio_dev_t));
    if (pp_max7312_gpio_dev_init(o, i2cdev, i2caddr, pres_cond) == PP_ERR) {
        pp_bmc_log_error("%s initialization failed", id);
	free(o); o = NULL;
        goto bail;	
    }

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)o;
}
