/**
 * msi_fan_ctrl.c
 *
 * Implements fan speed control for MSI servers.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/tp_ipmi_sens.h>
#include "drivers/msi_fan_ctrl.h"


#define SYS_TEMP    1
#define CPU_TEMP    2

/* internal prototypes */
static void sys_temp_recv_reading(pp_tp_sensdev_subscriber_t*, pp_tp_sensdev_t*, int);
static void cpu_temp_recv_reading(pp_tp_sensdev_subscriber_t*, pp_tp_sensdev_t*, int);
static void update_pwm(pp_msi_fan_ctrl_t* this);


pp_tp_obj_t* pp_msi_fan_ctrl_ctor(const char* id, vector_t* args)
{
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;  

    pp_msi_fan_ctrl_t*  fc = NULL;
    pp_tp_sensdev_t*    sys_temp_sens = NULL;
    pp_tp_sensdev_t*    cpu_temp_sens = NULL;
    pp_tp_pwm_act_t*    pwm_act = NULL;
    vector_t *          pwm_actors = NULL;
    int i;

    if (pp_tp_arg_scanf(args, 0, &err, "o<s>o<s>", &sys_temp_sens,
			&cpu_temp_sens) != 2) {
        pp_bmc_log_error("[msiFanCtrl] invalid ctor params: %s",
			 pp_strstream_buf(&err));
        goto bail;
    }
    
    pwm_actors = vector_new(NULL, 2, (vector_elem_del_func_simple)pp_tp_obj_release);
    // Note: vector is deleted in object destructor
    i = 2;
    while (pp_tp_arg_scanf(args, i, &err, "o<ap>", &pwm_act) == 1) {
        vector_add(pwm_actors, pwm_act);
        i++;
    }
    if (i == 0) {
        pp_bmc_log_warn("[msiFanCtrl] no fan actors registered");
    }

    fc = malloc(sizeof(pp_msi_fan_ctrl_t));
    if (pp_msi_fan_ctrl_init(fc, PP_TP_FAN_CTRL, id,
                             pp_msi_fan_ctrl_dtor,
                             sys_temp_sens, cpu_temp_sens, pwm_actors))
    {
        pp_bmc_log_error("[msiFanCtrl] initialization failed");
        free(fc);
        fc = NULL;
    }

    if (fc != NULL) {
        pp_bmc_log_info("[msiFanCtrl] initialized with %d PWMs",
			vector_size(pwm_actors));
    }
    
 bail:
    pp_strstream_free(&err);

    return (pp_tp_obj_t*)fc;
}

void pp_msi_fan_ctrl_dtor(pp_tp_obj_t* this)
{
    pp_msi_fan_ctrl_cleanup((pp_msi_fan_ctrl_t*)this);
    free(this);
}

int pp_msi_fan_ctrl_init(pp_msi_fan_ctrl_t* this,
                     pp_tp_obj_type_t type, const char* id,
                     pp_tp_obj_dtor_func_t dtor,
                     pp_tp_sensdev_t* sys_temp_sens,
			 pp_tp_sensdev_t* cpu_temp_sens, 
                     vector_t* pwm_actors)
{
    unsigned int i;
    
    if (pp_tp_ctrl_init(&this->base, type, id, dtor)) {
        return PP_ERR;
    }
    
    // duplicate
    this->sys_temp_sensor = pp_tp_sensdev_duplicate(sys_temp_sens);
    this->cpu_temp_sensor = pp_tp_sensdev_duplicate(cpu_temp_sens);
    this->pwm_actors = pwm_actors;
    for (i=0; i<vector_size(pwm_actors); i++) {
        pp_tp_pwm_act_duplicate(vector_get(pwm_actors, i));
    }

    // subscribe
    this->sys_temp_subscriber.recv_reading = sys_temp_recv_reading;
    pp_bmc_tp_sensdev_subscribe(sys_temp_sens, &this->sys_temp_subscriber);

    this->cpu_temp_subscriber.recv_reading = cpu_temp_recv_reading;
    pp_bmc_tp_sensdev_subscribe(cpu_temp_sens, &this->cpu_temp_subscriber);

    this->cpu_pwm_fn = 1;
    this->cpu_tmp = -1;

    return PP_SUC;
}

void pp_msi_fan_ctrl_cleanup(pp_msi_fan_ctrl_t* this)
{
    // unsubscribe
    pp_bmc_tp_sensdev_unsubscribe(this->sys_temp_sensor,
				  &this->sys_temp_subscriber);
    pp_bmc_tp_sensdev_unsubscribe(this->cpu_temp_sensor,
				  &this->cpu_temp_subscriber);

    // release
    pp_tp_sensdev_release(this->sys_temp_sensor);
    pp_tp_sensdev_release(this->cpu_temp_sensor);
    vector_delete(this->pwm_actors);     // members are released by vector->free handler
    
    pp_tp_ctrl_cleanup(&this->base);
}

static void
sys_temp_recv_reading(pp_tp_sensdev_subscriber_t* subs, 
		      pp_tp_sensdev_t* source UNUSED, int reading) {

    pp_msi_fan_ctrl_t* this =
	PP_TP_INTF_2_OBJ_CAST(subs, pp_msi_fan_ctrl_t, sys_temp_subscriber);
    
    if (reading < 0) {
	// error reading, do not update pwm
    } else if (reading < 25) {
	if (this->cpu_pwm_fn != 1) {
	    this->cpu_pwm_fn = 1;
	    update_pwm(this);
	}
    } else {
	if (this->cpu_pwm_fn != 2) {
	    this->cpu_pwm_fn = 2;
	    update_pwm(this);
	}
    }
}

static void
cpu_temp_recv_reading(pp_tp_sensdev_subscriber_t* subs, 
		      pp_tp_sensdev_t* source UNUSED, int reading) {
    pp_msi_fan_ctrl_t* this =
	PP_TP_INTF_2_OBJ_CAST(subs, pp_msi_fan_ctrl_t, cpu_temp_subscriber);
    this->cpu_tmp = reading;
    update_pwm(this);
}

static void update_pwm(pp_msi_fan_ctrl_t* this) {
    // y = f(x) = ((amult*x) / adiv) + b
    int amult;
    int adiv;
    int b;
    int ymin;
    int ymax;
    
    int pwm_percent;
    int abs_pwm;
    unsigned int i;
    
    pp_tp_pwm_act_t * pwm_act;
    
    //pp_bmc_log_info("[msiFanCtrl] updating pwm, using fn=%d, reading=%d", this->cpu_pwm_fn, this->cpu_tmp);
    
    if (this->cpu_tmp >= 0) {
        // regular reading, apply temp/pwm function
        if (this->cpu_pwm_fn == 1) {
            amult = 119;
            adiv = 100;
            b = -41;
            ymin = 10;
            ymax = 55;
        } else {
            amult = 333;
            adiv = 100;
            b = -169;
            ymin = 13;
            ymax = 100;
        }
    
        // calculate pwm_percent, abs pwm
        pwm_percent = ((amult * this->cpu_tmp) / adiv) + b;
        if (pwm_percent < ymin) pwm_percent = ymin;
        if (pwm_percent > ymax) pwm_percent = ymax;
    
        abs_pwm = ((pwm_percent * 255) / 100);
        if (abs_pwm < 0) abs_pwm = 0;
        if (abs_pwm > 255) abs_pwm = 255;

        pp_bmc_log_debug("[msiFanCtrl] setting new duty cycle %x (%d)",
			 abs_pwm, pwm_percent);
        for (i = 0; i < vector_size(this->pwm_actors); i++) {
            pwm_act = (pp_tp_pwm_act_t*)vector_get(this->pwm_actors, i);
            pp_tp_pwm_set_duty_cycle(pwm_act, abs_pwm);
        }
    }
}
