/**
 * pp_pca9556_gpio_dev.c
 *
 * Device driver class for the Philips PCA9556 I2C GPIO Extender
 *                 and for the Philips PCA9554 I2C GPIO Extender
 *                 (PCA9554 is logically identical to the PCA9556,
 *                 but works with different voltage levels)
 * 
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 */

#include <unistd.h>
#include <fcntl.h>

#include <pp/base.h>
#include <pp/selector.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>

#include "pca9556_gpio_dev.h"

#define GPIO_VALID_MASK	0x000000FF

static int
pca9556_get_val(pp_pca955x_gpio_dev_t* d)
{
    int value = -1;
    u_char data;
    
    if (PP_FAILED(d->i2cdev->rx_byte_data(d->i2cdev, d->i2caddr,
					  REG_INPUT, &data))) {
	goto bail;
    }
    value = data;
    value &= GPIO_VALID_MASK;

 bail:
    return value;
}

static int
pca9556_set_reg(pp_pca955x_gpio_dev_t* d, u_char reg, u_long mask, u_long val)
{
    int ret = PP_ERR;
    u_char data;

    mask &= GPIO_VALID_MASK;
    val  &= GPIO_VALID_MASK;
    
    if (PP_FAILED(d->i2cdev->rx_byte_data(d->i2cdev, d->i2caddr,
					  reg, &data))) {

	goto bail;
    }

    data &= ~mask;
    data |= val & mask;
    
    if (PP_FAILED(d->i2cdev->tx_byte_data(d->i2cdev, d->i2caddr,
					  reg, data))) {

	goto bail;
    }
    
    ret = PP_SUC;
    
 bail:
    return ret;   
}

int pp_pca9556_gpio_dev_init(pp_pca9556_gpio_dev_t* d, const char* id,
			     pp_tp_i2c_comdev_t* i2cdev,
			     u_char i2caddr, pp_tp_cond_t* pres_cond) {
      
    pp_pca955x_gpio_dev_init((pp_pca955x_gpio_dev_t*)d, PP_PCA955X_GPIO_DEV,
			     id, i2cdev, i2caddr, GPIO_VALID_MASK,
			     pca9556_get_val, pca9556_set_reg, pres_cond);
			     
    return PP_SUC;
}


pp_tp_obj_t* pp_pca9556_gpio_dev_ctor(const char* id, vector_t* args) {
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;  
    pp_pca9556_gpio_dev_t* o = NULL;
    pp_tp_cond_t* pres_cond;
    pp_tp_i2c_comdev_t* i2cdev;
    u_char i2caddr;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>d<c>|o<sc>",
			&i2cdev, &i2caddr, &pres_cond) != 3) {
        pp_bmc_log_error("%s: '%s' failed: %s (%s)", ___F, id,
			 strerror(errno), pp_strstream_buf(&err));
	goto bail;
    }
    
    o = malloc(sizeof(pp_pca9556_gpio_dev_t));
    if (pp_pca9556_gpio_dev_init(o, id, i2cdev, i2caddr, pres_cond)
	== PP_ERR) {
        pp_bmc_log_error("%s initialization failed", id);
	free(o); o = NULL;
        goto bail;	
    }

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)o;
}
