/**
 * pp_pca955x_gpio_dev.c
 *
 * Device driver base class for the Philips PCA955x family
 * of I2C GPIO Extenders
 * 
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 */

#include <unistd.h>
#include <fcntl.h>

#include <pp/base.h>
#include <pp/selector.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>

#include "pca955x_gpio_dev.h"

static int
pp_pca955x_gpio_get_val(pp_tp_gpio_dev_t* s, u_long mask) {
    int value = -1;
    pp_pca955x_gpio_dev_t *dev = (pp_pca955x_gpio_dev_t*) s;
    assert((mask & ~s->valid_mask) == 0);

    if (dev->pres_cond && !pp_bmc_tp_cond_is_true(dev->pres_cond)) {
	errno = ENODEV; /* device not available */
	goto bail;
    }

    if ((value = dev->value) >= 0) {
	value &= mask;
    }
 bail:
    return value;
}

static int
pp_pca955x_gpio_set_val_internal(pp_pca955x_gpio_dev_t *dev)
{
    int ret = PP_ERR;

    if (PP_FAILED(pp_tp_i2c_comdev_pre_com(dev->i2cdev, dev->i2caddr))) {
	goto bail;
    }
    
    /* set output value for !tristate pins */
    if (PP_FAILED(dev->set_reg(dev, REG_OUTPUT, dev->output_valid_mask, dev->output_driven_state))) {
	goto bail_post;
    }

    /* handle tristate */
    if (PP_FAILED(dev->set_reg(dev, REG_CONFIG, dev->output_valid_mask, dev->output_tri_state))) {
	goto bail_post;
    }
    
    ret = PP_SUC;

 bail_post:
    pp_tp_i2c_comdev_post_com(dev->i2cdev, dev->i2caddr);
    
 bail:
    return ret;
}

static int
pp_pca955x_gpio_set_val(pp_tp_gpio_dev_t* d, u_long mask, u_long val,
			u_long tristate)
{
    pp_pca955x_gpio_dev_t *dev = (pp_pca955x_gpio_dev_t*) d;

    /* first determine the output states and remeber it for possible later use */
    dev->output_driven_state &= ~(mask & ~tristate);
    dev->output_driven_state |= mask & ~tristate & val;
    dev->output_tri_state &= ~mask;
    dev->output_tri_state |= mask & tristate;
    dev->output_valid_mask |= mask;

    // check whether access allowed per presence condition
    if (dev->pres_cond == NULL || pp_bmc_tp_cond_is_true(dev->pres_cond)) {
	return pp_pca955x_gpio_set_val_internal(dev);
    }

    return PP_ERR;
}

/** Call to (re)read the sensor and notify subscribers */
static void read_sensor(pp_pca955x_gpio_dev_t* d)
{
    int new_value = -1;
    int old_value = d->value;
    unsigned long change_mask;

    // check whether access allowed per presence condition
    if (d->pres_cond != NULL && !pp_bmc_tp_cond_is_true(d->pres_cond)) {
	goto bail;
    }
    
    if (PP_FAILED(pp_tp_i2c_comdev_pre_com(d->i2cdev, d->i2caddr))) {
	goto bail;
    }

    new_value = d->get_val(d);
    
    pp_tp_i2c_comdev_post_com(d->i2cdev, d->i2caddr);

 bail:
    if (new_value != old_value) {
	d->value = new_value;
	if (new_value < 0) change_mask = 0;
	else change_mask = old_value < 0 ? d->base.valid_mask : 
		(unsigned long)(old_value ^ new_value);
	pp_tp_gpio_dev_notify_subscribers(&d->base, (unsigned long)new_value, change_mask, new_value < 0 ? 0 : 1);
    }
}

/** Currently we poll the gpio device FIXME: replace by interrupt */
static int read_sensor_to(const int item_id UNUSED, void* ctx) {
    read_sensor((pp_pca955x_gpio_dev_t*)ctx);
    return PP_SUC;
}

static void
pres_cond_recv_reading(pp_tp_sensdev_subscriber_t* subscriber, 
		       pp_tp_sensdev_t* source UNUSED, int reading)
{
    pp_pca955x_gpio_dev_t* dev;

    dev = PP_TP_INTF_2_OBJ_CAST(subscriber, pp_pca955x_gpio_dev_t, pres_cond_subs);
    if (reading > 0) {
	/* chip is available now: output state values */
	pp_pca955x_gpio_set_val_internal(dev);
    }
}

void pp_pca955x_gpio_dev_cleanup(pp_tp_obj_t* d) {
    pp_pca955x_gpio_dev_t *this = (pp_pca955x_gpio_dev_t*)d;
    
    pp_bmc_log_debug("[%s] cleanup", d->id);
    
    if (this->pres_cond) {
	pp_tp_cond_unsubscribe(this->pres_cond, &this->pres_cond_subs);
    }

    pp_select_remove_to(this->to_hndl);
    
    if (this->pres_cond) pp_tp_cond_release(this->pres_cond);
    pp_tp_obj_release((pp_tp_obj_t*)(this->i2cdev));
    pp_tp_gpio_dev_cleanup((pp_tp_gpio_dev_t*)this);
}

int pp_pca955x_gpio_dev_init(pp_pca955x_gpio_dev_t* d,
			     pp_tp_obj_type_t type, const char* id,
			     pp_tp_i2c_comdev_t* i2cdev, u_char i2caddr,
			     u_long valid_mask,
			     int (*get_val)(pp_pca955x_gpio_dev_t* d),
			     int (*set_reg)(pp_pca955x_gpio_dev_t* d,
					    u_char reg, u_long mask,
					    u_long val),
			     pp_tp_cond_t* pres_cond)
{
    pp_bmc_log_debug("[%s] init", id);
      
    pp_tp_gpio_dev_init((pp_tp_gpio_dev_t*)d,
			type, id,
                        valid_mask,
			pp_pca955x_gpio_dev_cleanup,
			NULL, /* no init necessary */
			pp_pca955x_gpio_get_val,
			pp_pca955x_gpio_set_val,			
                        NULL);

    d->i2cdev = (pp_tp_i2c_comdev_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)i2cdev);
    d->i2caddr = i2caddr;
    d->pres_cond = pres_cond == NULL ? NULL : pp_tp_cond_duplicate(pres_cond);
    d->set_reg = set_reg;
    d->get_val = get_val;
    d->value = -1;
    d->to_hndl = pp_select_add_to(500, 1, read_sensor_to, d);

    d->output_valid_mask = 0; /* invalidate saved state */

    /*
     * NOTE: If someone sets a GPIO when the chip is not powered up yet we
     *       have to remember the value and delay the GPIO operation until
     *       the chip is alive. We do this using the init-condition of the
     *       i2c-chip we are bound to. We assume that the callback is
     *       called after the power-up callback that is registered on the
     *       same condition!
     */

    /* register for init condition to call init code */
    if (d->pres_cond) {
	d->pres_cond_subs.recv_reading = pres_cond_recv_reading;
	pp_tp_cond_subscribe(d->pres_cond, &d->pres_cond_subs);
    }
    
    return PP_SUC;
}
