/**
 * pp_gpio_disc_sens.c
 *
 * This is a gpio sensor that produces IPMI conformant readings
 * for discrete sensors instead of simple asserted/deasserted
 * readings. In contrast to generic gpio sensors, this sensor
 * is designed to be exported through an IPMI_SENS.
 *
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 */

#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_cmd_sensor.h>
#include <pp/bmc/tp_gpio_sens.h>
#include <pp/bmc/debug.h>

#include <pp/bmc/ipmi_sdr.h>
#include "pp_gpio_disc_sens.h"

static ipmi_sdr_header_t* gpio_disc_default_sdr(pp_tp_sensdev_t* s);
static void gpio_sens_recv_reading(pp_tp_sensdev_subscriber_t* o, pp_tp_sensdev_t* source, int reading);

pp_tp_obj_t*
pp_gpio_disc_sens_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[DiscGPIO] %s: '%s' failed (%s)";
    pp_gpio_disc_sens_t* gpio_disc_sens = NULL;
    pp_tp_gpio_sens_t* gpio_sens;
    u_char sens_type = IPMI_SENSOR_TYPE_BUTTON;
    u_char reading_type = IPMI_EVENT_READING_TYPE_DISCRETE_STATE;
    u_short state_asserted;
    u_short state_deasserted;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;

    if (pp_tp_arg_scanf(args, 0, &err, "o<sg>d<c>d<c>d<s>d<s>",
			&gpio_sens, &sens_type, &reading_type,
			&state_asserted, &state_deasserted)
	!= 5) {
	pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	goto bail;
    }
    
    gpio_disc_sens = malloc(sizeof(pp_gpio_disc_sens_t));
    if (PP_FAILED(pp_gpio_disc_sens_init(gpio_disc_sens, PP_TP_SENS_DEV, id,
					 pp_gpio_disc_sens_dtor,
					 gpio_disc_default_sdr,
					 gpio_sens,
					 sens_type, reading_type,
					 state_asserted, state_deasserted))) {
	pp_bmc_log_error(errmsg, ___F, id, "init");
	free(gpio_disc_sens);
	gpio_disc_sens = NULL;
    }

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)gpio_disc_sens;
}

void
pp_gpio_disc_sens_dtor(pp_tp_obj_t* this)
{
    pp_gpio_disc_sens_cleanup((pp_gpio_disc_sens_t *)this);
    free(this);
}

int
pp_gpio_disc_sens_init(pp_gpio_disc_sens_t* this, pp_tp_obj_type_t type,
		       const char* id, pp_tp_obj_dtor_func_t dtor,
		       pp_tp_sensdev_default_sdr_func_t default_sdr,
		       pp_tp_gpio_sens_t* gpio_sens,
		       u_char sens_type, u_char reading_type,
		       u_short state_asserted, u_short state_deasserted)
{
    pp_bmc_log_debug("[DiscGPIO] init '%s', gpio sens '%s', type 0x%02x, "
		     "reading 0x%02x", id, ((pp_tp_obj_t*)gpio_sens)->id,
		     sens_type, reading_type);

    if (PP_SUCCED(pp_tp_sensdev_init(&this->base, type, id, dtor,
				     default_sdr))) {
	this->gpio_sens = pp_tp_gpio_sens_duplicate(gpio_sens);
	this->sens_type = sens_type;
	this->reading_type = reading_type;
	this->state_asserted = state_asserted;
	this->state_deasserted = state_deasserted;
	this->gpio_subscriber.recv_reading = gpio_sens_recv_reading;
	pp_tp_gpio_sens_subscribe(this->gpio_sens, &this->gpio_subscriber);
	return PP_SUC;
    }
    return PP_ERR;
}

void
pp_gpio_disc_sens_cleanup(pp_gpio_disc_sens_t* this)
{
    pp_tp_gpio_sens_unsubscribe(this->gpio_sens, &this->gpio_subscriber);
    pp_tp_gpio_sens_release(this->gpio_sens);
    pp_tp_sensdev_cleanup(&this->base);
}

static ipmi_sdr_header_t*
gpio_disc_default_sdr(pp_tp_sensdev_t* s)
{
    pp_gpio_disc_sens_t * this = (pp_gpio_disc_sens_t*)s;
    ipmi_sdr_header_t* mysdr;

#ifdef PRODUCT_SMIDC
    // Supermicro's IPMIview only supports full sensor records!
    mysdr = (ipmi_sdr_header_t*)ipmi_sdr_full_part_create(
            this->sens_type, this->reading_type, 0, 0, 0, 0, 0, 0, 0, 0);
#else
    mysdr = (ipmi_sdr_header_t*)ipmi_sdr_compact_part_create(this->sens_type, this->reading_type);
#endif
    return mysdr;
}

static void gpio_sens_recv_reading(pp_tp_sensdev_subscriber_t* o, 
				   pp_tp_sensdev_t* source UNUSED, int reading)
{
    pp_gpio_disc_sens_t *this =
	PP_TP_INTF_2_OBJ_CAST(o, pp_gpio_disc_sens_t, gpio_subscriber);
    int new_reading = 0;

    if (reading >= 0) { // no error
	new_reading = reading ? this->state_asserted : this->state_deasserted;
    } else {
	new_reading = reading;
    }

    if (new_reading != this->base.reading) {
	this->base.reading = new_reading;
	pp_bmc_tp_sensdev_notify_subscribers(&this->base, new_reading);	
    }
}
