/**
 * pp_powersupply_sens.c
 *
 * Discrete sensor for power supply monitoring. Combines several gpio
 * readings in one IPMI-conformant discrete 'power supply' reading
 * (see table 42.2) Currently made for the AMD Warthog board,
 * to be extended when necessary.
 * 
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 */

#include <pp/bmc/bmc_config.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_sdr.h>

#include "pp_powersupply_sens.h"

static void powersupply_sens_dtor(pp_tp_obj_t* sens);
static void powersupply_sens_recv_reading(pp_tp_sensdev_subscriber_t*, pp_tp_sensdev_t*, int);

static ipmi_sdr_header_t*
powersupply_sens_default_sdr(struct pp_tp_sensdev_s* s UNUSED)
{
    ipmi_sdr_compact_sensor_t* sdr;

    sdr = ipmi_sdr_compact_part_create(IPMI_SENSOR_TYPE_POWER_SUPPLY,
			       IPMI_EVENT_READING_TYPE_SENSOR_SPECIFIC);
	
    return (ipmi_sdr_header_t*)sdr;      
}

static void
build_reading_and_notify(pp_powersupply_sens_t* this)
{
    int new_reading = 0;
    int read_presence, read_fault, read_input;

    /* TODO(miba): use ACPI state information to check for fault only in S0 state */

    read_presence = pp_tp_gpio_sens_get_reading(this->gpio_presence);
    read_fault = pp_tp_gpio_sens_get_reading(this->gpio_fault);
    read_input = pp_tp_gpio_sens_get_reading(this->gpio_input);

    if (read_presence < 0) {
	new_reading = -1;
    } else if (!read_presence) {
	new_reading |= (0x01 <<
			IPMI_SENSOR_STATE_POWER_SUPPLY_PRESENCE_DETECTED);

	/* check for fault and AC state only when present */
	if (read_fault >= 0 && !read_fault) {
	    new_reading |= (0x01 <<
			    IPMI_SENSOR_STATE_POWER_SUPPLY_FAILURE_DETECTED);
	}
	if (read_input >= 0 && !read_input) {
	    new_reading |= (0x01 << IPMI_SENSOR_STATE_POWER_SUPPLY_INPUT_LOST);
	}
    }

    if (new_reading != this->base.reading) {
	this->base.reading = new_reading;
	pp_bmc_tp_sensdev_notify_subscribers(&this->base, new_reading);
    }
}

static void
powersupply_sens_recv_reading(pp_tp_sensdev_subscriber_t* o, 
			      pp_tp_sensdev_t* source UNUSED, int reading UNUSED)
{
    pp_powersupply_sens_t *this =
	PP_TP_INTF_2_OBJ_CAST(o, pp_powersupply_sens_t, sens_subscr);
    build_reading_and_notify(this);
}

void
pp_powersupply_sens_init(pp_powersupply_sens_t* this,
			 const char *id,
			 pp_tp_gpio_sens_t* gpio_presence,
			 pp_tp_gpio_sens_t* gpio_fault,
			 pp_tp_gpio_sens_t* gpio_input)
{
    pp_tp_sensdev_init(&this->base,  PP_TP_SENS_DEV, id,
		       powersupply_sens_dtor, powersupply_sens_default_sdr);
   
    this->sens_subscr.recv_reading = powersupply_sens_recv_reading;

    this->gpio_presence = pp_tp_gpio_sens_duplicate(gpio_presence);
    this->gpio_fault = pp_tp_gpio_sens_duplicate(gpio_fault);
    this->gpio_input = pp_tp_gpio_sens_duplicate(gpio_input);
    
    pp_tp_gpio_sens_subscribe(this->gpio_presence, &this->sens_subscr);
    pp_tp_gpio_sens_subscribe(this->gpio_fault, &this->sens_subscr);
    pp_tp_gpio_sens_subscribe(this->gpio_input, &this->sens_subscr);
}

pp_tp_obj_t*
pp_powersupply_sens_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[PowersupplySens] %s c'tor: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_powersupply_sens_t* sens;
    pp_tp_gpio_sens_t* gpio_presence;
    pp_tp_gpio_sens_t* gpio_fault;
    pp_tp_gpio_sens_t* gpio_input;
    
    if (pp_tp_arg_scanf(args, 0, &err, "o<sg>o<sg>o<sg>",
			&gpio_presence, &gpio_fault, &gpio_input) != 3) {
	pp_bmc_log_perror(errmsg, id, pp_strstream_buf(&err));
	return NULL;
    }

    sens = malloc(sizeof(pp_powersupply_sens_t));
    
    pp_powersupply_sens_init(sens, id,
			     gpio_presence, gpio_fault, gpio_input);
    
    pp_strstream_free(&err);
    
    return (pp_tp_obj_t*)sens;        
}

static void powersupply_sens_dtor(pp_tp_obj_t* o) {
    pp_powersupply_sens_t* this = (pp_powersupply_sens_t*)o;
    if (this != NULL) {
	pp_tp_gpio_sens_unsubscribe(this->gpio_presence, &this->sens_subscr);
	pp_tp_gpio_sens_unsubscribe(this->gpio_fault, &this->sens_subscr);
	pp_tp_gpio_sens_unsubscribe(this->gpio_input, &this->sens_subscr);
	pp_tp_gpio_sens_release(this->gpio_presence);
	pp_tp_gpio_sens_release(this->gpio_fault);
	pp_tp_gpio_sens_release(this->gpio_input);
	
	pp_tp_sensdev_cleanup(&this->base);
	free(this);
    }
}

