/**
 * rc_cy8c26.h
 *
 * This implements a special communication object to RS485
 * connected cypress relay controller. The communication object
 * will shadow registers of cypress, so that it will be possible
 * to execute bulk operations for polling all register values.
 *
 * (c) 2006 Peppercon AG, 2006/09/05 tbr@raritan.com
 */

#include <pp/base.h>
#include <pp/selector.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_rs485_comdev.h>
#include <pp/bmc/tp_rs485_chip.h>
#include <pp/bmc/tp_scan_sensdev.h>
#include <pp/bmc/tp_gpio_dev.h>
#include <pp/bmc/tp_gpio_act.h>
#include "sensor_scanner.h"
#include "rc_cy8c26.h"

/*
 * The cy8c26 chip, beeing a rs485 chip
 ****************************************************/

/*
 * object structures
 * -----------------
 */
typedef struct rc_cy8c26_chip_s {
    pp_tp_rs485_chip_t base;
    unsigned int nr_outlets;
    int reg_initialized; /* whether register cache was filled */
    unsigned char reg[RC_CY8C26_REGISTER_CNT]; /* register cache */
    unsigned char notify[RC_CY8C26_REGISTER_CNT]; /* notify vector */
    unsigned int notify_sz; /* size of notify vector */
    /* regarding scannable, refer to docu in tp_scan_sensdev.h */
    pp_sensor_scannable_t scannable;
    pthread_mutex_t mtx;
    int update_sf_id;
    int setbyte_to_id;
    int setbyte_sf_id;
    char setbyte_chg[RC_CY8C26_REGISTER_CNT];
} rc_cy8c26_chip_t;

#define PP_ERR_IMAGE -2
#if (PP_ERR_IMAGE >= PP_ERR)
#  error PP_ERR_IMAGE conflicts with PP_ERR
#endif

#define CY8_BMS_BCOK   (1 << 0)
#define CY8_BMS_IVERR  (1 << 1)
#define CY8_BMS_FCERR  (1 << 2)
#define CY8_BMS_FPERR  (1 << 3)
#define CY8_BMS_CCERR  (1 << 4)
#define CY8_BMS_BM     (1 << 5)
#define CY8_BMS_IVKERR (1 << 6)
#define CY8_BMS_IVCERR (1 << 7)

#define CY8_RS_FWUPD_INIT_TO              20   /*ms*/
#define CY8_RS_FWUPD_BLOCK_TO             100  /*ms*/
#define CY8_RS_FWUPD_VERIFY_TO            1000 /*ms*/
#define CY8_RS_FWUPD_SWT_BOOTLOADER_DELAY 50   /*ms*/

#define SETBYTE_CHG_SEND 0x01
#define SETBYTE_CHG_NOTI 0x10
#define SETBYTE_CHG_MARK (SETBYTE_CHG_SEND | SETBYTE_CHG_NOTI);
#define SETBYTE_COLLECT_TO                20   /*ms*/

/*
 * static methods and definitions
 * --------------------------------
 */
static void rc_cy8c26_chip_dtor(pp_tp_obj_t* o);
static int rc_cy8c26_get_byte(pp_tp_rs485_chip_t* this, unsigned char addr,
			      unsigned char* val);
static int rc_cy8c26_set_byte(pp_tp_rs485_chip_t* this, unsigned char addr,
			      unsigned char val);
static int rc_cy8c26_fwupdate(pp_tp_rs485_chip_t* this,
			      const char* data, size_t len);
static int rc_cy8c26_fwupd_write_block(pp_tp_rs485_chip_t* this,
				       const char* data, size_t len,
				       int timeout);
static void rc_cy8c26_update(pp_sensor_scannable_t* s);
static int rc_cy8c26_notify(void* ctx);
static int rc_cy8c26_setbyte_sf_hndl(void* ctx);
static int rc_cy8c26_setbyte_to_hndl(int item_id, void* ctx);

static inline int rc_cy8c26_write_fwupd(pp_tp_rs485_chip_t* this,
					const char* data,
					unsigned char len, int timeout) {
    return pp_tp_rs485_comdev_write_fwupd(this->rs485dev, this->rs485addr,
					  data, len, timeout);
}

/*
 * public & private methods
 * -----------------------------
 */
pp_tp_obj_t* pp_rc_cy8c26_chip_ctor(const char* id, vector_t* args) {
    const char* fn = "[CY8C26] ctor";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    rc_cy8c26_chip_t* this = NULL;
    pp_tp_rs485_comdev_t* rs485dev;
    unsigned char rs485addr;
    unsigned char nr_outlets;

    if (pp_tp_arg_scanf(args, 0, &err, "o<4>d<c>d<c>",
			&rs485dev, &rs485addr, &nr_outlets) != 3) {
	pp_bmc_log_perror("%s(): '%s' failed: %s", fn, id,
			  pp_strstream_buf(&err));
    } else if (nr_outlets < 3 || nr_outlets > 5) {
	/* we may implement a live check here, but luxury for the time beeing*/
	errno = EINVAL;
	pp_bmc_log_error("%s(): '%s' failed: no of outlets (%d) invalid ",
			 fn, id, nr_outlets);
    } else {
	this = malloc(sizeof(rc_cy8c26_chip_t));
	pp_tp_rs485_chip_init(&this->base, PP_TP_RS485_CHIP, id,
			      rc_cy8c26_chip_dtor,
			      RC_CY8C29_MODEL_STRING,
			      rs485dev, rs485addr, 
			      rc_cy8c26_get_byte, rc_cy8c26_set_byte,
			      rc_cy8c26_fwupdate);
	this->nr_outlets = nr_outlets;
	this->reg_initialized = 0;
	memset(this->reg, 0, sizeof(this->reg));
	memset(this->setbyte_chg, 0, sizeof(this->setbyte_chg));
	this->setbyte_to_id = 0;
	this->setbyte_sf_id = 0;
	MUTEX_CREATE_RECURSIVE(&this->mtx);
	this->update_sf_id = 0;
	this->scannable.update = rc_cy8c26_update;
	pp_sensor_scanner_add(&this->scannable, 0);
	pp_bmc_log_debug("%s(): '%s' initialized at %s:%hhd", fn, id,
			 pp_tp_rs485_comdev_to_string(rs485dev), rs485addr);

    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void rc_cy8c26_chip_dtor(pp_tp_obj_t* o) {
    rc_cy8c26_chip_t* this = (rc_cy8c26_chip_t*)o;
    assert(this != NULL);
    // TODO: cancel timers and scheduled functions
    pp_bmc_log_debug("[CY8C26] dtor: '%s' deinit at %s:%hhd",
		     pp_tp_rs485_chip_to_string(&this->base),
		     pp_tp_rs485_comdev_to_string(this->base.rs485dev),
		     this->base.rs485addr);
    pp_sensor_scanner_remove(&this->scannable);
    pp_tp_rs485_chip_cleanup(&this->base);
    free(this);
}

/*
 * virtual functions implementing interface of pp_tp_rs485_chip_t
 */
static int rc_cy8c26_get_byte(pp_tp_rs485_chip_t* o,
			      unsigned char addr, unsigned char* val) {
    int ret;
    rc_cy8c26_chip_t* this = (rc_cy8c26_chip_t*)o;
    unsigned char max_addr = RC_CY8C26_REG_MAX_PER_OUTLET(this->nr_outlets);
    
    if (addr > max_addr) {
	ret = PP_ERR;
    } else {
	*val = this->reg[addr];
	ret = PP_SUC;
    }
    return ret;
}

/*
 * TODO: currently there is a write back strategy per byte, i.e.
 *       we collect changes for a single register over time and write
 *       it back after SETBYTE_COLLECT_TO
 *       we may also think about collecting changes for multiple
 *       registers and write them back at once. however, this would
 *       require 'interrupted bulk' operations for write requests.
 */
static int rc_cy8c26_set_byte(pp_tp_rs485_chip_t* o,
			      unsigned char addr, unsigned char val) {
    rc_cy8c26_chip_t* this = (rc_cy8c26_chip_t*)o;

    pp_bmc_log_debug("%s(): %s(%#.2hhx): addr=%#.2hhx val=%#.2hhx",
		     ___F, pp_tp_rs485_chip_to_string(o), o->rs485addr,
		     addr, val);
    assert(addr < RC_CY8C26_REGISTER_CNT);
    
    MUTEX_LOCK(&this->mtx);
    if (this->reg[addr] != val) { /* only if something has really changed */
	this->reg[addr] = val;
	this->setbyte_chg[addr] = SETBYTE_CHG_MARK;
	if (this->setbyte_to_id == 0) {
	    this->setbyte_to_id = pp_select_add_to(SETBYTE_COLLECT_TO, 0,
						   rc_cy8c26_setbyte_to_hndl,
						   this);
	} else {
	    /* timeout pending, so lock is held by our thread, anyway */
	    MUTEX_UNLOCK(&this->mtx);
	}
	if (this->setbyte_sf_id == 0) {
	    this->setbyte_sf_id = pp_select_add_sf(rc_cy8c26_setbyte_sf_hndl,
						   this);
	}
    } else {
	MUTEX_UNLOCK(&this->mtx);
    }
    /* there will be no unlock here, unlock will be done by to
     * either to is already running or will be set                     */
    return PP_SUC; /* no error reporting, because of delayed execution */
}

static int rc_cy8c26_setbyte_sf_hndl(void* ctx) {
    rc_cy8c26_chip_t* this = (rc_cy8c26_chip_t*)ctx;
    unsigned char regcnt = RC_CY8C26_REG_CNT_PER_OUTLET(this->nr_outlets);
    int addr;

    /* reset sf_id here so that recursive changes will cause new sf */
    this->setbyte_sf_id = 0;

    /* check for all changes whether notify has happened already */
    for (addr = 0; addr < regcnt; ++addr) {
	if ((this->setbyte_chg[addr] & SETBYTE_CHG_NOTI) != 0) {
	    this->setbyte_chg[addr] &= ~SETBYTE_CHG_NOTI;
	    pp_tp_rs485_chip_notify_subscribers(&this->base, addr,
						this->reg[addr]);
	}
    }
    return PP_SUC;
}

static int rc_cy8c26_setbyte_to_hndl(int item_id UNUSED, void* ctx) {
    rc_cy8c26_chip_t* this = (rc_cy8c26_chip_t*)ctx;
    pp_tp_rs485_comdev_t* rs485dev = this->base.rs485dev;
    unsigned char rs485addr = this->base.rs485addr;
    unsigned char regcnt = RC_CY8C26_REG_CNT_PER_OUTLET(this->nr_outlets);
    int addr;

    /* lock already aquired by set_byte !!*/
    for (addr = 0; addr < regcnt; ++addr) {
	if ((this->setbyte_chg[addr] & SETBYTE_CHG_SEND) != 0) {
	    pp_bmc_log_debug("%s(): %s(%#.2hhx): addr=%#.2hhx val=%#.2hhx",
			     ___F, pp_tp_rs485_chip_to_string(&this->base),
			     rs485addr, addr, this->reg[addr]);
	    if (PP_ERR == pp_tp_rs485_comdev_write_data(rs485dev, rs485addr,
							addr, 1,
							&this->reg[addr])) {
		pp_bmc_log_perror("%s(): pp_tp_rs485_comdev_write_data(%s, "
				  "rs485addr=%#.2hhx, idx=%hhx, len=%hhd) "
				  "for '%s' failed",
				  ___F, pp_tp_rs485_comdev_to_string(rs485dev),
				  rs485addr, addr, 1,
				  pp_tp_rs485_chip_to_string(&this->base));
	    }
	    this->setbyte_chg[addr] &= ~SETBYTE_CHG_SEND;
	}
    }
    this->setbyte_to_id = 0;
    MUTEX_UNLOCK(&this->mtx);
    return PP_SUC;
}

/*
 * scanner callback updating register bank and issue notifications
 */
static void rc_cy8c26_update(pp_sensor_scannable_t* s) {
    const char* fn = ___F;
    rc_cy8c26_chip_t* this = PP_TP_INTF_2_OBJ_CAST(s, rc_cy8c26_chip_t,
						   scannable);
    pp_tp_rs485_comdev_t* rs485dev = this->base.rs485dev;
    unsigned char rs485addr = this->base.rs485addr;
    unsigned char regidx = 0;
    unsigned char regcnt = RC_CY8C26_REG_CNT_PER_OUTLET(this->nr_outlets);
    unsigned char i, newreg[RC_CY8C26_REGISTER_CNT]; /* temp register bank */
    int nsz, newcnt;

    /* fetch register bank from chip */
    assert(regcnt < RC_CY8C26_REGISTER_CNT);
    pp_bmc_log_debug("%s(): reg fetch from %s (%#.2hhx) of idx=%hhx len=%hhd",
		     fn, pp_tp_rs485_comdev_to_string(rs485dev), rs485addr,
		     regidx, regcnt);
    MUTEX_LOCK(&this->mtx);
    if (this->update_sf_id == 0) { /* don't do, if scanner is too fast */
	if (PP_ERR ==
	    (newcnt = pp_tp_rs485_comdev_read_data(rs485dev, rs485addr,
						   regidx, regcnt, newreg))) {
	    pp_bmc_log_perror("%s(): pp_tp_rs485_comdev_read_data(%s, "
			      "rs485addr=%#.2hhx, idx=%hhx, len=%hhd) for '%s' "
			      "failed",
			      fn, pp_tp_rs485_comdev_to_string(rs485dev),
			      rs485addr, regidx, regcnt,
			      pp_tp_rs485_chip_to_string(&this->base));
	} else {
	    if (newcnt < regcnt) {
		pp_bmc_log_warn("%s(): reg fetch from %s (%#.2hhx) was short "
				"%hhd, exp %hhd", fn,
				pp_tp_rs485_chip_to_string(&this->base),
				rs485addr, newcnt, regcnt);
	    }
	    /* loop over regs and check whether something has changed,
	       if yes, remember this in notify vector, that is going to be
	       used by scheduled function calling back subscribers */
	    if (!this->reg_initialized) { /* first time, notify always */
		for (i = 0; i < newcnt; ++i) this->notify[i] = i;
		nsz = i;
		this->reg_initialized = 1;
	    } else {                      /* all other times           */
		for (i = 0, nsz = 0; i < newcnt; ++i) {
		    if (newreg[i] != this->reg[i]) {
			this->notify[nsz++] = i;
		    }
		}
	    }
	    if (nsz > 0) { /* something has changed, schedule notify */
		memcpy(this->reg, newreg, sizeof(newreg));
		this->notify_sz = nsz;
		this->update_sf_id = pp_select_add_sf(rc_cy8c26_notify, this);
	    }
	}
    }
    MUTEX_UNLOCK(&this->mtx);
}

static int rc_cy8c26_notify(void* ctx) {
    rc_cy8c26_chip_t* this = (rc_cy8c26_chip_t*)ctx;
    unsigned int i;
    MUTEX_LOCK(&this->mtx);
    for (i = 0; i < this->notify_sz; ++i) {
	pp_tp_rs485_chip_notify_subscribers(&this->base, this->notify[i],
					    this->reg[this->notify[i]]);
    }
    this->notify_sz = 0;
    this->update_sf_id = 0;
    MUTEX_UNLOCK(&this->mtx);
    return 0;
}


static int rc_cy8c26_fwupdate(pp_tp_rs485_chip_t* this,
			      const char* data, size_t len) {
    const char* fn = ___F;
    char switch_fw_upload_cmd = 0x01;
    int idx, i;
    int bnum, retry_cnt = 0;
    int r, ret = PP_ERR;
    /* could we assert somehow, that BMC has really been stopped? */

    pp_bmc_log_info("%s: for cy8 at %#x", fn, this->rs485addr);
    
    /* check data len and number of blocks to flash */
    bnum = (len - 20) / 78;
    if (bnum == 0 || ((len - 20) % 78) != 0) {
	pp_bmc_log_error("%s: data len (%d) invalid", fn, len);
	return PP_ERR;
    }

    /* try to switch to bootloader mode, ignore error, *
     * if error, we assume we are already there        */
    pp_bmc_log_debug("%s: switch cy8 at %#x into bootloader mode",
		     fn, this->rs485addr);
    if (pp_tp_rs485_comdev_write_data(this->rs485dev, this->rs485addr,
				      RC_CY8C26_SWITCH_FW_UPLOAD, 1,
				      &switch_fw_upload_cmd) == PP_ERR) {
	pp_bmc_log_warn("%s: switch to bootloader failed, "
    			"assuming we are there already", fn);
    } else {
	/* wait until switch is complete */
	usleep(CY8_RS_FWUPD_SWT_BOOTLOADER_DELAY * 1000);
    }

    /* retry applies in case of image verify errors.                *
     * communication errors will be retried by underlying RS485-com */
    do {
	if (retry_cnt > 0) {
	    pp_bmc_log_info("%s: %d. retry for cy8 at %#x",
			    fn, retry_cnt, this->rs485addr);
	}
	
	/* send init sequence */
	idx = 0;
	if (PP_SUC != rc_cy8c26_fwupd_write_block(this, &data[idx], 10,
						  CY8_RS_FWUPD_INIT_TO)) {
	    pp_bmc_log_error("%s: failed to write init sequence", fn);
	    return PP_ERR;
	}
	idx += 10;

	/* send data blocks */
	pp_bmc_log_info("%s: writing data", fn);
	for (i = 0; i < bnum; ++i) {
	    if (PP_SUC != rc_cy8c26_fwupd_write_block(this, &data[idx], 78,
						      CY8_RS_FWUPD_BLOCK_TO)) {
		pp_bmc_log_error("%s: failed to write data block %d",
				 fn, i);
		return PP_ERR;
	    } else {
		idx += 78;
		if ((i % 8) == 0) {
		    fputc('.', stderr); fflush(stderr);
		}
	    }
	}
	fputc('\n', stderr), fflush(stderr);
	
	/* verify and exit sequence */
	pp_bmc_log_info("%s: verifying...", fn);
	if (PP_ERR == (r = rc_cy8c26_fwupd_write_block(this, &data[idx], 10,
						       CY8_RS_FWUPD_VERIFY_TO))) {
	    pp_bmc_log_error("%s: failed to write verify sequence", fn);
	    return PP_ERR;
	} else if (r == PP_ERR_IMAGE) {
	    pp_bmc_log_error("%s: verify of image failed!", fn);
	} else {
	    pp_bmc_log_info("%s: done for cy8 at %#x", fn, this->rs485addr);
	    ret = PP_SUC;
	}
    } while (ret == PP_ERR && retry_cnt++ < 3);
    
    return ret;
}

static int rc_cy8c26_fwupd_write_block(pp_tp_rs485_chip_t* this,
				       const char* data,
				       size_t len, int timeout) {
    const char* fn = ___F;
    int status, ret = PP_ERR;
    int do_retry, retry_cnt = 0;

    do {
	if ((status = rc_cy8c26_write_fwupd(this, data, len, timeout))
	    == PP_ERR) {
	    pp_bmc_log_perror("%s: RS485 write_fwupd to %#x failed",
			      fn, this->rs485addr);
	    return PP_ERR;
	}

        if ((status == CY8_BMS_BM) || (status == (CY8_BMS_BM | CY8_BMS_BCOK))){
	    /* successfully finished */
	    ret = PP_SUC;
            do_retry = 0;
        } else {
	    /* error occured */
	    if (status & CY8_BMS_CCERR) {
		pp_bmc_log_error("%s: CY8_BMS_CCERR, retrying\n", fn);
		do_retry = 1;
	    } else if (status & CY8_BMS_IVERR) {
		pp_bmc_log_error("%s: CY8_BMS_IVERR\n", fn);
		ret = PP_ERR_IMAGE;
		do_retry = 0;
	    } else {
		pp_bmc_log_error("%s: failed with status: %2x\n", fn, status);
		do_retry = 0;
	    }
	}
    } while (do_retry && retry_cnt++ < 3);
    
    return ret;
}


/*
 * sensor objets for cy8c26 chip
 ****************************************************/
typedef struct rc_cy8c26_sens_type_cfg_s {
    unsigned char sens_type;
    unsigned char reading_type;
    unsigned short analog_type;
    unsigned short unit_type;
    char sdr_rexp;
    void (*update)(pp_tp_reg_subscriber_t*, unsigned char, unsigned char);
} rc_cy8c26_sens_type_cfg_t;

typedef struct rc_cy8c26_sens_cfg_s {
    unsigned char reg;
    rc_cy8c26_sens_type_cfg_t* type;
    unsigned short sdr_mtol;
    unsigned short sdr_bacc;	
    unsigned short sdr_bexp;
} rc_cy8c26_sens_cfg_t;

typedef struct rc_cy8c26_sens_s {
    pp_tp_sensdev_t base;
    pp_tp_rs485_chip_t* chip;
    pp_tp_reg_subscriber_t csubs;
    rc_cy8c26_sens_cfg_t* cfg;
} rc_cy8c26_sens_t;

static void rc_cy8c26_sens_destroy(pp_tp_obj_t* o);
static ipmi_sdr_header_t* rc_cy8c26_sens_default_sdr(pp_tp_sensdev_t* s);

static void rc_cy8c26_update_reading(pp_tp_reg_subscriber_t*,
				     unsigned char, unsigned char);
static void rc_cy8c26_update_phase_reading(pp_tp_reg_subscriber_t*,
					   unsigned char, unsigned char);
static void rc_cy8c26_update_temp_reading(pp_tp_reg_subscriber_t*,
					  unsigned char, unsigned char);


rc_cy8c26_sens_type_cfg_t rc_cy8c26_sens_type_cfg_voltage = {
    .sens_type    = IPMI_SENSOR_TYPE_VOLTAGE,
    .reading_type = IPMI_EVENT_READING_TYPE_THRESHOLD,
    .analog_type  = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
    .unit_type    = IPMI_UNIT_TYPE_VOLTS,
    .sdr_rexp     = 0,
    .update       = rc_cy8c26_update_reading,
};

rc_cy8c26_sens_type_cfg_t rc_cy8c26_sens_type_cfg_freq = {
    .sens_type    = IPMI_SENSOR_TYPE_OTHER_UNITS_BASED_SENSOR,
    .reading_type = IPMI_EVENT_READING_TYPE_THRESHOLD,
    .analog_type  = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
    .unit_type    = IPMI_UNIT_TYPE_HZ,
    .sdr_rexp     = 0,
    .update       = rc_cy8c26_update_reading,
};

rc_cy8c26_sens_type_cfg_t rc_cy8c26_sens_type_cfg_current = {
    .sens_type    = IPMI_SENSOR_TYPE_CURRENT,
    .reading_type = IPMI_EVENT_READING_TYPE_THRESHOLD,
    .analog_type  = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
    .unit_type    = IPMI_UNIT_TYPE_AMPS,
    .sdr_rexp     = -1,
    .update       = rc_cy8c26_update_reading,
};

rc_cy8c26_sens_type_cfg_t rc_cy8c26_sens_type_cfg_phase = {
    .sens_type    = IPMI_SENSOR_TYPE_POWER_UNIT,
    .reading_type = IPMI_EVENT_READING_TYPE_THRESHOLD,
    .analog_type  = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
    .unit_type    = IPMI_UNIT_TYPE_RADIANS,
    .sdr_rexp     = -1,
    .update       = rc_cy8c26_update_phase_reading,
};

rc_cy8c26_sens_type_cfg_t rc_cy8c26_sens_type_cfg_temp = {
    .sens_type    = IPMI_SENSOR_TYPE_TEMPERATURE,
    .reading_type = IPMI_EVENT_READING_TYPE_THRESHOLD,
    .analog_type  = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
    .unit_type    = IPMI_UNIT_TYPE_DEGREES_C,
    .sdr_rexp     = -2,
    .update       = rc_cy8c26_update_temp_reading,
};

rc_cy8c26_sens_cfg_t rc_cy8c26_sens_cfg[13] = {
    { .reg = RC_CY8C26_RMS_VOLTAGE,                       /* 0 */
      .type = &rc_cy8c26_sens_type_cfg_voltage,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
    { .reg = RC_CY8C26_FREQUENCY,                        /* 1 */
      .type = &rc_cy8c26_sens_type_cfg_freq,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0), 
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
    { .reg = RC_CY8C26_TEMPERATURE,                      /* 2 */
      .type = &rc_cy8c26_sens_type_cfg_temp,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(25, 0),
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(300, 0),
      .sdr_bexp = 1},
    { .reg = RC_CY8C26_CURRENT_0,                        /* 3 */
      .type = &rc_cy8c26_sens_type_cfg_current,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0), 
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
    { .reg = RC_CY8C26_CURRENT_1,                        /* 4 */
      .type = &rc_cy8c26_sens_type_cfg_current,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
    { .reg = RC_CY8C26_CURRENT_2,                        /* 5 */
      .type = &rc_cy8c26_sens_type_cfg_current,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
    { .reg = RC_CY8C26_CURRENT_3,                        /* 6 */
      .type = &rc_cy8c26_sens_type_cfg_current,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
    { .reg = RC_CY8C26_CURRENT_4,                        /* 7 */
      .type = &rc_cy8c26_sens_type_cfg_current,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
    { .reg = RC_CY8C26_PHASE_SHIFT_0,                    /* 8 */
      .type = &rc_cy8c26_sens_type_cfg_phase,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
    { .reg = RC_CY8C26_PHASE_SHIFT_1,                    /* 9 */
      .type = &rc_cy8c26_sens_type_cfg_phase,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
    { .reg = RC_CY8C26_PHASE_SHIFT_2,                    /* 10 */
      .type = &rc_cy8c26_sens_type_cfg_phase,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
    { .reg = RC_CY8C26_PHASE_SHIFT_3,                    /* 11 */
      .type = &rc_cy8c26_sens_type_cfg_phase,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
    { .reg = RC_CY8C26_PHASE_SHIFT_4,                    /* 12 */
      .type = &rc_cy8c26_sens_type_cfg_phase,
      .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
      .sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
      .sdr_bexp = 0},
};

#define R_SENS_OFFSET 3 /* receptacle sensor start here */
#define R_SENS_SKEW   5 /* next sensor type changes with this skew */

pp_tp_obj_t* pp_rc_cy8c26_sens_ctor(const char* id, vector_t* args) {
    const char* errmsg = "%s(): '%s' failed: %s";
    rc_cy8c26_sens_t* this = NULL;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_rs485_chip_t* chip;
    unsigned int sens_no, r_no, r_sens_no;

    if (vector_size(args) > 2) {
	if (pp_tp_arg_scanf(args, 0, &err, "o<8>dd", &chip, &r_sens_no,
			    &r_no) != 3) {
	    pp_bmc_log_perror(errmsg, ___F, id, pp_strstream_buf(&err));
	    goto fail;
	} else if (r_no >= R_SENS_SKEW || r_sens_no >= 2) {
	    pp_bmc_log_perror(errmsg, ___F, id, "outlet or sensor too big");
	    errno = EINVAL;
	    goto fail;
	} else {
	    sens_no = R_SENS_OFFSET + r_sens_no * R_SENS_SKEW + r_no;
	}
    } else {
	if (pp_tp_arg_scanf(args, 0, &err, "o<8>d", &chip, &sens_no) != 2) {
	    pp_bmc_log_perror(errmsg, ___F, id, pp_strstream_buf(&err));
	    goto fail;
	}
    }
    
    if (strcmp(chip->model, RC_CY8C29_MODEL_STRING)) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "incorrect chip model");
    } else if (sens_no > 12) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "invalid sensor number");
    } else {
	rc_cy8c26_sens_cfg_t* cfg = &rc_cy8c26_sens_cfg[sens_no];
	this = malloc(sizeof(rc_cy8c26_sens_t));
	pp_tp_sensdev_init(&this->base, PP_TP_SENS_DEV, id,
			   (void(*)(pp_tp_obj_t*))rc_cy8c26_sens_destroy,
			   rc_cy8c26_sens_default_sdr);
	this->chip = pp_tp_rs485_chip_duplicate(chip);
	this->cfg = cfg;

	/* register for change notification */
	this->csubs.reg_changed = cfg->type->update;
	pp_tp_rs485_chip_register(chip, &this->csubs, cfg->reg);
	pp_bmc_log_debug("[CY8C26_Sens] %s: sens %d initialized at reg %#.2hhx"
			 , id, sens_no, cfg->reg);
    }
 fail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void rc_cy8c26_sens_destroy(pp_tp_obj_t* o) {
    rc_cy8c26_sens_t* this = (rc_cy8c26_sens_t*)o;
    assert(this != NULL);
    pp_tp_rs485_chip_unregister(this->chip, &this->csubs);
    pp_tp_rs485_chip_release(this->chip);
    pp_tp_sensdev_cleanup(&this->base);
    free(this);
}

static ipmi_sdr_header_t* rc_cy8c26_sens_default_sdr(pp_tp_sensdev_t* s) {
    rc_cy8c26_sens_t* this = (rc_cy8c26_sens_t*)s;
    rc_cy8c26_sens_cfg_t* cfg = this->cfg;
    ipmi_sdr_full_sensor_t* sdr;	

    sdr = ipmi_sdr_full_part_create(cfg->type->sens_type,
				    cfg->type->reading_type,
				    cfg->type->unit_type,
				    cfg->type->analog_type,
				    cfg->sdr_mtol, 
				    cfg->sdr_bacc,
				    cfg->sdr_bexp, cfg->type->sdr_rexp,
				    ipmi_sdr_max_by_form(cfg->type->analog_type),
				    ipmi_sdr_min_by_form(cfg->type->analog_type));
    return (ipmi_sdr_header_t*)sdr;
}


static void rc_cy8c26_update_reading(pp_tp_reg_subscriber_t* s,
				     unsigned char addr UNUSED,
				     unsigned char val UNUSED) {
    rc_cy8c26_sens_t* this = PP_TP_INTF_2_OBJ_CAST(s, rc_cy8c26_sens_t, csubs);
    pp_tp_rs485_chip_t* chip = this->chip;
    rc_cy8c26_sens_cfg_t* cfg = this->cfg;
    unsigned char value;
    int newreading;

    if (rc_cy8c26_get_byte(chip, cfg->reg, &value) == PP_SUC) {
	newreading = value;
    } else {
	newreading = -1;
	pp_bmc_log_error("[RS485_Sens]: ERROR: reg=%#.2hhx", cfg->reg); 
    }
    if (this->base.reading != newreading) {
	this->base.reading = newreading;
	pp_bmc_tp_sensdev_notify_subscribers(&this->base, this->base.reading);
    }
}


static void rc_cy8c26_update_phase_reading(pp_tp_reg_subscriber_t* s,
					   unsigned char addr UNUSED,
					   unsigned char val UNUSED) {
    rc_cy8c26_sens_t* this = PP_TP_INTF_2_OBJ_CAST(s, rc_cy8c26_sens_t, csubs);
    pp_tp_rs485_chip_t* chip = this->chip;
    rc_cy8c26_sens_cfg_t* cfg = this->cfg;
    unsigned char value;
    int newreading;
    
    if (rc_cy8c26_get_byte(chip, cfg->reg, &value) == PP_SUC) {
	newreading = (value & 0x80) ? value * -1 : value;
    } else {
	newreading = -1;
	pp_bmc_log_error("[RS485_Sens]: ERROR: reg=%#.2hhx", cfg->reg); 
    }
    if (this->base.reading != newreading) {
	this->base.reading = newreading;
	pp_bmc_tp_sensdev_notify_subscribers(&this->base, this->base.reading);
    }
}

static void rc_cy8c26_update_temp_reading(pp_tp_reg_subscriber_t* s,
					  unsigned char addr UNUSED,
					  unsigned char val UNUSED) {
    rc_cy8c26_sens_t* this = PP_TP_SCANNABLE_2_OBJ_CAST(s, rc_cy8c26_sens_t);
    pp_tp_rs485_chip_t* chip = this->chip;
    rc_cy8c26_sens_cfg_t* cfg = this->cfg;
    unsigned char value;
    int newreading;

    /* ATTENTION: this code ist definitely not correct,          !! *
     * do fix once cypress temp sensor is implemented and needed !! */
    assert(0);
    if (rc_cy8c26_get_byte(chip, cfg->reg, &value) == PP_SUC) {
	// FIXME: tbr: looks strange, really correct?
	newreading = ((((value & 0xff) << 4)
		       + ((value & 0xf000) >> 12)) / 4) - 120;
    } else {
	newreading = -1;
	pp_bmc_log_error("[RS485_Sens]: ERROR: reg=%#.2hhx", cfg->reg); 
    }
    if (this->base.reading != newreading) {
	this->base.reading = newreading;
	pp_bmc_tp_sensdev_notify_subscribers(&this->base, this->base.reading);
    }
}

/*
 * gpiodev for cy8c26 chip
 ****************************************************/

/*
 * object structures
 * -----------------
 */
typedef struct rc_cy8c26_gpio_s {
    pp_tp_gpio_dev_t base;
    pp_tp_rs485_chip_t* chip;
    unsigned char regaddr;
    pp_tp_reg_subscriber_t regsubsc;
    int oldval;
} rc_cy8c26_gpio_t;

/*
 * static methods and definitions
 * --------------------------------
 */
static void rc_cy8c26_gpio_dtor(pp_tp_obj_t* o);
static int  rc_cy8c26_gpio_get_val(pp_tp_gpio_dev_t* d, unsigned long mask);
static int  rc_cy8c26_gpio_set_val(pp_tp_gpio_dev_t* d, unsigned long mask,
				   unsigned long val, unsigned long tristate);
static void rc_cy8c26_gpio_changed(pp_tp_reg_subscriber_t*, unsigned char,
				   unsigned char);


/*
 * public & private methods
 * -----------------------------
 */
pp_tp_obj_t* pp_rc_cy8c26_gpio_ctor(const char* id, vector_t* args) {
    const char* errmsg = "%s(): '%s' failed: %s";
    const char* fn = "[RCCY8C26Gpio] ctor";
    rc_cy8c26_gpio_t* this = NULL;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_rs485_chip_t* chip;
    unsigned char outlet;

    if (pp_tp_arg_scanf(args, 0, &err, "o<8>d<c>", &chip, &outlet) != 2) {
	pp_bmc_log_perror(errmsg, fn, id, pp_strstream_buf(&err));
    } else if (strcmp(chip->model, RC_CY8C29_MODEL_STRING)) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, fn, id, "incorrect chip model");
    } else if (outlet >= ((rc_cy8c26_chip_t*)chip)->nr_outlets){
	errno = EINVAL;
	pp_bmc_log_error(errmsg,  fn, id, "invalid outlet");
    } else {
	this = malloc(sizeof(rc_cy8c26_gpio_t));
	pp_tp_gpio_dev_init(&this->base, PP_FPGA_GPIO_DEV,
			    id, 0x00000031 /* 2 bit led, 1 bit relay */,
			    rc_cy8c26_gpio_dtor,
			    NULL,/* no init necessary */
			    rc_cy8c26_gpio_get_val,
			    rc_cy8c26_gpio_set_val,
			    NULL);
	this->chip = pp_tp_rs485_chip_duplicate(chip);
	this->regaddr = RC_CY8C26_REG(RECEPTACLE_CTRL, outlet);
	this->oldval = -1;
	this->regsubsc.reg_changed = rc_cy8c26_gpio_changed;
	pp_tp_rs485_chip_register(this->chip, &this->regsubsc, this->regaddr);
	pp_bmc_log_debug("%s(): '%s' initialized for reg %#.2hhx",
			 fn, id, this->regaddr);
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void rc_cy8c26_gpio_dtor(pp_tp_obj_t* o) {
    rc_cy8c26_gpio_t* this = (rc_cy8c26_gpio_t*)o;
    assert(this != NULL);
    pp_bmc_log_debug("[RCCY8C26Gpio] dtor: '%s' deinit for reg %#.2hhx",
		     pp_tp_gpio_dev_to_string(&this->base), this->regaddr);
    pp_tp_rs485_chip_unregister(this->chip, &this->regsubsc);
    pp_tp_rs485_chip_release(this->chip);
    pp_tp_gpio_dev_cleanup(&this->base);
    free(this);
}

static int rc_cy8c26_gpio_get_val(pp_tp_gpio_dev_t* d, unsigned long mask) {
    rc_cy8c26_gpio_t* this = (rc_cy8c26_gpio_t*)d;
    unsigned char val;
    assert((mask & ~d->valid_mask) == 0);
    pp_tp_rs485_chip_get_byte(this->chip, this->regaddr, &val);
    return val & mask;
}

static int  rc_cy8c26_gpio_set_val(pp_tp_gpio_dev_t* d, unsigned long mask,
				   unsigned long val, unsigned long tristate) {
    rc_cy8c26_gpio_t* this = (rc_cy8c26_gpio_t*)d;
    unsigned char curval, setm, resetm;
    int ret = PP_SUC;

    assert((mask & ~d->valid_mask) == 0);

    /* remove tristate requests from mask, not supported for this device */
    mask &= ~tristate;
    
    /* set values for remaining bits in mask */
    if (mask != 0) {
	pp_tp_rs485_chip_get_byte(this->chip, this->regaddr, &curval);
	
	/* set one's */
	setm = val & mask;
	curval |= setm;
	/* set zero's */
	resetm = val | ~mask; 
	curval &= resetm;

	ret = pp_tp_rs485_chip_set_byte(this->chip, this->regaddr, curval);
    }
    return ret;
}

static void rc_cy8c26_gpio_changed(pp_tp_reg_subscriber_t* s,
				   unsigned char addr UNUSED,
				   unsigned char val) {
    rc_cy8c26_gpio_t* this = PP_TP_INTF_2_OBJ_CAST(s, rc_cy8c26_gpio_t,
						   regsubsc);
    int newval = val;
    unsigned long change_mask = 0;
    
    if (this->oldval < 0) {
	change_mask = this->base.valid_mask;
	this->oldval = val;
    } else if (newval != this->oldval) {
	change_mask = this->oldval ^ newval;
	this->oldval = val;
    }
    if (change_mask != 0) {
	pp_tp_gpio_dev_notify_subscribers(&this->base, this->oldval,
					  change_mask, 1);
    }
}
