/**
 * rc_rs485_dev.h
 *
 * This implements a special communication object for
 * RPC RS485 protocoll
 *
 * (c) 2006 Peppercon AG, 2006/09/05 tbr@raritan.com
 */
#include <fcntl.h>
#include <sys/poll.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <pp/termios.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_gpio_act.h>
#include <pp/bmc/tp_rs485_comdev.h>
#include "rc_rs485_dev.h"

#define noDEBUG
#ifdef DEBUG
#  define DB(h, b, l) buf_dump(h, b, l)
#else
#  define DB(h, b, l)
#endif

/*
 * some RS485 protocoll definitions
 * -------------------------------
 */

/* serial config, fixed currently */
#define RS_TTY_SPEED          57600
#define RS_TTY_PARITY         ""
#define RS_TTY_BITS           8
#define RS_TTY_STOP           1
#define RS_TTY_HWF            0
#define RS_TTY_SWF            0

#define RS_LSR_WAIT_TIME_MAX  (50000 / (RS_TTY_SPEED / 1000))

/* data link layer provisions */
#define RS_DL_START_CHR       0xa0
#define RS_DL_START_TOK       0xb0
#define RS_DL_END_CHR         0xa5
#define RS_DL_END_TOK         0xb5
#define RS_DL_ESC_CHR         0xaa
#define RS_DL_ESC_TOK         0xba

//#define RS_DL_CSUM_LEN      2       /* CRC16, implement later */
#define RS_DL_CSUM_LEN        1       /* simple checksum byte, currently */
#define RS_DL_HDR_LEN         1       /* addr */
#define RS_DL_FRM_LEN         (RS_DL_HDR_LEN + RS_DL_CSUM_LEN) /* addr+crc */
#define RS_DL_MAX_PKT_LEN     128     /* inclusive framing, exclusive escs */
#define RS_DL_MAX_PAYLOAD_LEN (RS_DL_MAX_PKT_LEN - RS_DL_FRM_LEN)
#define RS_DL_PKT_LEN(x)      (RS_DL_FRM_LEN + (x))
#define RS_DL_RETRIES         2       /* exclusive first attempt */
/* delay before send after recv
 * ATTENTION: this is a bussy wait, do not expand too much!!! */
#define RS_DL_SEND_AFTER_RECV_DELAY_US 1000LL

/* application layer provisions */
#define RS_AL_CMD_UNKNOWN     0xff
#define RS_AL_CMD_REGACC      0x01
#define RS_AL_CMD_FWUPD       0x02

/* register access provisions */
#define RS_RA_RECV_TIMEOUT_MS  20        /* ms */
#define RS_RA_WRITE_CMD_LEN(x) (2 + (x)) /* cmd + addr + data */
#define RS_RA_WRITE_RSP_LEN    1         /* cmd */
#define RS_RA_READ_CMD_LEN     3         /* cmd + addr + len  */
#define RS_RA_READ_RSP_LEN(x)  (1 + (x)) /* cmd + data */     
#define RS_RA_RW_MASK          0x80
#define RS_RA_READ_BIT         0x80
#define RS_RA_ADDR_MASK        0x7f

/* firmware update provisions */
#define RS_FW_UPD_CMD_LEN(x)   (1 + (x)) /* cmd + data */
#define RS_FW_UPD_RSP_LEN      2         /* cmd + status */


/*
 * structures and prototypes
 */
typedef struct {
    pp_tp_rs485_comdev_t base;
    char* uart_dev;
    int uart_fd;
    pp_tp_gpio_act_t* dir_gpio;
    pthread_mutex_t mtx;
    unsigned char msgbuf[RS_DL_MAX_PKT_LEN]; /* fixed message buffer */
    u_int64_t time_of_last_recv_us;
} rc_rs485_dev_t;

static void rc_rs485_dev_dtor(pp_tp_obj_t*);
static int rc_rs485_dev_read_data(pp_tp_rs485_comdev_t* this,
				  unsigned char rs485addr,
				  unsigned char addr, unsigned char len,
				  unsigned char* data);
static int  rc_rs485_dev_write_data(pp_tp_rs485_comdev_t* this,
				    unsigned char rs485addr,
				    unsigned char addr, unsigned char len,
				    unsigned char* data);
static int rc_rs485_dev_write_fwupd(pp_tp_rs485_comdev_t* this,
				    unsigned char rs485addr,
				    const char* buf, unsigned char len,
				    int timeout);

static int ra_send_read_data(rc_rs485_dev_t* this, unsigned char rs485addr,
			     unsigned char addr, unsigned char len,
			     unsigned char* rdata);
static int ra_fill_read_req(unsigned char* wp, unsigned char addr,
			    unsigned char len);

static int ra_send_write_data(rc_rs485_dev_t* this, unsigned char rs485addr,
			      unsigned char addr, unsigned char len,
			      const char* data);
static int ra_fill_write_req(unsigned char* wp, unsigned char addr,
			     unsigned char len, const char* data);

static int fw_send_upd_data(rc_rs485_dev_t* this, unsigned char rs485addr,
			    const char* data, unsigned char len, int timeout);
static int fw_fill_upd_req(unsigned char* wp, const char* data,
			   unsigned char len);

static int dl_send_recv(rc_rs485_dev_t* this, unsigned char rs485addr,
			unsigned char* sbuf, unsigned char* plptr,
			unsigned char pllen, unsigned char* rbuf,
			unsigned char rbuflen, int timeout);
static int dl_send(rc_rs485_dev_t* this, unsigned char* buf, unsigned int len);
static int dl_recv(rc_rs485_dev_t* this, unsigned char* buf,
		   unsigned int buflen, int timeout);
static unsigned int dl_escape(unsigned char* sbuf, unsigned char slen,
			      unsigned char* escbuf);
static unsigned int dl_unescape(unsigned char* escbuf, unsigned int esclen,
				unsigned char* rbuf);
static inline void dl_buf_request(rc_rs485_dev_t* this, unsigned char len,
				  unsigned char** sbuf, unsigned char** plptr);
static inline void dl_buf_release(rc_rs485_dev_t* this, unsigned char* sbuf);
static inline unsigned char dl_csum_calc(unsigned char* buf,
					 unsigned char len);
static inline int dl_csum_check(unsigned char* buf, unsigned char len);
static unsigned char buf_sum(unsigned char* buf, unsigned char len);
static int buf_search(unsigned char* buf, unsigned char chr, int len);
#ifdef DEBUG
static void buf_dump(const char* header, unsigned char* buf, int len);
#endif /* DEBUG */

/*
 * public and private implementation
 */
pp_tp_obj_t* pp_rc_rs485_dev_ctor(const char* id, vector_t* args) {
    const char* fn = "[RS485] ctor";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    rc_rs485_dev_t* this = NULL;
    pp_tp_gpio_act_t* dir_gpio;
    char* uart_dev;
    int fd;
    
    if (vector_size(args) > 2) {
	pp_bmc_log_error("%s(): '%s' failed: too many arguments", fn, id);
    } else if (pp_tp_arg_scanf(args, 0, &err, "so<ag>", &uart_dev, &dir_gpio)  
        != 2) {
	pp_bmc_log_error("%s(): '%s' failed: %s (%s)",
                fn, id, strerror(errno), pp_strstream_buf(&err));
    } else if (0 > (fd = open(uart_dev, O_RDWR)) ||
               PP_ERR == pp_base_set_tty_params(fd, RS_TTY_SPEED,RS_TTY_PARITY,
                                                RS_TTY_BITS, RS_TTY_STOP,  
                                                RS_TTY_HWF, RS_TTY_SWF)) {
        pp_bmc_log_perror("%s(): '%s' open(%s) failed", fn, id, uart_dev);
    } else {
	this = malloc(sizeof(rc_rs485_dev_t));
	pp_tp_rs485_comdev_init(&this->base, PP_TP_RS485_COM_DEV, id,
				rc_rs485_dev_dtor,
				rc_rs485_dev_read_data,
				rc_rs485_dev_write_data,
				rc_rs485_dev_write_fwupd);
        this->dir_gpio = pp_tp_gpio_act_duplicate(dir_gpio);
	this->uart_dev = strdup(uart_dev);
        this->uart_fd = fd;
	this->time_of_last_recv_us = pp_hrtime_us()
	    - RS_DL_SEND_AFTER_RECV_DELAY_US;
	MUTEX_CREATE_ERRORCHECKING(&this->mtx);
	pp_bmc_log_debug("%s(): '%s' initialized for %s", fn, id, uart_dev);
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void rc_rs485_dev_dtor(pp_tp_obj_t* o) {
    rc_rs485_dev_t* this = (rc_rs485_dev_t*)o;
    pp_bmc_log_debug("[RS485] dtor: '%s' deinit for %s",
		     pp_tp_rs485_comdev_to_string(&this->base),
		     this->uart_dev);
    close(this->uart_fd);
    free(this->uart_dev);
    pp_tp_gpio_act_release(this->dir_gpio);
    pp_tp_rs485_comdev_cleanup(&this->base);
    free(this);
}

/*
 * abstract API: read and write data
 * ------------------------------------
 * these functions need to be synchronized among multiple threads,
 * since scanner thread and bmc-main thread may call at the same time
 */
static int  rc_rs485_dev_read_data(pp_tp_rs485_comdev_t* o,
				   unsigned char rs485addr,
				   unsigned char addr,
				   unsigned char len,
				   unsigned char* data) {
    int ret;
    rc_rs485_dev_t* this = (rc_rs485_dev_t*)o;
    
    if (len > RS_DL_MAX_PAYLOAD_LEN || (addr & RS_RA_RW_MASK) != 0) {
	errno = EINVAL;
	return PP_ERR;
    }
    MUTEX_LOCK(&this->mtx);
    ret = ra_send_read_data(this, rs485addr, addr, len, data);
    MUTEX_UNLOCK(&this->mtx);
    return ret;
}

static int  rc_rs485_dev_write_data(pp_tp_rs485_comdev_t* o,
				    unsigned char rs485addr,
				    unsigned char addr,
				    unsigned char len,
				    unsigned char* data) {
    int ret;
    rc_rs485_dev_t* this = (rc_rs485_dev_t*)o;
    
    if (len > RS_DL_MAX_PAYLOAD_LEN || (addr & RS_RA_RW_MASK) != 0) {
	errno = EINVAL;
	return PP_ERR;
    }
    MUTEX_LOCK(&this->mtx);
    ret = ra_send_write_data(this, rs485addr, addr, len, data);
    MUTEX_UNLOCK(&this->mtx);
    return ret;
}

static int rc_rs485_dev_write_fwupd(pp_tp_rs485_comdev_t* o,
				    unsigned char rs485addr,
				    const char* buf, unsigned char len,
				    int timeout) {
    int ret;
    rc_rs485_dev_t* this = (rc_rs485_dev_t*)o;
    
    if (len > RS_DL_MAX_PAYLOAD_LEN) {
	errno = EINVAL;
	return PP_ERR;
    }
    MUTEX_LOCK(&this->mtx);
    ret = fw_send_upd_data(this, rs485addr, buf, len, timeout);
    MUTEX_UNLOCK(&this->mtx);
    return ret;
}

/*
 * common application layer functions
 * -----------------------------------
 */
/* check response cmd byte and copy requested len into user data */
static int 
al_proc_rsp(unsigned char cmd, unsigned char* rbuf, unsigned char rcnt,
	    unsigned char* data, unsigned char len) {
    int ret;
    
    if (rcnt < 1 || rbuf[0] != cmd) {
	pp_bmc_log_debug("%s(): invalid response: %.2x, exp %.2x",
			 ___F, rcnt < 1 ? -1 : rbuf[0], cmd);
	errno = EINVAL;
	return PP_ERR;
    }
    if (data != NULL) {
	rcnt -= 1; /* skip cmd byte */
	ret = len < rcnt ? len : rcnt;
	memcpy(data, &rbuf[1], ret);
    } else {
	ret = 0;
    }
    return ret;
}

/*
 * firmware update application protocoll functions: prefix: fw
 * ------------------------------------------------------------
 */
static int
fw_send_upd_data(rc_rs485_dev_t* this, unsigned char rs485addr,
		 const char* data, unsigned char len, int timeout) {
    unsigned char *sbuf, *plptr;
    unsigned char rbuf[RS_FW_UPD_RSP_LEN];
    int pllen, rcnt;
    unsigned char status;
    
    dl_buf_request(this, RS_FW_UPD_CMD_LEN(len), &sbuf, &plptr);
    pllen = fw_fill_upd_req(plptr, data, len);
    if (PP_ERR != (rcnt = dl_send_recv(this, rs485addr, sbuf, plptr, pllen,
				       rbuf, sizeof(rbuf), timeout))) {
	rcnt = al_proc_rsp(RS_AL_CMD_FWUPD, rbuf, rcnt,
			  &status, sizeof(status));
    }
    dl_buf_release(this, sbuf);
    return (rcnt < 1) ? PP_ERR : status;
}    

static int
fw_fill_upd_req(unsigned char* wp, const char* data, unsigned char len) {
    assert(len < RS_DL_MAX_PAYLOAD_LEN);
    
    wp[0] = RS_AL_CMD_FWUPD;
    memcpy(&wp[1], data, len);
    return (RS_FW_UPD_CMD_LEN(len));
}


/*
 * register access application protocoll functions: prefix: ra
 * ------------------------------------------------------------
 */
static int
ra_send_read_data(rc_rs485_dev_t* this, unsigned char rs485addr,
		  unsigned char addr, unsigned char len, unsigned char* data) {
    unsigned char *sbuf, *plptr;
    unsigned char rbuf[RS_DL_MAX_PAYLOAD_LEN];
    int pllen, rcnt;
    
    dl_buf_request(this, RS_RA_READ_CMD_LEN, &sbuf, &plptr);
    pllen = ra_fill_read_req(plptr, addr, len);
    if (PP_ERR != (rcnt = dl_send_recv(this, rs485addr, sbuf, plptr, pllen,
				       rbuf, sizeof(rbuf),
				       RS_RA_RECV_TIMEOUT_MS))) {
	rcnt = al_proc_rsp(RS_AL_CMD_REGACC, rbuf, rcnt, data, len);
    }
    dl_buf_release(this, sbuf);
    return rcnt;
}    

static int
ra_fill_read_req(unsigned char* wp, unsigned char addr, unsigned char len) {
    assert((addr & RS_RA_RW_MASK) == 0);
    assert(len < RS_DL_MAX_PAYLOAD_LEN);
    
    wp[0] = RS_AL_CMD_REGACC;
    wp[1] = addr | RS_RA_READ_BIT;
    wp[2] = len;
    return RS_RA_READ_CMD_LEN;
}


static int
ra_send_write_data(rc_rs485_dev_t* this, unsigned char rs485addr,
		   unsigned char addr, unsigned char len,
		   const char* data) {
    unsigned char *sbuf, *plptr;
    unsigned char rbuf[RS_RA_WRITE_RSP_LEN];
    int pllen;
    int rcnt;

    dl_buf_request(this, RS_RA_WRITE_CMD_LEN(len), &sbuf, &plptr);
    pllen = ra_fill_write_req(plptr, addr, len, data);
    if (PP_ERR != (rcnt = dl_send_recv(this, rs485addr, sbuf, plptr, pllen,
				       rbuf, sizeof(rbuf),
				       RS_RA_RECV_TIMEOUT_MS))) {
	rcnt = al_proc_rsp(RS_AL_CMD_REGACC, rbuf, rcnt, NULL, 0);
    }
    dl_buf_release(this, sbuf);
    return rcnt;
}

static int
ra_fill_write_req(unsigned char* wp, unsigned char addr, unsigned char len,
		  const char* data) {
    assert((addr & RS_RA_RW_MASK) == 0);
    assert(len > 0);
    assert(len <= (RS_DL_MAX_PAYLOAD_LEN - RS_RA_WRITE_CMD_LEN(0)));
    
    wp[0] = RS_AL_CMD_REGACC;
    wp[1] = addr;
    memcpy(&wp[2], data, len);
    return (RS_RA_WRITE_CMD_LEN(len));
 }

/*
 * datalink layer protocoll functions: prefix: dl
 * -----------------------------------------------
 */
static int
dl_send_recv(rc_rs485_dev_t* this, unsigned char rs485addr,
	     unsigned char* sbuf, unsigned char* plptr,	unsigned char pllen,
	     unsigned char* rbuf, unsigned char rbuflen, int timeout) {
    const char* fn = "[RS485] dl_send_recv";
    const char* id = pp_tp_rs485_comdev_to_string(&this->base);
    unsigned char  escbuf[(2 * RS_DL_MAX_PKT_LEN) + 2/*start+stop*/];
    unsigned int   esclen;
    unsigned char  rcvbuf[(2 * RS_DL_MAX_PKT_LEN) + 2/*start+stop*/];
    int            rcvlen;
    unsigned char* hdr = sbuf;
    unsigned char* tail = plptr + pllen;
    int retries, ret = PP_ERR;
    u_int64_t time_since_last_recv_us;
    assert((sbuf + RS_DL_HDR_LEN) == plptr);
    assert(pllen <= RS_DL_MAX_PAYLOAD_LEN);

    hdr[0] = rs485addr;
    tail[0] = dl_csum_calc(hdr, pllen + RS_DL_HDR_LEN);

    /* leave space for start and end */
    esclen = dl_escape(sbuf, RS_DL_PKT_LEN(pllen), &escbuf[1]);
    escbuf[0] = RS_DL_START_CHR;
    escbuf[esclen + 1] = RS_DL_END_CHR;

    for (retries = 0; retries < RS_DL_RETRIES + 1; ++retries) {
	if (retries > 0) {
	    pp_bmc_log_debug("%s(): %s: dl_send %d. retry, "
			     "saddr=%#hhx, len=%d",
			     fn, id, retries, rs485addr, esclen);
	}
	
	/* make sure the moment of last receive was long enough ago *
	 * if not delay minimum delay time                          */
	time_since_last_recv_us = pp_hrtime_us() - this->time_of_last_recv_us;
	if (time_since_last_recv_us < RS_DL_SEND_AFTER_RECV_DELAY_US) {
	    pp_hrdelay_us(RS_DL_SEND_AFTER_RECV_DELAY_US
			  - time_since_last_recv_us);
	}
	
	DB("dl_send: ", escbuf, esclen + 2);
	if (PP_ERR == dl_send(this, escbuf, esclen + 2)) {
	    pp_bmc_log_perror("%s(): %s: dl_send failed: all-len=%d",
			      fn, id, esclen + 2);
	    return PP_ERR;
	}
	
	if (PP_ERR == (rcvlen = dl_recv(this, rcvbuf, sizeof(rcvbuf),
					timeout))) {
	    pp_bmc_log_perror("%s(): %s: dl_recv failed", fn, id);
	    return PP_ERR;
	}
	if (rcvlen == 0) { /* timeout */
	    pp_bmc_log_debug("%s(): %s: dl_recv timeout", fn, id);
	    continue;
	}
	
	/* we really received something, remember time so that next *
	 * send can be delayed as required by relay controller      */
	this->time_of_last_recv_us = pp_hrtime_us();
	DB("dl_recv: ", rcvbuf, rcvlen);
	
	if (rcvlen < (RS_DL_FRM_LEN + 2) /* frm + start + stop */) {
	    pp_bmc_log_debug("%s(): %s: dl_recv msg too short, rlen=%d",
			     fn, id, rcvlen);
	    continue;
	}
	/* unescape and cut off start and end,  *
	 * start and end are checked by dl_recv */
	rcvlen = dl_unescape(&rcvbuf[1], rcvlen - 2, rcvbuf);
        if (dl_csum_check(rcvbuf, rcvlen) == PP_ERR) {
	    pp_bmc_log_debug("%s(): %s: dl_recv checksum wrong, rlen=%d",
			     fn, id, rcvlen);
	    continue;
	}
	if (rcvbuf[0] != rs485addr) {
	    pp_bmc_log_debug("%s(): %s: dl_recv slave addr wrong: "
			     "addr=%#hhx, expet=%#hhx",
			     fn, id, rcvbuf[0], rs485addr);
	    continue;
	}

	/* everything seems ok, prepare receive data to return */
	ret = rcvlen - RS_DL_FRM_LEN;
	if (ret > rbuflen) {
	    pp_bmc_log_warn("%s(): %s: dl_recv data discarded: "
			    "recv=%d, storer=%d", fn, id, ret, rbuflen);
	    ret = rbuflen;
	}
	if (rbuf != NULL && ret > 0) {
	    memcpy(rbuf, &rcvbuf[RS_DL_HDR_LEN], ret);
	}
	break;
    }
    return ret;
}

static int
dl_send(rc_rs485_dev_t* this, unsigned char* buf, unsigned int len) {
    int ret;
    int lsr = 0, lsrcnt;

    pp_tp_gpio_act_set_gpio(this->dir_gpio, 1); /* assert send direction */
    ret = write(this->uart_fd, buf, len);
    tcdrain(this->uart_fd); /* wait for buffer to become empty */
    /* bussy wait until transmission register becomes empty, counter timeout
     * was empirically determined and is depended on line speed */
    for (lsrcnt = 0;
	 !(lsr & TIOCSER_TEMT) && lsrcnt < RS_LSR_WAIT_TIME_MAX;
	 ++lsrcnt) {
	ioctl(this->uart_fd, TIOCSERGETLSR, &lsr);
    }
    pp_tp_gpio_act_set_gpio(this->dir_gpio, 0); /* deassert send */
    
    if (ret >= 0 && ret != (int)len) { /* should never ever happen */
	errno = ECOMM;
	ret = PP_ERR;
    }
    return ret;
}   

#define DL_RECV_START_NOT_SEEN 0
#define DL_RECV_START_SEEN     1
static int
dl_recv(rc_rs485_dev_t* this, unsigned char* buf, unsigned int bsz,
	int timeout) {
    int state = DL_RECV_START_NOT_SEEN;
    unsigned char* rptr = buf;
    unsigned int rsz = bsz, bcnt = 0, acnt = 0;
    int i, ret, rcnt;
    struct pollfd fd[1];

    while(1) {
	fd[0].fd = this->uart_fd;
	fd[0].events = POLLIN;
	fd[0].revents = 0;

	if (0 > (rcnt = poll(fd, 1, timeout))) {   /* poll error */
	    if (errno == EINTR) continue;
	    ret = PP_ERR;
	    break;
	}
	if (rcnt == 0) {                           /* poll timeeout */
	    errno = ETIMEDOUT;
	    ret = 0;
	    break;
	}
	if ((fd[0].revents & POLLIN) == 0) {       /* fd error */
	    errno = EIO;
	    ret = PP_ERR;
	    break;
	}
	if (0 >= (rcnt = read(this->uart_fd, rptr, rsz))) {  /* selected */
	    ret = PP_ERR;
	    break;
	}

	/* search start char */
	if (state == DL_RECV_START_NOT_SEEN) {
	    if ((i = buf_search(buf, RS_DL_START_CHR, rcnt)) >= 0) {
		state = DL_RECV_START_SEEN;
		if (i > 0) {
		    rcnt -= i;
		    memcpy(&buf[0], &buf[i], rcnt);
		}
	    } else {
		if ((acnt += rcnt) >= bsz) { /* too mutch garbage */
		    errno = EIO;
		    ret = PP_ERR;
		    break;
		} else {
		    continue;
		}
	    }
	}
	
	/* search for end char if start has seen previously */
	if (state == DL_RECV_START_SEEN) {
	    if ((i = buf_search(rptr, RS_DL_END_CHR, rcnt)) >= 0) {
		ret = bcnt + i + 1;
		break;
	    } else {
		rptr += rcnt;
		bcnt += rcnt;
		if ((rsz  -= rcnt) <= 0) { /* buf full, too mutch garbage */
		    errno = EIO;
		    ret = PP_ERR;
		    break;
		}
	    }
	}
    }
    return ret;
}

static unsigned int
dl_escape(unsigned char* sbuf, unsigned char slen, unsigned char* escbuf) {
    unsigned char si, c;
    unsigned int  ei;
    assert(slen > 0);

    for (si = 0, ei = 0; si < slen; ++si) {
	c = sbuf[si];
	switch(c) {
	  case RS_DL_START_CHR:
	      escbuf[ei++] = RS_DL_ESC_CHR;
	      escbuf[ei++] = RS_DL_START_TOK;
	      break;
	  case RS_DL_END_CHR:
	      escbuf[ei++] = RS_DL_ESC_CHR;
	      escbuf[ei++] = RS_DL_END_TOK;
	      break;
	  case RS_DL_ESC_CHR:
	      escbuf[ei++] = c;
	      escbuf[ei++] = RS_DL_ESC_TOK;
	      break;
	  default:
	      escbuf[ei++] = c;
	}
    }
    return ei;
}

static unsigned int
dl_unescape(unsigned char* escbuf, unsigned int esclen, unsigned char* rbuf) {
    unsigned char ri, c;
    unsigned int  ei;
    assert(esclen > 0);

    for (ei = 0, ri = 0; ei < esclen; ++ei) {
	c = escbuf[ei];
	if (c == RS_DL_ESC_CHR) {
	    if (ei + 1 >= esclen) {
		rbuf[ri++] = 0;
	    } else {
		switch(escbuf[++ei]) {
		  case RS_DL_START_TOK:
		      rbuf[ri++] = RS_DL_START_CHR;
		      break;
		  case RS_DL_END_TOK:
		      rbuf[ri++] = RS_DL_END_CHR;
		      break;
		  case RS_DL_ESC_TOK:
		      rbuf[ri++] = RS_DL_ESC_CHR;
		      break;
		  default:
		      rbuf[ri++] = 0; // this is an error, should not happen
		}
	    }
	} else {
	    rbuf[ri++] = c;
	}
    }
    return ri;
}


static inline void
dl_buf_request(rc_rs485_dev_t* this, unsigned char len,
	       unsigned char** sbuf, unsigned char** plptr) {
    PP_ASSERT(len, len <= RS_DL_MAX_PAYLOAD_LEN);
    *sbuf = this->msgbuf;
    *plptr = this->msgbuf + RS_DL_HDR_LEN;
}

static inline void
dl_buf_release(rc_rs485_dev_t* this UNUSED, unsigned char* sbuf UNUSED) {
    /* nothing, we use fixed buffer in this */
}

static inline unsigned char
dl_csum_calc(unsigned char* buf, unsigned char len) {
    return 0 - buf_sum(buf, len);
}

static inline int
dl_csum_check(unsigned char* buf, unsigned char len) {
    return buf_sum(buf, len) == 0 ? PP_SUC : PP_ERR;
}

static unsigned char
buf_sum(unsigned char* buf, unsigned char len) {
    unsigned char i, sum = 0;
    for (i = 0; i < len; ++i) {
	sum += buf[i];
    }
    return sum;
}


static int
buf_search(unsigned char* buf, unsigned char chr, int len) {
    int i;
    for (i = 0; i < len; ++i) {
	if (buf[i] == chr) {
	    return i;
	}
    }
    return PP_ERR;
}

#ifdef DEBUG
static void
buf_dump(const char* header, unsigned char* buf, int len) {
    int i;
    fprintf(stderr, "%s", header);
    for (i = 0; i < len; ++i) {
	fprintf(stderr, "%.2x ", buf[i]);
    }
    fprintf(stderr, "\n");
}
#endif
