/**
 * rpc_disp_ctrl.c
 *
 * implements the PDU 7 Segment Display Controller
 * This file includes all related objects (sensors, gpio, actuator)
 * 
 * (c) 2006 Peppercon AG, 2006/4/26, george@raritan.com
 */
 
#include <pp/bmc/tp_gpio_multi_sens.h>
#include <pp/bmc/tp_gpio_sens.h>
#include <pp/bmc/debug.h>
#include <pp/selector.h>
#include <pp/bmc/tp_i2c_chip.h>
#include "drivers/rpc_disp_ctrl.h"

const unsigned char Seg7[10]={ 0x3f, 0x06, 0x5b, 0x4f, 0x66,
			       0x6d, 0x7d, 0x27, 0x7f, 0x6f };

/*
 * pwms
 **************************/

typedef struct rpc_disp_s {
    pp_tp_disp_act_t base;
    pp_tp_i2c_chip_t* chip;
    unsigned int disp_no;
} rpc_disp_act_t;

static void rpc_disp_act_dtor(pp_tp_obj_t* o);
static int rpc_disp_act_set_mode(pp_tp_disp_act_t* o, pp_tp_disp_mode_t mode);
static int rpc_disp_act_set_value(pp_tp_disp_act_t* o, unsigned char );

pp_tp_obj_t* pp_rpc_disp_act_ctor(const char* id, vector_t* args) {
    const char* errmsg = "%s(): '%s' failed: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
	
    rpc_disp_act_t* this = NULL;	
    pp_tp_i2c_chip_t* chip;
    unsigned int chip_no;

    if (pp_tp_arg_scanf(args, 0, &err, "o<h>d", &chip, &chip_no) != 2) {
	pp_bmc_log_perror(errmsg, ___F, id, pp_strstream_buf(&err));
    } else if (strcmp(chip->model, PP_TP_I2C_CHIP_GENERIC)) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "incorrect chip model");
    } else if (chip_no > RPC_DISP_MAX_NUMBER) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "invalid pwm number");
    } else {
	this = malloc(sizeof(rpc_disp_act_t));
	pp_tp_disp_act_init(&this->base, PP_TP_PWM_ACT, id,
			    rpc_disp_act_dtor,
			    rpc_disp_act_set_mode,
			    rpc_disp_act_set_value);
	this->chip = pp_tp_i2c_chip_duplicate(chip);	
	if (PP_SUC == pp_tp_i2c_chip_pre_com(chip)) {
	    if(chip_no) {
		pp_tp_i2c_chip_tx_byte_data(chip, 0, 0x47);
		pp_tp_i2c_chip_tx_byte_data(chip, 2, 0x3f); 		
	    } else
		pp_tp_i2c_chip_tx_byte_data(chip, 0, 0x27);
	    pp_tp_i2c_chip_tx_byte_data(chip, 1, 0x06); 		
	    pp_tp_i2c_chip_tx_byte_data(chip, 3, 0x3f);
	    pp_tp_i2c_chip_post_com(chip);
    	}
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void rpc_disp_act_dtor(pp_tp_obj_t* o) {
    rpc_disp_act_t* this = (rpc_disp_act_t*)o;
    assert(this != NULL);
    pp_tp_i2c_chip_release(this->chip);
    pp_tp_disp_act_cleanup(&this->base);
    free(this);
}

/* set and get mode not supported for this chip, currently */
static int rpc_disp_act_set_mode(pp_tp_disp_act_t* o ,
			       pp_tp_disp_mode_t mode ) {
    rpc_disp_act_t* this = (rpc_disp_act_t*)o;
    this->disp_no = mode;
    return PP_SUC;
}

static int rpc_disp_act_set_value(pp_tp_disp_act_t* o,
				unsigned char value) {
    rpc_disp_act_t* this = (rpc_disp_act_t*)o;
    pp_tp_i2c_chip_t* chip = this->chip;
    unsigned int data1, data2, data3, disp_no = this->disp_no;    
    int ret;
    //pp_bmc_log_debug("[RPCDISP] VOLT %d->%d",disp_no,value);

    if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	if(disp_no==PP_TP_DISP_CBE) {
	    data1=0x79;
	    data2=0x7c;
	    data3=0x39;
	} else if(disp_no==PP_TP_DISP_WAIT) {
	    data1=0x40;
	    data2=0x40;
	    data3=0x40;		
	} else {
	    data3=Seg7[(value/100)];
	    data2=Seg7[((value/10)%10)];
	    if(disp_no==PP_TP_DISP_CURR) data2 |= 0x80;
	    data1=Seg7[(value%10)];  
	}
	
	if(disp_no==PP_TP_DISP_FLASH) {
	    if(value)
		pp_tp_i2c_chip_tx_byte_data(chip, 0, 0x41);
	    else
		pp_tp_i2c_chip_tx_byte_data(chip, 0, 0x47);
	} else { 
	    pp_tp_i2c_chip_tx_byte_data(chip, 1, data1);	
	    if(disp_no==PP_TP_DISP_CHAN)	    
		pp_tp_i2c_chip_tx_byte_data(chip, 3, data2);
	    else {
		pp_tp_i2c_chip_tx_byte_data(chip, 2, data2);		
		pp_tp_i2c_chip_tx_byte_data(chip, 3, data3); 
	    }
	}
	pp_tp_i2c_chip_post_com(chip);
    }
	
    return ret;
}

/*
 * 7 Segment Display Controler
 **************************/
typedef struct rpc_disp_ctrl_s {
    pp_tp_ctrl_t base;
    unsigned int receptacle;	 //0x8000-disp voltage, 0x4000-hold button, 0x2000-CBE, 0x1800-flash
    int accu_to_hndl;
    vector_t * disp_actors;   // vector of tp_disp_act_t* 	
    vector_t * sens;	
    vector_t * sens_subscribers;
    //pp_tp_sensdev_subscriber_t sens_subscr[RPC_DISP_SENS_MAX];
} rpc_disp_ctrl_t;

typedef struct subsc_entry_s {
    rpc_disp_ctrl_t* obj;
    pp_tp_gpio_sens_t* gpios;
    pp_tp_sensdev_t* sens;
    pp_tp_sensdev_subscriber_t subscriber;
    size_t                   index;	
} subsc_entry_t;

static int rpc_disp_ctrl_init(rpc_disp_ctrl_t* this, pp_tp_obj_type_t type,
			      const char* id, pp_tp_obj_dtor_func_t dtor,
			      vector_t* sens, vector_t* disp_actors);

static void rpc_disp_ctrl_dtor(pp_tp_obj_t* this);
static void rpc_disp_ctrl_cleanup(rpc_disp_ctrl_t* this);
static void rpc_disp_gpio_sens_recv_reading(pp_tp_sensdev_subscriber_t* subsc,
					    pp_tp_sensdev_t* source, int reading);
static void rpc_disp_sens_recv_reading(rpc_disp_ctrl_t* this,
					 int reading, int ctx);
static int read_sensor_to(const int item_id UNUSED, void* ctx);


pp_tp_obj_t* pp_rpc_disp_ctrl_ctor(const char* id, vector_t* args) {

    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    const char* errmsg = "[RPC7SegCtrl] c'tor: '%s' failed %s";
	
    rpc_disp_ctrl_t* disp = NULL;	
    vector_t * disp_actors = NULL;	
    vector_t * sens = NULL;
    int i, arg_cnt, sens_cnt;

    arg_cnt = vector_size(args);
    //pp_bmc_log_debug("%s() : vector_new -> disp_actors\n", id);

    if (arg_cnt < 8) {
	pp_bmc_log_error(errmsg, id, "missing argument(s)");
	goto bail;
    }

    disp_actors = vector_new(NULL, 2,
			     (vector_elem_del_func_simple)pp_tp_obj_release);
    // Note: vector is deleted in object destructor
    
    for (i = 0; i <2; ++i) {
        pp_tp_disp_act_t* disp_act ;
		
        if(pp_tp_arg_scanf(args, i, &err, "o<ap>", &disp_act) != 1) {
	    pp_bmc_log_error(errmsg, id, pp_strstream_buf(&err));
	    vector_delete(disp_actors);
            goto bail;
        }
        vector_add(disp_actors, pp_tp_disp_act_duplicate(disp_act));
    }
    	
    sens_cnt = arg_cnt;
    sens = vector_new(NULL, sens_cnt,
		      (vector_elem_del_func_simple)pp_tp_obj_release);
	
    for (i = 2; i <4; ++i) {
	pp_tp_gpio_sens_t * gpio_sens;
        if (pp_tp_arg_scanf(args, i, &err, "o<sg>", &gpio_sens) != 1) {
	    pp_bmc_log_error(errmsg, id, pp_strstream_buf(&err));
	    vector_delete(sens);
            goto bail;
        }
	vector_add(sens, pp_tp_gpio_sens_duplicate(gpio_sens));
    }
    pp_bmc_log_notice("sensor arg %d",sens_cnt);
	
    for (i = 4; i <sens_cnt; ++i) {
	pp_tp_sensdev_t * pwr_sens;
        if (pp_tp_arg_scanf(args, i, &err, "o<s>", &pwr_sens) != 1) {
	    pp_bmc_log_error(errmsg, id, pp_strstream_buf(&err));
	    vector_delete(sens);
            goto bail;
        }
	vector_add(sens, pp_tp_sensdev_duplicate(pwr_sens));
    }

    disp = calloc(1, sizeof(rpc_disp_ctrl_t));
    if (PP_FAILED(rpc_disp_ctrl_init(disp, PP_TP_DISP_CTRL, id,
				     rpc_disp_ctrl_dtor, sens, disp_actors))) {
	pp_bmc_log_error(errmsg, id, "init");
	vector_delete(sens);
	free(disp);
	disp = NULL;
    }   

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)disp;	
}

static void rpc_disp_ctrl_dtor(pp_tp_obj_t* this) {
    rpc_disp_ctrl_cleanup((rpc_disp_ctrl_t*)this);
    free(this);
    //pp_bmc_log_debug("[DTOR] : %s()", ___F);	
}


static int rpc_disp_ctrl_init(rpc_disp_ctrl_t* this, pp_tp_obj_type_t type,
			      const char* id, pp_tp_obj_dtor_func_t dtor,
			      vector_t* sens, vector_t* disp_actors) {
    pp_tp_gpio_sens_t* gpio;
    pp_tp_sensdev_t* pwr_sens;
    subsc_entry_t* subsc_entry;
    size_t i, sens_cnt= vector_size(sens);

    if (pp_tp_ctrl_init(&this->base, type, id, dtor)) {
	pp_bmc_log_debug("%s() : initial fail", id);
        return PP_ERR;
    }
    this->receptacle = 1;
    this->sens = sens; /* sens are already duplicates */
    this->disp_actors = disp_actors; /* actors are already duplicates */
    this->sens_subscribers = vector_new(NULL, sens_cnt, free);
    
    for (i = 0; i < 2; ++i) {
	gpio = (pp_tp_gpio_sens_t*)vector_get(this->sens, i);
	subsc_entry = calloc(1, sizeof(subsc_entry_t));
	subsc_entry->obj = this;
	subsc_entry->gpios = gpio;
	subsc_entry->sens = NULL;	   
	subsc_entry->index = i;
	subsc_entry->subscriber.recv_reading =rpc_disp_gpio_sens_recv_reading;
	vector_add(this->sens_subscribers, subsc_entry);   
	pp_tp_gpio_sens_subscribe(gpio, &subsc_entry->subscriber);  
    }
    for (i = 2; i < sens_cnt; ++i) {
	pwr_sens = (pp_tp_sensdev_t*)vector_get(this->sens, i);
	subsc_entry = calloc(1, sizeof(subsc_entry_t));
	subsc_entry->obj = this;
	subsc_entry->gpios = NULL;	   
	subsc_entry->sens = pwr_sens;
	subsc_entry->index = i;
	subsc_entry->subscriber.recv_reading =rpc_disp_gpio_sens_recv_reading;
	vector_add(this->sens_subscribers, subsc_entry);   
	pp_bmc_tp_sensdev_subscribe(pwr_sens, &subsc_entry->subscriber);  
    }  
  
    return PP_SUC;
}

static void rpc_disp_ctrl_cleanup(rpc_disp_ctrl_t* this)
{
    size_t i, sens_cnt = vector_size(this->sens);
    pp_tp_gpio_sens_t* gpio;
    pp_tp_sensdev_t* pwr_sens;
    pp_tp_sensdev_subscriber_t* subscriber;

    for (i = 0; i < 2; ++i) {
	gpio = (pp_tp_gpio_sens_t*)vector_get(this->sens, i);
	subscriber =
	    (pp_tp_sensdev_subscriber_t*)vector_get(this->sens_subscribers, i);
	pp_tp_gpio_sens_unsubscribe(gpio, subscriber);
    }
    for (i = 2; i < sens_cnt; ++i) {
	pwr_sens = (pp_tp_sensdev_t*)vector_get(this->sens, i);
	subscriber =
	    (pp_tp_sensdev_subscriber_t*)vector_get(this->sens_subscribers, i);
	pp_bmc_tp_sensdev_unsubscribe(pwr_sens, subscriber);
    }
    vector_delete(this->disp_actors);
    vector_delete(this->sens_subscribers);
    vector_delete(this->sens);	   
   
    pp_tp_ctrl_cleanup(&this->base);    
}



static void rpc_disp_gpio_sens_recv_reading(pp_tp_sensdev_subscriber_t* subsc,
					      pp_tp_sensdev_t* source UNUSED, int reading) {
    // find my index
    subsc_entry_t* subsc_entry =
	PP_TP_INTF_2_OBJ_CAST(subsc, subsc_entry_t, subscriber);
    rpc_disp_ctrl_t* this = subsc_entry->obj;
    rpc_disp_sens_recv_reading(this, reading, (int)subsc_entry->index);    
}

static int read_sensor_to(const int item_id UNUSED, void* ctx) {
    rpc_disp_ctrl_t *o = ctx;
    pp_tp_disp_act_t* disp_act;
    pp_tp_sensdev_t* curr_sens;	
    int f;
	
    if(o->receptacle & 0x8000) {
        o->receptacle &= 0x3fff;
	curr_sens = (pp_tp_sensdev_t*)vector_get(o->sens, (o->receptacle+3));
	
	disp_act = (pp_tp_disp_act_t*)vector_get(o->disp_actors, 1);
	//disp_act->set_mode(disp_act, PP_TP_DISP_WAIT);	  
	disp_act->set_mode(disp_act, PP_TP_DISP_CURR);
	rpc_disp_act_set_value(disp_act, curr_sens->reading);
	//pp_bmc_log_debug("[RPCDISP] BUTTON %x",o->receptacle);
    }
    if(o->receptacle & 0x1000) {
	disp_act = (pp_tp_disp_act_t*)vector_get(o->disp_actors, 1);
	disp_act->set_mode(disp_act, PP_TP_DISP_FLASH);
	o->receptacle = (o->receptacle^0x800);
	f=(o->receptacle>>11)&0x01;
	rpc_disp_act_set_value(disp_act, f);		
    }
    return PP_SUC;	
}

void rpc_disp_sens_recv_reading(rpc_disp_ctrl_t* this,
				  int reading, int ctx)
{
    unsigned int sensor_id = (int)(long)ctx;
    pp_tp_disp_act_t* disp_act;
    pp_tp_gpio_sens_t * gpio_sens;
    pp_tp_sensdev_t* volt_sens;	
    pp_tp_sensdev_t* curr_sens;	
	
    unsigned int data = (this->receptacle&0x801f) + 3;
    size_t sens_cnt= vector_size(this->sens);
    volt_sens = (pp_tp_sensdev_t*)vector_get(this->sens, 2);
    	
    if((volt_sens->reading < 10)&&(sensor_id>1)) {
        sensor_id=DISP_SENSOR_CBE;
    } else if(this->receptacle & 0x2000) {
	this->receptacle &= ~0x2000;
	disp_act = (pp_tp_disp_act_t*)vector_get(this->disp_actors, 1);
	disp_act->set_mode(disp_act, PP_TP_DISP_VOLT);         
	rpc_disp_act_set_value(disp_act, volt_sens->reading);
	pp_select_add_to(5000, 0, read_sensor_to, this);
    }
    switch (sensor_id) {
      case DISP_SENSOR_BUT_UP:
	  gpio_sens = (pp_tp_gpio_sens_t*)vector_get(this->sens, 1);
	  /* pp_bmc_log_debug("[RPCDISP] BUT_UP %x %x",
	     gpio_sens->base.reading, reading);
	  */
	  if((gpio_sens->base.reading==0)&&(reading==0)) {
	      this->receptacle |= 0xC000;
	      disp_act = (pp_tp_disp_act_t*)vector_get(this->disp_actors, 1);
	      if(volt_sens->reading < 10) {
		  disp_act->set_mode(disp_act, PP_TP_DISP_CBE);
	      } else {
		  disp_act->set_mode(disp_act, PP_TP_DISP_VOLT);
	      }
	      rpc_disp_act_set_value(disp_act, volt_sens->reading);
	  } else if(reading) {
	      if(this->receptacle&0x4000) {
		  this->receptacle &= ~0x4000;
		  pp_select_add_to(5000, 0, read_sensor_to, this);
	      }

	      this->receptacle ++;
	 
	      if((this->receptacle&0x1f) > (sens_cnt-4))
		  //this->receptacle = (this->receptacle & 0xE000)+1;
		  this->receptacle = (this->receptacle & 0xE000);
	      disp_act = (pp_tp_disp_act_t*)vector_get(this->disp_actors, 0);
	      disp_act->set_mode(disp_act, PP_TP_DISP_CHAN);	  
	      rpc_disp_act_set_value(disp_act, (this->receptacle&0x1f));
	      
	      if(!(data & 0xE000)) {
		  curr_sens = (pp_tp_sensdev_t*)vector_get(this->sens, ((this->receptacle&0x1f)+3));
		  disp_act = (pp_tp_disp_act_t*)vector_get(this->disp_actors, 1);
		  if(volt_sens->reading < 10) {
		      disp_act->set_mode(disp_act, PP_TP_DISP_CBE);
		  } else {
		      disp_act->set_mode(disp_act, PP_TP_DISP_CURR);
		      if((this->receptacle&0x1f)==0) {
   		          rpc_disp_act_set_value(disp_act,
						 curr_sens->reading * 2);
		      } else {
			  rpc_disp_act_set_value(disp_act, curr_sens->reading);
		      }
		  }
	      }
	  }
	  break;
      case DISP_SENSOR_BUT_DOWN:
	  gpio_sens = (pp_tp_gpio_sens_t*)vector_get(this->sens, 0);
	  //pp_bmc_log_debug("[RPCDISP] BUT_DOWN %x %x",gpio_sens->base.reading,reading);
	  if((gpio_sens->base.reading==0)&&(reading==0)) {
	      this->receptacle |= 0xC000;
	      disp_act = (pp_tp_disp_act_t*)vector_get(this->disp_actors, 1);
	      if(volt_sens->reading < 10) {
		  disp_act->set_mode(disp_act, PP_TP_DISP_CBE);		  
	      } else {
		  disp_act->set_mode(disp_act, PP_TP_DISP_VOLT);
	      }
	      rpc_disp_act_set_value(disp_act, volt_sens->reading);
	  } else if(reading) {
	      if(this->receptacle&0x4000) {
		  this->receptacle &= ~0x4000;
		  pp_select_add_to(5000, 0, read_sensor_to, this);
	      }	  
	 
	      if((this->receptacle & 0x1f) == 0) {
		  this->receptacle = (this->receptacle & 0xE000)+(sens_cnt-4);
	      } else {
		  this->receptacle --;
	      }
	      
	      disp_act = (pp_tp_disp_act_t*)vector_get(this->disp_actors, 0);
	      disp_act->set_mode(disp_act, PP_TP_DISP_CHAN);	  
		  
	      rpc_disp_act_set_value(disp_act, (this->receptacle&0x1f));
	      if(!(data & 0xE000)) {
		  curr_sens = (pp_tp_sensdev_t*)vector_get(this->sens,
						((this->receptacle&0x1f)+3));
		  disp_act = (pp_tp_disp_act_t*)vector_get(this->disp_actors,
							   1);
		  if(volt_sens->reading < 10) {
		      disp_act->set_mode(disp_act, PP_TP_DISP_CBE);
		  } else {
		      disp_act->set_mode(disp_act, PP_TP_DISP_CURR);
		      if((this->receptacle&0x1f)==0) {
   		          rpc_disp_act_set_value(disp_act,
						 curr_sens->reading * 2);
		      } else {
		          rpc_disp_act_set_value(disp_act, curr_sens->reading);
		      }
		  }
	      }
	  }
	  break;
      case DISP_SENSOR_VOLT:
	  if(!(data & 0x8000)) break;	  	
	  //pp_bmc_log_debug("[RPCDISP] VOLT %d",reading);
	  disp_act = (pp_tp_disp_act_t*)vector_get(this->disp_actors, 1);
	  if(volt_sens->reading < 10) {
	      disp_act->set_mode(disp_act, PP_TP_DISP_CBE);
	  } else {
	      disp_act->set_mode(disp_act, PP_TP_DISP_VOLT);
	  }
	  rpc_disp_act_set_value(disp_act, volt_sens->reading);
	  break;
      case DISP_SENSOR_CURR00:	  
      case DISP_SENSOR_CURR01:
      case DISP_SENSOR_CURR02:
      case DISP_SENSOR_CURR03:
      case DISP_SENSOR_CURR04:
      case DISP_SENSOR_CURR05:
      case DISP_SENSOR_CURR06:
      case DISP_SENSOR_CURR07:
      case DISP_SENSOR_CURR08:
      case DISP_SENSOR_CURR09:
      case DISP_SENSOR_CURR10:
      case DISP_SENSOR_CURR11:
      case DISP_SENSOR_CURR12:
      case DISP_SENSOR_CURR13:
      case DISP_SENSOR_CURR14:
      case DISP_SENSOR_CURR15:
      case DISP_SENSOR_CURR16:
      case DISP_SENSOR_CURR17:
      case DISP_SENSOR_CURR18:
      case DISP_SENSOR_CURR19:
      case DISP_SENSOR_CURR20:	  	
	  if((data & 0x801f) != sensor_id) break;
	  //pp_bmc_log_debug("[RPCDISP] CURR_%d %d",(sensor_id-3), reading);
	  disp_act = (pp_tp_disp_act_t*)vector_get(this->disp_actors, 1);
	  disp_act->set_mode(disp_act, PP_TP_DISP_CURR);
	  if(sensor_id==DISP_SENSOR_CURR00) {
   	      rpc_disp_act_set_value(disp_act, (reading*2));
	  } else {
	      rpc_disp_act_set_value(disp_act, reading);
	  }
	  break;
      case DISP_SENSOR_CBE:
	  this->receptacle |= 0x2000;
	  disp_act = (pp_tp_disp_act_t*)vector_get(this->disp_actors, 1);
	  disp_act->set_mode(disp_act, PP_TP_DISP_CBE);
	  rpc_disp_act_set_value(disp_act, 0x00);
	  break;
      case DISP_SENSOR_FLASH:
	  if((this->receptacle&0x1000)==0) {
	      this->receptacle |= 0x1000;
	      this->accu_to_hndl = pp_select_add_to(1000, 1, read_sensor_to,
						    this);
	  }
	  break;	  	
      case DISP_SENSOR_NOFLASH:
	  this->receptacle &= ~0x1800;
	  if(this->accu_to_hndl) {
	      disp_act = (pp_tp_disp_act_t*)vector_get(this->disp_actors, 1);
	      disp_act->set_mode(disp_act, PP_TP_DISP_FLASH);
	      rpc_disp_act_set_value(disp_act, 0x00);
	      pp_select_remove_to(this->accu_to_hndl);
	      this->accu_to_hndl = 0;
	  }
	  break;	  		  
      default:	  	
	  break;
    }    
}

