/**
 * w83791d_gpio.c
 *
 * implements W83791D gpio device
 * 
 * (c) 2005 Peppercon AG, 2005/11/08, thomas@peppercon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/selector.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_chip.h>
#include <pp/bmc/tp_scan_sensdev.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/tp_gpio_dev.h>

#include "drivers/w83791d.h"
#include "drivers/w83791d_gpio.h"

/*
 * gpio dev
 **************************/

typedef struct w83791d_gpio_s {
    pp_tp_gpio_dev_t base;
    pp_tp_i2c_chip_t* chip;
    int reading;
    u_short state; /* saved output state */
    u_short state_valid_mask; /* output state validity */
    pp_tp_sensdev_subscriber_t init_cond_subs;
    unsigned long last_change_mask;
    /* regarding scannable, refer to docu in tp_scan_sensdev.h */
    pp_sensor_scannable_t scannable;
    pthread_mutex_t mtx;
    int update_sf_id;
} w83791d_gpio_t;

static void w83791d_gpio_destroy(pp_tp_obj_t* o);
static int w83791d_gpio_get_val(pp_tp_gpio_dev_t* d, unsigned long mask);
static int w83791d_gpio_set_val(pp_tp_gpio_dev_t* d, unsigned long mask,
				unsigned long val, unsigned long tristate);
static void w83791d_gpio_update(pp_sensor_scannable_t* s);
static int w83791d_gpio_notify(void* ctx);
static void init_cond_recv_reading(pp_tp_sensdev_subscriber_t* subscriber, pp_tp_sensdev_t* source, int reading);

pp_tp_obj_t* pp_sensor_w83791d_gpio_ctor(const char* id, vector_t* args) {
    const char* errmsg = "%s(): '%s' failed: %s";
    w83791d_gpio_t* this = NULL;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_i2c_chip_t* chip;

    if (pp_tp_arg_scanf(args, 0, &err, "o<h>", &chip) != 1) {
	pp_bmc_log_perror(errmsg, ___F, id, pp_strstream_buf(&err));
    } else if (strcmp(chip->model, W83791D_MODEL_STRING)) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "incorrect chip model");
    } else {
	this = malloc(sizeof(w83791d_gpio_t));
	pp_tp_gpio_dev_init(&this->base, PP_FPGA_GPIO_DEV,
			    id, ((pp_tp_w83791d_chip_t*)chip)->gpio_mask,
			    w83791d_gpio_destroy,
			    NULL, /* no init necessary */
			    w83791d_gpio_get_val, w83791d_gpio_set_val,
			    NULL);
	this->chip    = pp_tp_i2c_chip_duplicate(chip);
        this->scannable.update = w83791d_gpio_update;
        this->update_sf_id = 0;
	this->reading = -1;
	this->state = 0;
	this->state_valid_mask = 0;
        MUTEX_CREATE_ERRORCHECKING(&this->mtx);
        pp_sensor_scanner_add(&this->scannable, 0);
	/*
	 * NOTE: If someone sets a GPIO when the chip is not powered up yet we
	 *       have to remember the value and delay the GPIO operation until
	 *       the chip is alive. We do this using the init-condition of the
	 *       i2c-chip we are bound to. We assume that the callback is
	 *       called after the power-up callback that is registered on the
	 *       same condition!
	 */

	/* register for init condition to call init code */
	if (this->chip->init_cond) {
	    this->init_cond_subs.recv_reading = init_cond_recv_reading;
	    pp_tp_cond_subscribe(this->chip->init_cond, &this->init_cond_subs);
	}
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void w83791d_gpio_destroy(pp_tp_obj_t* o) {
    w83791d_gpio_t* this = (w83791d_gpio_t*)o;
    assert(this != NULL);
    if (this->chip->init_cond) {
	pp_tp_cond_unsubscribe(this->chip->init_cond, &this->init_cond_subs);
    }
    pp_sensor_scanner_remove(&this->scannable);
    if (this->update_sf_id != 0) {
        pp_select_remove_sf(this->update_sf_id);
    }
    pp_tp_i2c_chip_release(this->chip);
    pp_tp_gpio_dev_cleanup(&this->base);
    free(this);
}

static int w83791d_gpio_get_val(pp_tp_gpio_dev_t* d, unsigned long mask) {
    int reading = -1;
    w83791d_gpio_t* this = (w83791d_gpio_t*)d;
    assert((mask & ~d->valid_mask) == 0);

    if (this->chip->init_cond && !pp_bmc_tp_cond_is_true(this->chip->init_cond)) {
	errno = ENODEV; /* device not available */
    } else if ((reading = this->reading) >= 0) {
	reading &= mask;
    }
    return reading;
}

static int w83791d_gpio_set_val_internal(w83791d_gpio_t* this)
{
    unsigned char r0, r1;
    int ret = PP_ERR;

    if (PP_SUCCED(pp_tp_i2c_chip_pre_com(this->chip))) {
	if (PP_FAILED(pp_tp_i2c_chip_rx_byte_data(this->chip, W83791D_GPIO_DATA_STATUS_REG_1, &r0))
	    || PP_FAILED(pp_tp_i2c_chip_rx_byte_data(this->chip, W83791D_GPIO_DATA_STATUS_REG_2, &r1))) {
	    goto bail_post;
	}
	r0 &= ~this->state_valid_mask & 0xff;
	r1 &= ~(this->state_valid_mask >> 8) & 0xff;
	r0 |= this->state & this->state_valid_mask & 0xff; 
	r1 |= ((this->state & this->state_valid_mask) >> 8) & 0x0f;
	if (PP_FAILED(pp_tp_i2c_chip_tx_byte_data(this->chip, W83791D_GPIO_DATA_STATUS_REG_1, r0)) ||
	    PP_FAILED(pp_tp_i2c_chip_tx_byte_data(this->chip, W83791D_GPIO_DATA_STATUS_REG_2, r1))) {
	    goto bail_post;
	}
	ret = PP_SUC;
    bail_post:
	pp_tp_i2c_chip_post_com(this->chip);
    }

    return ret;
}


static int w83791d_gpio_set_val(pp_tp_gpio_dev_t* d, unsigned long mask,
				unsigned long val,
				unsigned long tristate UNUSED) {
    w83791d_gpio_t* this = (w83791d_gpio_t*)d;
    pp_tp_w83791d_chip_t* w83791d = (pp_tp_w83791d_chip_t*)this->chip;
    unsigned short m = w83791d->gpio_mask & w83791d->gpio_io_mask & mask; // mask out
    assert((mask & ~d->valid_mask) == 0);

    /* first determine the output states and remeber it for possible later use */
    this->state &= ~m;
    this->state |= val & m;
    this->state_valid_mask |= m;

    /* check whether access allowed per init condition */
    if (this->chip->init_cond == NULL || pp_bmc_tp_cond_is_true(this->chip->init_cond)) {
	return w83791d_gpio_set_val_internal(this);
    }

    return PP_ERR;
}

static void w83791d_gpio_update(pp_sensor_scannable_t* s) {
    w83791d_gpio_t* this = PP_TP_INTF_2_OBJ_CAST(s, w83791d_gpio_t, scannable);
    pp_tp_i2c_chip_t* chip = this->chip;
    pp_tp_w83791d_chip_t* w83791d = (pp_tp_w83791d_chip_t*)chip;
    unsigned short m = w83791d->gpio_mask & ~w83791d->gpio_io_mask; // mask in
    unsigned char r0, r1;
    int reading = -1, old_reading;

    if (pp_tp_i2c_chip_is_initialized(chip) &&
	PP_SUC == pp_tp_i2c_chip_pre_com(chip)) {
	if ((pp_tp_i2c_chip_rx_byte_data(chip, W83791D_GPIO_DATA_STATUS_REG_1,
					 &r0) == PP_SUC) &&
	    (pp_tp_i2c_chip_rx_byte_data(chip, W83791D_GPIO_DATA_STATUS_REG_2,
					 &r1) == PP_SUC)) {
	    reading = ((r1 << 8) | r0) & m;
	}
        pp_tp_i2c_chip_post_com(chip);
    }

    MUTEX_LOCK(&this->mtx);
    if (this->update_sf_id == 0) {
        if (this->reading != reading) {
	    old_reading = this->reading;
            this->reading = reading;
	    if (reading < 0) {
		this->last_change_mask = 0;
	    } else {
		if (old_reading < 0) {
		    this->last_change_mask = this->base.valid_mask;
		} else {
		    this->last_change_mask =
			(unsigned long)(old_reading ^ reading);
		}
	    }
            this->update_sf_id = pp_select_add_sf(w83791d_gpio_notify, this);
        }
    }
    MUTEX_UNLOCK(&this->mtx);
}

static int w83791d_gpio_notify(void* ctx) {
    w83791d_gpio_t* this = (w83791d_gpio_t*)ctx;
    MUTEX_LOCK(&this->mtx);
    pp_tp_gpio_dev_notify_subscribers(&this->base,
				      (unsigned long)this->reading,
				      this->last_change_mask,
				      this->reading < 0 ? 0 : 1);
    this->update_sf_id = 0;
    this->last_change_mask = 0;
    MUTEX_UNLOCK(&this->mtx);
    return 0;
}

static void
init_cond_recv_reading(pp_tp_sensdev_subscriber_t* subscriber, 
		       pp_tp_sensdev_t* source UNUSED, int reading)
{
    w83791d_gpio_t* this;

    this = PP_TP_INTF_2_OBJ_CAST(subscriber, w83791d_gpio_t, init_cond_subs);
    if (reading > 0) {
	/* chip is available now: output state values */
	w83791d_gpio_set_val_internal(this);
    }
}
