/**
 * topo_factory.h
 *
 * Defines the object factory interface that can be used to implement
 * object factories that create object instances with arbitrary
 * signatures in a unified way. Used by the topology parser.
 * 
 * (c) 2005 Peppercon AG, 3/9/2005, tbr@peppecon.de
 */

#ifndef __TOPO_FACTORY_H__
#define __TOPO_FACTORY_H__

#include <pp/base.h>
#include <pp/vector.h>

#include <pp/bmc/topo_base_obj.h>

/*
 * object constructor definitions (object factory)
 * 
 * the object constructor signature is used
 * by predefined object classes for registering their code
 * and by the topology parser to create object instances
 * -----------------------------------------------------------
 */

typedef union {
    int   intval;
    char* string;
    pp_tp_obj_t* obj;
} pp_tp_arg_val_t;

typedef enum {
    PP_TP_ARG_INT,
    PP_TP_ARG_BOOL,
    PP_TP_ARG_STRING,
    PP_TP_ARG_OBJ,
    PP_TP_ARG_PLACEHOLDER, /* used by the parser, c'tor won't see it */
    PP_TP_ARG_SYMBOL       /* used by the parser, c'tor won't see it */
} pp_tp_arg_type_t;
    
typedef struct {
    pp_tp_arg_type_t type;
    pp_tp_arg_val_t  value;
} pp_tp_arg_t;

const char* pp_tp_arg_type_to_string(pp_tp_arg_t* arg);

pp_tp_arg_t pp_tp_arg_copy(pp_tp_arg_t* arg);

void pp_tp_arg_cleanup(pp_tp_arg_t* arg);

void pp_tp_arg_destroy(pp_tp_arg_t* arg);

/* signature of an object constructor definition */
typedef pp_tp_obj_t* (*pp_tp_obj_ctor_func_t)(const char* id, vector_t* args);

/** 
 * argument parsing function, to be used by the constructor
 * 
 * @param format assumes the following syntax:
 *    format: '"' mandatory_types '|' optional_types '"'
 *    mandatory_types: type_specifier
 *                     mandatory_types type_specifier
 *    optional_types: type_specifier
 *                    optional_types type_specifier
 *    type_specifier: type_char
 *                    | type_char < type_args >
 *
 * where type_char and type_args (indented) is one of the following 
 *  b - PP_TP_ARG_BOOL
 *  d - PP_TP_ARG_INT
 *     c - unsigned char
 *     s - unsigned short
 *  s - PP_TP_ARG_STRING
 *  o - PP_TP_ARG_OBJ
 *     i - PP_TP_I2C_COM_DEV
 *     h - PP_TP_I2C_CHIP
 *     4 - PP_TP_RS485_COM_DEV
 *     8 - PP_TP_RS485_CHIP
 *     g - PP_TP_GPIO_DEV
 *     s - PP_TP_SENS_DEV
 *        g - PP_TP_GPIO_SENS
 *        m - PP_TP_GPIO_MULTI_SENS
 *	  c - PP_TP_COND
 *        a - PP_TP_SCAN_SENS_DEV
 *     p - PP_TP_IPMI_SENSOR
 *        t - PP_TP_TRESH_IPMI_SENSOR
 *        d - PP_TP_DISC_IPMI_SENSOR
 *     r - PP_TP_CFG_SDR
 *        t - PP_TP_TRESH_SDR
 *        d - PP_TP_DISC_SDR
 *     a - PP_TP_ACTOR
 *        g - PP_TP_GPIO_ACT
 *        m - PP_TP_GPIO_MULTI_ACT
 *        p - PP_TP_PWM_ACT
 *        x - PP_TP_SMX_ACT
 *     c - PP_TP_CTRL
 *        f - PP_TP_FAN_CTRL
 *     m - PP_TP_MUXED_SCANNER
 *  p - PP_TP_ARG_PLACEHOLDER
 *
 * @return number of args that have been successfully scanned
 *         Note: if there are optional types but there is no argument
 *         in the args vector, they still will be treated as
 *         successfully scanned
 */
int pp_tp_arg_scanf(vector_t* args, int idx, pp_strstream_t* errstr,
		    const char *format, ...);

#endif
