/**
 * tp_gpio_act.h
 * 
 * A gpio actor that can set an output signal.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef __TP_GPIO_ACT_H__
#define __TP_GPIO_ACT_H__

#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/tp_gpio_dev.h>

typedef struct pp_tp_gpio_act_s {
    pp_tp_obj_t base;
    
    /* the gpio device that this signal belongs to */
    pp_tp_gpio_dev_t* gpio_dev;
    
    /* the number of the gpio signal that this object refers to */
    unsigned char gpio_no;
    
    /* the values of the signal if asserted/released */
    unsigned char sig_asserted;
    unsigned char sig_released;

    /* the last written value */
    int value;
   
    /* get the last written value of the selected gpio line
     * -1 = unknown, 1 = asserted, 0 = released
     */
    int (*get_gpio)(struct pp_tp_gpio_act_s* o);

    /* set the value of the selected gpio line
     * 1 = assert, 0 = release
     */
    int (*set_gpio)(struct pp_tp_gpio_act_s* o, unsigned char value);
    
} pp_tp_gpio_act_t;

/* ppdoc
 * name:   GPIO actor
 * tpname: GpioAct
 * desc:   A generic GPIO actor that abstracts one signal from a GPIO device.
 * desc:   The signal can be logically asserted and deasserted. The related
 * desc:   concrete signals for each state can be high, low and tristate.
 * arg 0: [OBJ (GPIO_DEV)] gpiodev: The gpio device which accesses this signal 
 * arg 1: [INT] signal_no:          The gpio signal number within the gpio device
 * arg 2: [INT] assert_value:       The value of the signal if signal is asserted,
 * arg 2:                           0=low, 1=high, 2=tristate('z')
 * arg 3: [INT] deassert_value:     The value of the signal if signal is deasserted,
 * arg 3:                           0=low, 1=high, 2=tristate('z')
 * arg 4: [BOOL] initial_value:     (optional) initial actor value, may be omitted
 * arg 4:                           to leave the sensor untouched
 * return: [OBJ(GPIO_ACT)]
 */
pp_tp_obj_t* pp_tp_gpio_act_ctor(const char* id, vector_t* args);

/**
 * object initializer
 */
void pp_tp_gpio_act_init(pp_tp_gpio_act_t* o,
			 const char* id,
			 pp_tp_gpio_dev_t* gpio_dev,
			 unsigned char gpio_no,
			 unsigned char sig_asserted,
			 unsigned char sig_released,
			 unsigned char initial_ass);

void pp_tp_gpio_act_cleanup(pp_tp_gpio_act_t* o);

static inline pp_tp_gpio_act_t* pp_tp_gpio_act_duplicate(pp_tp_gpio_act_t* o) {
    return (pp_tp_gpio_act_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)o);
}

static inline void pp_tp_gpio_act_release(pp_tp_gpio_act_t* o) {
    pp_tp_obj_release((pp_tp_obj_t*)o);
}

static inline int pp_tp_gpio_act_set_gpio(pp_tp_gpio_act_t* o,
					  unsigned char value) {
    return o->set_gpio(o, value);
}


#endif




