/**
 * tp_gpio_dev.h
 *
 * An abstract gpio device
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef __PP_BMC_TP_GPIO_DEV_H__
#define __PP_BMC_TP_GPIO_DEV_H__

#include <pp/bmc/topo_base_obj.h>
#include <pp/vector.h>
#include <pp/base.h>

/**
 * gpio_subscriber callback interface
 */
typedef struct gpio_subscriber_s {
    /**
     * called to notify the subscriber that the gpio bank has changed.
     * The gpio device cannot be encoded in one value, so no new gpio
     * device reading is passed with this function. Instead the notified
     * device must query the gpio itself to find out its new value.
     */
    void (*gpio_changed)(struct gpio_subscriber_s* o, unsigned long value, int valid);
    unsigned long (*get_gpio_mask)(struct gpio_subscriber_s* o);
} pp_tp_gpio_subscriber_t;

/**
 * abstract active gpio device, extends active_sensor and has
 * gpio manipulation functions. The gpio device is an active
 * sensor and will notify all subscribers if one bit changes.
 */
typedef struct pp_tp_gpio_dev_s {

    pp_tp_obj_t base;

    vector_t* gpio_subscribers;
    unsigned long valid_mask;

    /**
     * Get the value (0,1) of the specified gpio signal.
     * @return 0 or 1 if ok
     *         -1 if invalid reading
     */ 
    int (*get_gpio)(struct pp_tp_gpio_dev_s* s, int gpio_num);

    /**
     * Set the value (0,1,'z') of the specified gpio signal.
     * @return 0 if oK
     *         -1 if error
     */
    int (*set_gpio)(struct pp_tp_gpio_dev_s* s, int gpio_num,
		    unsigned char value);

    /**
     * Initialize a certain gpio pin, has to be done before
     * using this pin with the set/get functions
     *
     * this function can be implemented by inherited objects for
     * hardware access of they need initialization
     *
     * @return 0 if ok
     *         -1 if error
     */ 
    int (*init_gpio)(struct pp_tp_gpio_dev_s* s, unsigned long mask);

    /**
     * this function must be implemented by inherited objects for
     * hardware access
     * @return reading value
     *         -1 if invalid reading
     */
    int (*get_val)(struct pp_tp_gpio_dev_s* s, unsigned long mask);

    /**
     * this function must be implemented by inherited objects for
     * hardware access
     * @return 0 if ok
     *         -1 if error
     */
    int (*set_val)(struct pp_tp_gpio_dev_s* s, unsigned long mask,
		   unsigned long val, unsigned long tristate);

    /**
     * this function can be implemented by inherited objects to get notified
     * when subscribers (un)subscribe (can be NULL)
     */
    void (*subscribers_changed)(struct pp_tp_gpio_dev_s* s);

} pp_tp_gpio_dev_t;


/**
 * gpio_dev initializer
 */
void pp_tp_gpio_dev_init(pp_tp_gpio_dev_t* d, pp_tp_obj_type_t type,
                         const char* id, unsigned long valid_mask,
                         void (*dtor)(pp_tp_obj_t*),
                         int (*init_gpio)(struct pp_tp_gpio_dev_s* s,
					  unsigned long mask),
			 int (*get_val)(struct pp_tp_gpio_dev_s* s,
					unsigned long mask),
                         int (*set_val)(struct pp_tp_gpio_dev_s* s,
					unsigned long mask, unsigned long val,
					unsigned long open),
                         void(*subscribers_changed)(struct pp_tp_gpio_dev_s*));

/**
 * gpio device destructor
 */
void pp_tp_gpio_dev_cleanup(pp_tp_gpio_dev_t* d);

/* convinience functions */
static inline const char*
pp_tp_gpio_dev_to_string(pp_tp_gpio_dev_t* this) {
    return pp_tp_obj_to_string(&this->base);
}

static inline int
pp_tp_gpio_dev_init_gpio(pp_tp_gpio_dev_t* this, unsigned long mask)
{
    return this->init_gpio(this, mask);
}

static inline int
pp_tp_gpio_dev_get_gpio(pp_tp_gpio_dev_t* this, int gpio_num)
{
    return this->get_gpio(this, gpio_num);
}

static inline int
pp_tp_gpio_dev_set_gpio(pp_tp_gpio_dev_t* this, int gpio_num, u_char value)
{
    return this->set_gpio(this, gpio_num, value);
}

static inline int
pp_tp_gpio_dev_get_val(struct pp_tp_gpio_dev_s* this, unsigned long mask) {
    return this->get_val(this, mask);
}

static inline int
pp_tp_gpio_dev_set_val(struct pp_tp_gpio_dev_s* this, unsigned long mask,
		       unsigned long val, unsigned long tristate) {
    return this->set_val(this, mask, val, tristate);
}

static inline pp_tp_gpio_dev_t*
pp_tp_gpio_dev_duplicate(pp_tp_gpio_dev_t* this)
{
    return (pp_tp_gpio_dev_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)this);
}

static inline void
pp_tp_gpio_dev_release(pp_tp_gpio_dev_t* this)
{
    pp_tp_obj_release((pp_tp_obj_t*)this);
}

/**
 * register a subscriber to the gpio
 * @returns a subscriber_id that must be used for unsubscribing
 */
void pp_tp_gpio_dev_register_subscriber(struct pp_tp_gpio_dev_s* s,
					 pp_tp_gpio_subscriber_t* subscriber);

/**
 * unregister a subscriber at the gpio
 * @returns PP_ERR if the specified subscriber_id is not known
 */
int pp_tp_gpio_dev_unregister_subscriber(struct pp_tp_gpio_dev_s* s,
					 pp_tp_gpio_subscriber_t* subscriber);

/**
 * this function can be called by inherited objects to notify the
 * subscribers that a value has changed
 */
void pp_tp_gpio_dev_notify_subscribers(struct pp_tp_gpio_dev_s* s, unsigned long value,
				       unsigned long change_mask, int valid);



#endif
