/**
 * tp_i2c_chip
 *
 * Defines an abstract i2c communication endpoint.
 * It is quite equivalent to an i2c communication device
 * except that the i2c address is known and fix.
 *
 * Main intention to introduce this class is to have an
 * object instance for Multi-Sensor-Chips that need power-on
 * initialization, for instance setting up registers and enabling it
 * 
 * (c) 2005 Peppercon AG, 2005/10/27, tbr@peppecon.de
 */

#ifndef __TP_I2C_CHIP_H__
#define __TP_I2C_CHIP_H__

#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_cond.h>

/*
 * abstract i2c communication endpoint
 */
typedef struct pp_tp_i2c_chip_s pp_tp_i2c_chip_t;
 
struct pp_tp_i2c_chip_s {
    pp_tp_obj_t base;

    /**
     * this chip's model string (usally statically allocated)
     * (currently used for extended type checking, will be replaced
     *  by dynamic type-id generation)
     */
    const char* model;
    
    /** 
     * The bus this chip is connected to
     */
    pp_tp_i2c_comdev_t* i2cdev;
    
    /**
     * The i2caddr of the chip on the 'comdev' bus
     */
    unsigned char i2caddr;

    /**
     * power up init condition, triggers init code when becoming true
     */
    pp_tp_cond_t* init_cond;

    /**
     * status whether initialization has been completed or not
     */
    int initialized;

    /**
     * subscriber interface for init condition
     */
    pp_tp_sensdev_subscriber_t cond_subs;

    /**
     * will be called for power up init
     */
    void (*power_up_init)(pp_tp_i2c_chip_t*);
    
    /**
     * will be called for power down cleanup
     */
    void (*power_down_cleanup)(pp_tp_i2c_chip_t*);
    
};

#define PP_TP_I2C_CHIP_GENERIC "generic"

/**
 * Construct a generic i2c-chip object,
 * maybe useful to spare some typing work
 */
pp_tp_obj_t* pp_tp_i2c_chip_ctor(const char* id, vector_t* args);

/**
 * Initialize a pp_tp_i2c_chip_t.
 */
void pp_tp_i2c_chip_init(pp_tp_i2c_chip_t* this, pp_tp_obj_type_t type,
			 const char* id, void (*dtor)(pp_tp_obj_t*),
			 const char* model, pp_tp_i2c_comdev_t* i2cdev,
			 unsigned char i2caddr, pp_tp_cond_t* power_cond,
                         void (*power_up_init)(pp_tp_i2c_chip_t*),
                         void (*power_down_cleanup)(pp_tp_i2c_chip_t*));
			 
/**
 * i2c_comdev cleanup. Childs can use this method to cleanup their superclass.
 */
void pp_tp_i2c_chip_cleanup(pp_tp_i2c_chip_t* d);

/*
 * some convience functions for calling the virtual object methods
 */
static inline pp_tp_i2c_chip_t*
pp_tp_i2c_chip_duplicate(pp_tp_i2c_chip_t* o) {
    return (pp_tp_i2c_chip_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)o);
}

static inline void pp_tp_i2c_chip_release(pp_tp_i2c_chip_t* o) {
    pp_tp_obj_release((pp_tp_obj_t*)o);
}


/*
 * class methods
 */

/**
 * Checks whether the chip has been initialized
 */
static inline int pp_tp_i2c_chip_is_initialized(pp_tp_i2c_chip_t* this) {
    return this->initialized;
}

/**
 * This function must be called before communication.
 */
static inline int pp_tp_i2c_chip_pre_com(pp_tp_i2c_chip_t* this) {
    return pp_tp_i2c_comdev_pre_com(this->i2cdev, this->i2caddr);
}

/**
 * This function must be called after communication.
 */
static inline int pp_tp_i2c_chip_post_com(pp_tp_i2c_chip_t* this) {
    return pp_tp_i2c_comdev_post_com(this->i2cdev, this->i2caddr);
}
    
/**
 * Receive a byte from the chip.
 */
static inline int pp_tp_i2c_chip_rx_byte(pp_tp_i2c_chip_t* this,
					  unsigned char* data) {
    return pp_tp_i2c_comdev_rx_byte(this->i2cdev, this->i2caddr, data);
}

/**
 * Send a byte to the chip.
 */
static inline int pp_tp_i2c_chip_tx_byte(pp_tp_i2c_chip_t* this,
					  unsigned char data) {
    return pp_tp_i2c_comdev_tx_byte(this->i2cdev, this->i2caddr, data);
}
    
/**
 * Receive a byte from a specific register of the chip.
 */
static inline int pp_tp_i2c_chip_rx_byte_data(pp_tp_i2c_chip_t* this,
					       unsigned char reg,
					       unsigned char* data) {
    return pp_tp_i2c_comdev_rx_byte_data(this->i2cdev, this->i2caddr, reg,
					 data);
}
                    
/**
 * Send a byte to a specific register of the chip.
 */
static inline int pp_tp_i2c_chip_tx_byte_data(pp_tp_i2c_chip_t* this,
					       unsigned char reg,
					       unsigned char data) {
    return pp_tp_i2c_comdev_tx_byte_data(this->i2cdev, this->i2caddr, reg,
					 data);
}

/**
 * Receive a word from a specific register of the chip.
 */
static inline int pp_tp_i2c_chip_rx_word_data(pp_tp_i2c_chip_t* this,
					       unsigned char reg,
					       unsigned short* data) {					       
    return pp_tp_i2c_comdev_rx_word_data(this->i2cdev, this->i2caddr, reg,
					 data);
}
                    
/**
 * Send a word to a specific register of the chip.
 */
static inline int pp_tp_i2c_chip_tx_word_data(pp_tp_i2c_chip_t* this,
					       unsigned char reg,
					       unsigned short data) {
    return pp_tp_i2c_comdev_tx_word_data(this->i2cdev, this->i2caddr, reg,
					 data);
}
#endif
