/**
 * tp_i2c_comdev.h
 *
 * Defines an abstract i2c communication device.
 * Can be mapped to i2c-busses, muxes, ...
 * 
 * (c) 2005 Peppercon AG, 3/9/2005, tbr@peppecon.de
 */

#ifndef __TP_I2C_COMDEV_H__
#define __TP_I2C_COMDEV_H__

#include <pp/bmc/topo_base_obj.h>

/*
 * abstract communication device for I2C
 */
typedef struct pp_tp_i2c_comdev_s {
    pp_tp_obj_t base;

    /**
     * returns the handle of the enclosed communication
     * device. In case there is no one -1 will be returned
     *
     * NOTE: DON'T use this functions, since it exposes
     * internal implementation details. It is currently used by
     * the IPMB channel only!
     */
    int (*root_handle)(struct pp_tp_i2c_comdev_s* dev);
    
    /**
     * This function must be called before communication.
     */
    int (*pre_com)(struct pp_tp_i2c_comdev_s*, unsigned char i2caddr);

    /**
     * This function must be called after communication.
     */
    int (*post_com)(struct pp_tp_i2c_comdev_s*, unsigned char i2caddr);
    
    /**
     * Receive a byte from the device.
     */
    int (*rx_byte)(struct pp_tp_i2c_comdev_s*, unsigned char i2caddr,
		   unsigned char* data);
    
    /**
     * Send a byte to the device.
     */
    int (*tx_byte)(struct pp_tp_i2c_comdev_s*, unsigned char i2caddr,
		   unsigned char data);
    
    /**
     * Receive a byte from a specific register of the device.
     * Assumes that the target has generically addressable registers.
     */
    int (*rx_byte_data)(struct pp_tp_i2c_comdev_s*, unsigned char i2caddr,
                    unsigned char reg, unsigned char* data);
                    
    /**
     * Send a byte to a specific register of the device.
     * Assumes that the target has generically addressable registers.
     */
    int (*tx_byte_data)(struct pp_tp_i2c_comdev_s*, unsigned char i2caddr,
                    unsigned char reg, unsigned char data);
	
    /**
     * Receive a word from a specific register of the device.
     * Assumes that the target has generically addressable registers.
     */
    int (*rx_word_data)(struct pp_tp_i2c_comdev_s*, unsigned char i2caddr,
                    unsigned char reg, unsigned short* data);
                    
    /**
     * Send a word to a specific register of the device.
     * Assumes that the target has generically addressable registers.
     */
    int (*tx_word_data)(struct pp_tp_i2c_comdev_s*, unsigned char i2caddr,
                    unsigned char reg, unsigned short data);
                    
} pp_tp_i2c_comdev_t;


/**
 * Initialize a pp_tp_i2c_comdev_t.
 */
void pp_tp_i2c_comdev_init(pp_tp_i2c_comdev_t* d,
                           pp_tp_obj_type_t type,
                           const char* id,
                           void (*dtor)(pp_tp_obj_t*),
			   int (*root_fd)(pp_tp_i2c_comdev_t* dev),
                           int (*pre_com)(pp_tp_i2c_comdev_t*, unsigned char),
                           int (*post_com)(pp_tp_i2c_comdev_t*, unsigned char),
                           int (*rx_byte)(pp_tp_i2c_comdev_t*,
					  unsigned char,
					  unsigned char*),
                           int (*tx_byte)(pp_tp_i2c_comdev_t*,
					  unsigned char,
					  unsigned char),			   
                           int (*rx_byte_data)(pp_tp_i2c_comdev_t*,
                                               unsigned char,
                                               unsigned char,
                                               unsigned char*),
                           int (*tx_byte_data)(pp_tp_i2c_comdev_t*,
                                               unsigned char,
                                               unsigned char,
                                               unsigned char),
                           int (*rx_word_data)(pp_tp_i2c_comdev_t*,
                                               unsigned char,
                                               unsigned char,
                                               unsigned short*),
                           int (*tx_word_data)(pp_tp_i2c_comdev_t*,
                                               unsigned char,
                                               unsigned char,
                                               unsigned short));

/**
 * i2c_comdev cleanup. Childs can use this method to cleanup their superclass.
 */
static inline void pp_tp_i2c_comdev_cleanup(pp_tp_i2c_comdev_t* d) {
    pp_tp_obj_cleanup((pp_tp_obj_t*)d);
}


/*
 * some convience functions for calling the virtual object methods
 */

static inline pp_tp_i2c_comdev_t*
pp_tp_i2c_comdev_duplicate(pp_tp_i2c_comdev_t* o) {
    return (pp_tp_i2c_comdev_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)o);
}

static inline void pp_tp_i2c_comdev_release(pp_tp_i2c_comdev_t* o) {
    pp_tp_obj_release((pp_tp_obj_t*)o);
}

static inline int pp_tp_i2c_comdev_root_handle(pp_tp_i2c_comdev_t* dev) {
    return dev->root_handle(dev);
}

static inline int pp_tp_i2c_comdev_pre_com(pp_tp_i2c_comdev_t* dev,
                                       unsigned char i2caddr) {
    return dev->pre_com(dev, i2caddr);
}

static inline int pp_tp_i2c_comdev_post_com(pp_tp_i2c_comdev_t* dev,
                                         unsigned char i2caddr) {
    return dev->post_com(dev, i2caddr);
}

static inline int
pp_tp_i2c_comdev_rx_byte(pp_tp_i2c_comdev_t* i2cdev,
			 unsigned char i2caddr,
			 unsigned char* data) {
    return i2cdev->rx_byte(i2cdev, i2caddr, data);
}

static inline
int pp_tp_i2c_comdev_tx_byte(pp_tp_i2c_comdev_t* i2cdev,
			     unsigned char i2caddr,
			     unsigned char data) {
    return i2cdev->tx_byte(i2cdev, i2caddr, data);
}

static inline int
pp_tp_i2c_comdev_rx_byte_data(pp_tp_i2c_comdev_t* i2cdev,
			      unsigned char i2caddr,
			      unsigned char reg, unsigned char* data) {
    return i2cdev->rx_byte_data(i2cdev, i2caddr, reg, data);
}

static inline
int pp_tp_i2c_comdev_tx_byte_data(pp_tp_i2c_comdev_t* i2cdev,
				  unsigned char i2caddr,
				  unsigned char reg,
				  unsigned char data) {
    return i2cdev->tx_byte_data(i2cdev, i2caddr, reg, data);
}

static inline int
pp_tp_i2c_comdev_rx_word_data(pp_tp_i2c_comdev_t* i2cdev,
			      unsigned char i2caddr,
			      unsigned char reg, unsigned short* data) {			      
    return i2cdev->rx_word_data(i2cdev, i2caddr, reg, data);
}

static inline
int pp_tp_i2c_comdev_tx_word_data(pp_tp_i2c_comdev_t* i2cdev,
				  unsigned char i2caddr,
				  unsigned char reg,
				  unsigned short data) {
    return i2cdev->tx_word_data(i2cdev, i2caddr, reg, data);
}

#endif
