/**
 * tp_ipmi_sens.h
 *
 * A generic IPMI sensor object based on a sensdev object.
 * 
 * (c) 2005 Peppercon AG, 3/9/2005, tbr@peppecon.de
 */

#ifndef __PP_BMC_TP_IPMI_SENS_H__
#define __PP_BMC_TP_IPMI_SENS_H__

#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/tp_sensdev.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_cmd_sensor.h>
#include <pp/bmc/tp_sens_subscriber.h>
#include <pp/bmc/tp_ipmi_sens_subscriber.h>
#include <pp/bmc/tp_sdr.h>
#include <pp/bmc/tp_cond.h>

#include <pp/vector.h>

/* declare vtable, used later on */
typedef struct pp_tp_ipmi_sens_vtable_s pp_tp_ipmi_sens_vtable_t;

/**
 * A generic ipmi sensor object.
 */
struct pp_tp_ipmi_sens_s {
    pp_tp_obj_t base;
    vector_t* sensor_subscribers;

    /*
     * vtable: used here since many objects will share exactly the same,
     *         quite large, vtable, so no need to waste a lot of memory.
     *         includes the sensor subscriber interface, used to receive
     *         reading from the underlying sensor device
     */
    pp_tp_ipmi_sens_vtable_t* vt;

    /** The real sensor this object abstracts */
    pp_tp_sensdev_t* sensor;
    
    /** The IPMI sensor's SDR, duplicate from that in SDRR */
    ipmi_sdr_header_t* sdr;
    
    /**
     * subscriber interface, enables active sensors to call us back 
     * when their reading changed
     */     
    pp_tp_sensdev_subscriber_t subscriber;

    /** condition to evaluate whether the sensor is there or not *
     *  this may dynamically change                              */
    pp_tp_cond_t* presence;

    /** subscriber to get notification if presence condition changes */
    pp_tp_sensdev_subscriber_t presence_subscriber;
    
    /** logical ipmi sensor number */
    unsigned char number;
};

/** sensor's vtable, defined seperately, in order to save some bytes */
struct pp_tp_ipmi_sens_vtable_s {

    /**
     * Tries to set a new SDR for the ISens and initializes ISens for
     * it. The call will check whether the SDR fits to the sensors and
     * rejects setting the SDR if id doesn't fit.
     * @param s the sensor's instance pointer
     * @param sdr the new SDR
     * @return  PP_OK or
     *          PP_ERR 
     */
    int (*set_sdr)(pp_tp_ipmi_sens_t *s, ipmi_sdr_header_t* sdr);
    
    /**
     * Initiate the reading of the sensor as per spec
     * @param s the sensor's instance pointer
     * @param resp_code pointer to IPMI response code, will be filled
     * @param resp_data pointer to IPMI response data, will be filled
     *                  if resp_code is equal IPMI_ERR_SUCCESS
     * @return PP_OK or
     *         PP_ERR 
     */
    int (*get_reading)(pp_tp_ipmi_sens_t *s, unsigned char* resp_code,
		       ipmi_sensor_get_reading_rs_t* resp_data);

    /**
     * Set sensor Hysteresis
     * @param s the sensor's instance pointer
     * @return PP_OK, function successfully finished, "es" is valid
     *         PP_ERR errno will be set
     */
    int (*set_hysteresis)(pp_tp_ipmi_sens_t* s,
			  ipmi_set_sensor_hysteresis_rq_t* hyst);

    /**
     * Get sensor Hysteresis
     * @param s the sensor's instance pointer
     * @return PP_OK, function successfully finished, "es" is valid
     *         PP_ERR errno will be set
     */
    int (*get_hysteresis)(pp_tp_ipmi_sens_t* s,
			  ipmi_get_sensor_hysteresis_rs_t* hyst);
	 
    /**
     * Set sensor thresholds
     * @param s the sensor's instance pointer
     * @return PP_OK, function successfully finished, "es" is valid
     *         PP_ERR errno will be set
     */
    int (*set_thresholds)(pp_tp_ipmi_sens_t* s,
			  ipmi_set_sensor_thresholds_rq_t* thresh);
    
    /**
     * Get sensor thresholds
     * @param s the sensor's instance pointer
     * @return PP_OK, function successfully finished, "es" is valid
     *         PP_ERR errno will be set
     */
    int (*get_thresholds)(pp_tp_ipmi_sens_t* s,
			  ipmi_get_sensor_thresholds_rs_t* thresh);
    
    /**
     * Set sensor event enable
     * @param s the sensor's instance pointer
     * @return PP_OK, function successfully finished, "es" is valid
     *         PP_ERR errno will be set
     */
    int (*set_event_enable)(pp_tp_ipmi_sens_t* s,
			   ipmi_set_sensor_event_enable_rq_t* evten);
	 
    /**
     * Get sensor event enable
     * @param s the sensor's instance pointer
     * @return PP_OK, function successfully finished, "es" is valid
     *         PP_ERR errno will be set
     */
    int (*get_event_enable)(pp_tp_ipmi_sens_t* s,
			    ipmi_get_sensor_event_enable_rs_t* evten);
    
    /**
     * Rearm sensor events
     * @param s the sensor's instance pointer
     * @return PP_OK, function successfully finished, "es" is valid
     *         PP_ERR errno will be set
     */
    int (*rearm_events)(pp_tp_ipmi_sens_t* s,
			ipmi_rearm_sensor_events_rq_t* rearm);
    
    /**
     * Query the event status as per spec
     * @param s the sensor's instance pointer
     * @param es IPMI response structure of event status
     * @return PP_OK, function successfully finished, "es" is valid
     *         PP_ERR errno will be set
     */
    int (*get_event_status)(pp_tp_ipmi_sens_t* s, 
			    ipmi_sensor_get_event_status_rs_t* es);
    
    /**
     * Get sensor reading factors as per spec
     * @param s the sensor's instance pointer
     * @param facts IPMI response structure of GetReadingFactors
     * @return PP_OK, function successfully finished, "facts" is valid
     *         PP_ERR errno will be set
     */
    int (*get_reading_factors)(pp_tp_ipmi_sens_t* s,
			       unsigned char reading,
			       ipmi_get_sensor_reading_factors_rs_t *facts);

    /**
     * Get sensor reading offset (qlogic oem command)
     * @param s the sensor's instance pointer
     * @return  the currently used offset value
     */    
    signed char (*get_reading_offset)(pp_tp_ipmi_sens_t* s);

    /**
     * Set sensor reading offset (qlogic oem command)
     * @param s the sensor's instance pointer
     * @param offset 8 bit signed offset added to the reading
     * @return PP_OK, function successfully finished
     *         PP_ERR errno will be set
     */
    int (*set_reading_offset)(pp_tp_ipmi_sens_t* s, signed char offset);
};

/**
 * add a subscriber, the subscriber will be notified whenever the
 * sensor event mask changes. The subscriber object will be duplicated.
 * @returns the listener subscription id
 */
void pp_tp_ipmi_sens_subscribe(pp_tp_ipmi_sens_t* this,
			       pp_tp_ipmi_sens_subscriber_t* subscriber);

/**
 * remove a subscriber, release the subscriber object.
 * @returns PP_ERR if the listener is not known
 */
int pp_tp_ipmi_sens_unsubscribe(pp_tp_ipmi_sens_t* this,
				pp_tp_ipmi_sens_subscriber_t* subscriber);

/**
 * can be called by derived classes to notify all subscribers of
 * a new sensor reading.
 */
void pp_tp_ipmi_sens_notify_subscribers(pp_tp_ipmi_sens_t* this, int value);

/*
 * the isens subscriber interface marks the reading as
 * assertion/deassertion, here are macros to check and set this
 */
#define PP_ISENS_NOTIFY_DEASSERTION(x)		((x) | 0x80000000)
#define PP_ISENS_NOTIFY_IS_ASSERTION(x)		((reading & 0x80000000) ? 0 : 1)
#define PP_ISENS_NOTIFY_IS_DEASSERTION(x)	((reading & 0x80000000) ? 1 : 0)

/*
 * IPMI Sensor that supports the threshold reading/event type 
 * named Threshold sensor class in the spec (42.1)
 */

typedef struct {
    pp_tp_ipmi_sens_t base;

    /** Thresholds */
    ipmi_sdr_threshold_t threshold;

    /** Offset value added to the reading */
    signed char offset;

    /** Events */
    struct { // see IPMI_SENSOR_EVENT_xxxx
        struct {
            unsigned short enabled;
            unsigned short fired;
        } assertion;
        struct {
            unsigned short enabled;
            unsigned short fired;
        } deassertion;

	/** A bit mask where each bit represents
	 *  one of the sensors discretized values. */
        unsigned short state;
    } event;

    unsigned char auto_rearm;
    unsigned char scanning_enabled;
    unsigned char all_events_enabled;
    
} pp_tp_thresh_ipmi_sens_t;

typedef struct {
    pp_tp_ipmi_sens_t base;

    /** Events */
    struct {
        struct {
            unsigned short enabled;
            unsigned short fired;
        } assertion;
        struct {
            unsigned short enabled;
            unsigned short fired;
        } deassertion;

	unsigned short state;
    } event;   

    unsigned char auto_rearm;
    unsigned char scanning_enabled;
    unsigned char all_events_enabled;

} pp_tp_disc_ipmi_sens_t;

/* ppdoc
 * name:   IPMI Sensor object
 * tpname: ISens
 * desc:   An object that exposes an internal sensor through IPMI.
 * arg 0:  [OBJ(SEN_DEV)]:    a concrete sensor device
 * arg 1:  [INT] sens_no:     ipmi sensor dev sensor number
 * arg 2:  [STRING] name:     sensor name
 * arg 3:  [INT] entity_id:   entity id
 * arg 4:  [INT] entity_inst: entity instance
 * arg 5:  [OBJ(CFG_SDR)] sdr: SDR config object
 * arg 6:  [OBJ(COND)] presence: (optional) presence condition
 * return: [OBJ(IPMI_SENSOR)]
 */
pp_tp_obj_t* pp_tp_ipmi_sens_ctor(const char* id, vector_t* args);

/* ppdoc
 * name:   IPMI full SDR sensor
 * tpname: ISensFullSDR
 * desc:   ipmi sensor created with full sdr configuration data
 * arg 0:  [OBJ(SEN_DEV)] sensor: a concrete sensor device
 * arg 1:  [INT] num: sensor device number              (full sdr byte 8)
 * arg 2:  [STRING] ids: ID string/name              (full sdr byte 49+N)
 * arg 3:  [INT] eid: entity id			        (full sdr byte 9)
 * arg 4:  [INT] ein: entity instance,		       (full sdr byte 10)
 * arg 4:        Bit0-6->instance number,
 * arg 4:        Bit7  ->0b = physical
 * arg 4:                1b = logical container entity
 * arg 5:  [INT] cap: 1 byte sensor capabilities       (full sdr byte 12)
 * arg 6:  [INT] sty: sensor type		       (full sdr byte 13)
 * arg 7:  [INT] erc: event/reading type code	       (full sdr byte 14)
 * arg 7:
 * arg 8:  [INT] ltm: 2 bytes lower threshold/assertion event mask (full sdr byte 15+16)
 * arg 9:  [INT] utm: 2 bytes upper threshold/deassertion event mask (full sdr byte 17+18)
 * arg 10: [INT] stm: 1 byte setable threshold mask    (full sdr byte 19)
 * arg 11: [INT] rtm: 1 byte readable threshold mask   (full sdr byte 20)
 * arg 11:
 * arg 12: [INT] adf: analog data format         (full sdr byte 21 [7:6])
 * arg 13: [INT] run: rate unit			 (full sdr byte 21 [5:3])
 * arg 14: [INT] mnc: modifier unit config       (full sdr byte 21 [2:1])
 * arg 15: [INT] pct: percentage                   (full sdr byte 21 [0])
 * arg 16: [INT] bun: base unit                        (full sdr byte 22)
 * arg 17: [INT] mun: modifier unit                    (full sdr byte 23)
 * arg 17:
 * arg 18: [INT] lin: linearization                    (full sdr byte 24)
 * arg 19: [INT] m  : M                      (full sdr byte 25 & 26[7:6])
 * arg 20: [INT] tol: tolerance                   (full sdr byte 26[5:0])
 * arg 21: [INT] b  : B                      (full sdr byte 27 & 28[7:6])
 * arg 22: [INT] acc: accuracy          (full sdr byte 28[5:0] & 29[7:4])
 * arg 23: [INT] aex: accuracy exponent           (full sdr byte 29[3:2])
 * arg 24: [INT] sed: sensor direction            (full sdr byte 29[1:0])
 * arg 25: [INT] rex: R (result) exponent         (full sdr byte 30[7:4])
 * arg 26: [INT] bex: B exponent                  (full sdr byte 30[3:0])
 * arg 26:
 * arg 27: [INT] acf: analog characteristic flags      (full sdr byte 31)
 * arg 27:
 * arg 28: [INT] nrd: 1 byte nominal_read              (full sdr byte 32)
 * arg 29: [INT] nma: 1 byte normal_max                (full sdr byte 33)
 * arg 30: [INT] nmi: 1 byte normal_min                (full sdr byte 34)
 * arg 31: [INT] sma: 1 byte sensor maximum reading    (full sdr byte 35)
 * arg 32: [INT] smi: 1 byte sensor minimum reading    (full sdr byte 36)
 * arg 33: [INT] nru: 1 byte non_recover upper thresh  (full sdr byte 37)
 * arg 34: [INT] cru: 1 byte critical upper thresh     (full sdr byte 38)
 * arg 35: [INT] ncu: 1 byte non_critical upper thresh (full sdr byte 39)
 * arg 36: [INT] nrl: 1 byte non_recover lower thresh  (full sdr byte 40)
 * arg 37: [INT] crl: 1 byte critical lower thresh     (full sdr byte 41)
 * arg 38: [INT] ncl: 1 byte non_critical lower thresh (full sdr byte 42)
 * arg 39: [INT] pos: 1 byte positive hysteresis       (full sdr byte 43)
 * arg 40: [INT] neg: 1 byte negative hysteresis       (full sdr byte 44)
 * arg 41: [INT] oem: for oem use                      (full sdr byte 47)
 * arg 41:
 * arg 42: [OBJ(COND)] presence: (optional) presence condition
 * return: [OBJ(TRESH_SDR)]
 */
pp_tp_obj_t* pp_tp_ipmi_sens_full_sdr_ctor(const char* id, vector_t* args);


/* ppdoc
 * name:   IPMI compact SDR sensor
 * tpname: ISensCompactSDR
 * desc:   ipmi sensor created with compact sdr configuration data
 * arg 0:  [OBJ(SEN_DEV)] sensor: a concrete sensor device
 * arg 1:  [INT] num: ipmi sensor dev sensor number     (compact sdr byte 8)
 * arg 2:  [STRING] ids: ID string/name              (compact sdr byte 49+N)
 * arg 3:  [INT] eid: entity id			        (compact sdr byte 9)
 * arg 4:  [INT] ein: entity instance,		       (compact sdr byte 10)
 * arg 4:        Bit0-6->instance number,
 * arg 4:        Bit7  ->0b = physical
 * arg 4:                1b = logical container entity
 * arg 5:  [INT] cap: 1 byte sensor capabilities       (compact sdr byte 12)
 * arg 6:  [INT] sty: sensor type		       (compact sdr byte 13)
 * arg 7:  [INT] erc: event/reading type code	       (compact sdr byte 14)
 * arg 7: 
 * arg 8:  [INT] eam: 2 bytes assertion event mask    (compact sdr byte 15,16)
 * arg 9:  [INT] edm: 2 bytes deassertion event mask  (compact sdr byte 17,18)
 * arg 10: [INT] rdm: 2 bytes discrete reading mask   (compact sdr byte 19,20)
 * arg 10:
 * arg 11: [INT] adf: analog data format         (compact sdr byte 21 [7:6])
 * arg 12: [INT] run: rate unit			 (compact sdr byte 21 [5:3])
 * arg 13: [INT] mnc: modifier unit config       (compact sdr byte 21 [2:1])
 * arg 14: [INT] per: percentage                   (compact sdr byte 21 [0])
 * arg 15: [INT] bun: base unit                        (compact sdr byte 22)
 * arg 16: [INT] mun: modifier unit                    (compact sdr byte 23)
 * arg 16:
 * arg 17: [INT] sed: sensor direction            (compact sdr byte 24[7:6])
 * arg 17:
 * arg 18: [INT] ist: id string inst. mod. type  (compact sdr byte 24 [5:4])
 * arg 19: [INT] shc: share count                (compact sdr byte 24 [3:0])
 * arg 20: [INT] eis: entity instance sharing      (compact sdr byte 25 [7])
 * arg 21: [INT] iso: id string inst. mod. offset(compact sdr byte 25 [6:0])
 * arg 21:
 * arg 22: [INT] pos: 1 byte positive hysteresis       (compact sdr byte 43)
 * arg 23: [INT] neg: 1 byte negative hysteresis       (compact sdr byte 44)
 * arg 24: [INT] oem: reserved for oem use             (compact sdr byte 47)
 * arg 24:
 * arg 25: [OBJ(COND)] presence: (optional) presence condition
 * return: [OBJ(TRESH_SDR)]
 */
pp_tp_obj_t* pp_tp_ipmi_sens_compact_sdr_ctor(const char* id, vector_t* args);

/**
 * some convenience functions
 */
void pp_tp_ipmi_sens_init(pp_tp_ipmi_sens_t* o, pp_tp_obj_type_t type, 
			  const char* id, pp_tp_obj_dtor_func_t dtor, 
			  pp_tp_ipmi_sens_vtable_t* vt,
			  pp_tp_sensdev_t* sensor, u_char number,
			  ipmi_sdr_header_t* sdr, pp_tp_cond_t* presence_cond,
			  pp_tp_sensdev_subscriber_recv_reading_func_t recv_reading,
			  pp_tp_sensdev_subscriber_recv_reading_func_t recv_presence_reading);

void pp_tp_ipmi_sens_cleanup(pp_tp_ipmi_sens_t* o);

static inline pp_tp_ipmi_sens_t*
pp_tp_ipmi_sens_duplicate(pp_tp_ipmi_sens_t* this)
{
    return (pp_tp_ipmi_sens_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)this);
}

static inline void
pp_tp_ipmi_sens_release(pp_tp_ipmi_sens_t* this)
{
    pp_tp_obj_release((pp_tp_obj_t*)this);
}

static inline const char*
pp_tp_ipmi_sens_to_string(pp_tp_ipmi_sens_t* this)
{
    return this->base.to_string(&this->base);
}

static inline ipmi_sdr_header_t*
pp_tp_ipmi_sens_get_sdr(pp_tp_ipmi_sens_t* s)
{
    return s->sdr;
}

static inline int
pp_tp_ipmi_sens_set_sdr(pp_tp_ipmi_sens_t* this, ipmi_sdr_header_t* sdr)
{
    return this->vt->set_sdr(this, sdr);
}

static inline int
pp_tp_ipmi_sens_get_reading(pp_tp_ipmi_sens_t *this, unsigned char* resp_code,
			    ipmi_sensor_get_reading_rs_t* resp_data) {
    return this->vt->get_reading(this, resp_code, resp_data);
}

static inline int
pp_tp_ipmi_sens_set_hysteresis(pp_tp_ipmi_sens_t* this,
			       ipmi_set_sensor_hysteresis_rq_t* hys)
{
    if (!this->vt->set_hysteresis) { errno = ENOSYS; return PP_ERR; }
    return this->vt->set_hysteresis(this, hys);
}

static inline int
pp_tp_ipmi_sens_get_hysteresis(pp_tp_ipmi_sens_t* this,
			       ipmi_get_sensor_hysteresis_rs_t* hyst)
{
    if (!this->vt->get_hysteresis) { errno = ENOSYS; return PP_ERR; }
    return this->vt->get_hysteresis(this, hyst);
}

static inline int
pp_tp_ipmi_sens_set_thresholds(pp_tp_ipmi_sens_t* this,
			       ipmi_set_sensor_thresholds_rq_t* thresh)
{
    if (!this->vt->set_thresholds) { errno = ENOSYS; return PP_ERR; }
    return this->vt->set_thresholds(this, thresh);
}


static inline int
pp_tp_ipmi_sens_get_thresholds(pp_tp_ipmi_sens_t* this,
			       ipmi_get_sensor_thresholds_rs_t* thresh) {
    if (!this->vt->get_thresholds) { errno = ENOSYS; return PP_ERR; }
    return this->vt->get_thresholds(this, thresh);
}


static inline int
pp_tp_ipmi_sens_set_event_enable(pp_tp_ipmi_sens_t* this,
				 ipmi_set_sensor_event_enable_rq_t* et)
{
    if (!this->vt->set_event_enable) { errno = ENOSYS; return PP_ERR; }
    return this->vt->set_event_enable(this, et);
}


static inline int
pp_tp_ipmi_sens_get_event_enable(pp_tp_ipmi_sens_t* this,
				 ipmi_get_sensor_event_enable_rs_t* et)
{
    if (!this->vt->get_event_enable) { errno = ENOSYS; return PP_ERR; }
    return this->vt->get_event_enable(this, et);
}

static inline int
pp_tp_ipmi_sens_rearm_events(pp_tp_ipmi_sens_t* this,
			     ipmi_rearm_sensor_events_rq_t* rearm)
{
    if (!this->vt->rearm_events) { errno = ENOSYS; return PP_ERR; }
    return this->vt->rearm_events(this, rearm);
}

static inline int
pp_tp_ipmi_sens_get_event_status(pp_tp_ipmi_sens_t* this,
				 ipmi_sensor_get_event_status_rs_t* events)
{
    if (!this->vt->get_event_status) { errno = ENOSYS; return PP_ERR; }
    return this->vt->get_event_status(this, events);
}

static inline int
pp_tp_ipmi_sens_get_reading_factors(pp_tp_ipmi_sens_t* this, u_char reading,
				    ipmi_get_sensor_reading_factors_rs_t* facts)
{
    if (!this->vt->get_reading_factors) { errno = ENOSYS; return PP_ERR; }
    return this->vt->get_reading_factors(this, reading, facts);
}

static inline char
pp_bmc_tp_ipmi_sens_get_reading_offset(pp_tp_ipmi_sens_t* this)
{
    if (!this->vt->get_reading_offset) { errno = ENOSYS; return 0; }
    return this->vt->get_reading_offset(this);
}

static inline int
pp_bmc_tp_ipmi_sens_set_reading_offset(pp_tp_ipmi_sens_t* this, char offset)
{
    if (!this->vt->set_reading_offset) { errno = ENOSYS; return PP_ERR; }
    return this->vt->set_reading_offset(this, offset);
}
   
#endif /* __PP_BMC_TP_IPMI_SENS_H__ */
