/**
 * tp_rs485_chip.h
 *
 * Interface for chip having rs485 interface and
 * register bank access interface.
 *
 * (c) 2006 Peppercon AG, 2006/09/05 tbr@raritan.com
 */

#ifndef __PP_BMC_TP_RS485_CHIP_H__
#define __PP_BMC_TP_RS485_CHIP_H__

#include <pp/vector.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/tp_rs485_comdev.h>

/**
 * register bank subscriber interface
 */
typedef struct pp_tp_reg_subscriber_s {
    /**
     * called back from pp_tp_rs485_chip object in case
     * value at subscribed address has changed
     */
    void (*reg_changed)(struct pp_tp_reg_subscriber_s* s,
			unsigned char addr, unsigned char val);
} pp_tp_reg_subscriber_t;

/*
 * abstract RS485 chip object implementing the
 * a register access protocoll
 */
typedef struct pp_tp_rs485_chip_s {

    pp_tp_obj_t base;

    /**
     * this chip's model string (usally statically allocated)
     * (currently used for extended type checking, will be replaced
     *  by dynamic type-id generation)
     */
    const char* model;
    
    vector_t* reg_subscribers;

    /**
     * bus we are connected to
     */
    pp_tp_rs485_comdev_t* rs485dev;

    /**
     * address we have on the bus
     */
    unsigned char rs485addr;    

    /**
     * get value of register
     */
    int (*get_byte)(struct pp_tp_rs485_chip_s* this,
		    unsigned char addr, unsigned char* val);

    /**
     * set value of register
     */
    int (*set_byte)(struct pp_tp_rs485_chip_s* this,
		    unsigned char addr, unsigned char val);

    /**
     * update firmware
     */
    int (*fwupdate)(struct pp_tp_rs485_chip_s* this,
		    const char* data, size_t len);
    
} pp_tp_rs485_chip_t;

/**
 * rs485_chip initializer
 */
void pp_tp_rs485_chip_init(pp_tp_rs485_chip_t* this,
			   pp_tp_obj_type_t type,
			   const char* id,
			   void (*dtor)(pp_tp_obj_t*),
			   const char* model,
			   pp_tp_rs485_comdev_t* rs485dev,
			   unsigned char rs485addr,
			   int (*get_byte)(pp_tp_rs485_chip_t*,
					   unsigned char, unsigned char*),
			   int (*set_byte)(pp_tp_rs485_chip_t*,
					   unsigned char, unsigned char),
			   int (*fwupdate)(pp_tp_rs485_chip_t*,
					   const char* data, size_t len));
/**
 * rs485_chip cleanup
 */
void pp_tp_rs485_chip_cleanup(pp_tp_rs485_chip_t* this);

/*
 * object methods
 */

static inline pp_tp_rs485_chip_t*
pp_tp_rs485_chip_duplicate(pp_tp_rs485_chip_t* this) {
    return (pp_tp_rs485_chip_t*)pp_tp_obj_duplicate(&this->base);
}

static inline void
pp_tp_rs485_chip_release(pp_tp_rs485_chip_t* this) {
    pp_tp_obj_release(&this->base);
}

static inline const char*
pp_tp_rs485_chip_to_string(pp_tp_rs485_chip_t* this) {
    return pp_tp_obj_to_string(&this->base);
}

static inline int 
pp_tp_rs485_chip_get_byte(pp_tp_rs485_chip_t* this,
			  unsigned char addr, unsigned char* val) {
    return this->get_byte(this, addr, val);
}

static inline int
pp_tp_rs485_chip_set_byte(pp_tp_rs485_chip_t* this,
			  unsigned char addr, unsigned char val) {
    return this->set_byte(this, addr, val);
}

static inline int
pp_tp_rs485_chip_fwupdate(pp_tp_rs485_chip_t* this,
			  const char* data, size_t len) {
    if (this->fwupdate == NULL) return PP_ERR;
    return this->fwupdate(this, data, len);
}

/**
 * register subscribers for specified address
 */
void pp_tp_rs485_chip_register(pp_tp_rs485_chip_t* this,
			      pp_tp_reg_subscriber_t* subs,
			      unsigned char addr);

/**
 * unregister subscribers for specified address
 * returns the number of times this subscriber was registered
 */
int pp_tp_rs485_chip_unregister(pp_tp_rs485_chip_t* this,
				pp_tp_reg_subscriber_t* subs);

/**
 * notifies all registered listeners
 */
void pp_tp_rs485_chip_notify_subscribers(pp_tp_rs485_chip_t* this,
					 unsigned char addr,
					 unsigned char val);

#endif
