/**
 * tp_rs485_comdev.h
 *
 * abstract RS485 communication object
 *
 * (c) 2006 Peppercon AG, 2006/09/05 tbr@raritan.com
 */

#ifndef __PP_BMC_TP_RS485_COMDEV_H__
#define __PP_BMC_TP_RS485_COMDEV_H__

#include <pp/bmc/topo_base_obj.h>

typedef struct pp_tp_rs485_comdev_s {
    pp_tp_obj_t base;

    /**
     * reads specified number of bytes at specified address of RS485 device
     * @param this       this pointer
     * @param rs485addr  address of device on RS485 bus
     * @param addr       register address in device from which to start read
     * @param len        number of bytes to read
     * @param buf        buffer where to store read data
     * @return           PP_SUC or PP_ERR
     */
    int (*read_data)(struct pp_tp_rs485_comdev_s* this,
		     unsigned char rs485addr,
		     unsigned char addr, unsigned char len,
		     unsigned char* buf);
    
    /**
     * writes specified number of bytes to specified address to RS485 device
     * @param dev        this pointer
     * @param rs485addr  address of device on RS485 bus
     * @param addr       register address in device from which to start write
     * @param len        number of bytes to write
     * @param buf        buffer where to write data from
     * @return           PP_SUC or PP_ERR
     */
    int (*write_data)(struct pp_tp_rs485_comdev_s* this,
		      unsigned char rs485addr,
		      unsigned char addr, unsigned char len,
		      unsigned char* buf);

    /**
     * sends a firmware update block to RS485 device
     * @param dev        this pointer
     * @param rs485addr  address of device on RS485 bus
     * @param addr       register address in device from which to start write
     * @param len        number of bytes to write
     * @param buf        buffer where to write data from
     * @param timeout    timeout in ms
     * @return           PP_ERR or status
     */
    int (*write_fwupd)(struct pp_tp_rs485_comdev_s* this,
		       unsigned char rs485addr,
		       const char* buf, unsigned char len, int timeout);
    
} pp_tp_rs485_comdev_t;

void pp_tp_rs485_comdev_init(pp_tp_rs485_comdev_t* this,
			     pp_tp_obj_type_t type,
			     const char* id,
			     void (*dtor)(pp_tp_obj_t*),
			     int  (*read_data)(pp_tp_rs485_comdev_t* this,
					       unsigned char rs485addr,
					       unsigned char addr,
					       unsigned char len,
					       unsigned char* buf),
			     int  (*write_data)(pp_tp_rs485_comdev_t* this,
						unsigned char rs485addr,
						unsigned char addr,
						unsigned char len,
						unsigned char* buf),
			     int (*write_fwupd)(pp_tp_rs485_comdev_t* this,
						unsigned char rs485addr,
						const char* buf,
						unsigned char len,
						int timeout));

static inline void
pp_tp_rs485_comdev_cleanup(pp_tp_rs485_comdev_t* o) {
    pp_tp_obj_cleanup((pp_tp_obj_t*)o);
}

static inline pp_tp_rs485_comdev_t*
pp_tp_rs485_comdev_duplicate(pp_tp_rs485_comdev_t* o) {
    return (pp_tp_rs485_comdev_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)o);
}

static inline void
pp_tp_rs485_comdev_release(pp_tp_rs485_comdev_t* o) {
    pp_tp_obj_release((pp_tp_obj_t*)o);
}

static inline const char*
pp_tp_rs485_comdev_to_string(pp_tp_rs485_comdev_t* o) {
    return pp_tp_obj_to_string((pp_tp_obj_t*)o);
}

static inline int
pp_tp_rs485_comdev_read_data(pp_tp_rs485_comdev_t* this,
			     unsigned char rs485addr,
			     unsigned char addr, unsigned char len,
			     unsigned char* buf) {
    return this->read_data(this, rs485addr, addr, len, buf);
}
    
static inline int
pp_tp_rs485_comdev_write_data(pp_tp_rs485_comdev_t* this,
			      unsigned char rs485addr,
			      unsigned char addr, unsigned char len,
			      unsigned char* buf) {
    return this->write_data(this, rs485addr, addr, len, buf);
}

static inline int
pp_tp_rs485_comdev_write_fwupd(pp_tp_rs485_comdev_t* this,
			       unsigned char rs485addr, const char* buf,
			       unsigned char len, int timeout) {
    return this->write_fwupd(this, rs485addr, buf, len, timeout);
}


#endif /* __PP_BMC_TP_RS485_COMDEV_H__ */
