/**
 * tp_sensdev.h
 *
 * Basic abstract sensor device object.
 * 
 * (c) 2005 Peppercon AG, 3/9/2005, tbr@peppecon.de
 */

#ifndef _PP_BMC_TP_SENSDEV_H
#define _PP_BMC_TP_SENSDEV_H

#include <pp/vector.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/tp_sens_subscriber.h>
#include <pp/bmc/ipmi_sdr.h>

typedef ipmi_sdr_header_t*
      (*pp_tp_sensdev_default_sdr_func_t)(pp_tp_sensdev_t *);
typedef int (*pp_tp_sensdev_get_reading_func_t)(pp_tp_sensdev_t*);

/**
 * Abstract sensor device object
 * Each sensor is an active object and maintains a list of
 * sensor subscribers, that get notified if reading value changes
 */
struct pp_tp_sensdev_s {
    pp_tp_obj_t base;

    /**
     * list of subscribers
     */
    vector_t* subscribers;

    /**
     * reading value cache, the get_reading function will simply return
     * this value. I.e. get_reading is cheap and may be called frequently
     * reading is set to -1 (reading unavailable), during initialization.
     */
    int reading;
    
    /**
     * Creates the default SDR for this sensor, as far as this is known
     * by the implementation. Usually this includes sensor type, reading
     * type, etc...
     */
    pp_tp_sensdev_default_sdr_func_t default_sdr;
};

/**
 * sensor device initializer
 */
int pp_tp_sensdev_init(pp_tp_sensdev_t* this, pp_tp_obj_type_t type,
		       const char* id, pp_tp_obj_dtor_func_t dtor,
		       pp_tp_sensdev_default_sdr_func_t default_sdr);

/**
 * sensor device cleanup, can be used by childs to cleanup superclass.
 */
static inline void pp_tp_sensdev_cleanup(pp_tp_sensdev_t* this) {
    vector_delete(this->subscribers);
    pp_tp_obj_cleanup((pp_tp_obj_t*)this);
}

/* Convenience functions */

static inline pp_tp_sensdev_t*
pp_tp_sensdev_duplicate(pp_tp_sensdev_t* this) {
    return (pp_tp_sensdev_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)this);
}

static inline void pp_tp_sensdev_release(pp_tp_sensdev_t* this) {
    pp_tp_obj_release((pp_tp_obj_t*)this);
}

/**
 * Get the sensor reading. The result is written to *value.
 * 
 * @returns the sensor reading, 0..255 for ipmi threshold sensors
 *          or a bitmask of 15 bit for discrete sensors
 * @returns -1 if the sensor could not be read (PP_ERR)
 * @returns -2 if the sensor is not available (IPMI_ERR_DEST_UNAVAILABLE)
 */
static inline int pp_tp_sensdev_get_reading(pp_tp_sensdev_t *this) {
    return this->reading;
}

/** 
 * Creates a default SDR for this sensor.
 * Note: the function may also return NULL!
 */
static inline ipmi_sdr_header_t*
pp_bmc_tp_sensdev_create_default_sdr(pp_tp_sensdev_t* this) {
    ipmi_sdr_header_t* sdr;
    if (this->default_sdr != NULL) sdr = this->default_sdr(this);
    else sdr = NULL;
    return sdr;
}   

/**
 * add a subscriber, the subscriber will be notified whenever the
 * sensor value changes. The subscriber object will be duplicated.
 * @returns the listener subscription id
 */
void pp_bmc_tp_sensdev_subscribe(pp_tp_sensdev_t*, pp_tp_sensdev_subscriber_t*);

/**
 * remove a subscriber, release the subscriber object.
 * @returns PP_ERR if the listener is not known
 */
int pp_bmc_tp_sensdev_unsubscribe(pp_tp_sensdev_t*, pp_tp_sensdev_subscriber_t*);

/**
 * can be called by derived classes to notify all subscribers of
 * a new sensor reading.
 */
void pp_bmc_tp_sensdev_notify_subscribers(pp_tp_sensdev_t*, int);

#endif /* _PP_BMC_TP_SENSDEV_H */
