/**
 * sensor_scanner.h
 *
 * scans scanables peridically within another thread
 * 
 * (c) 2005 Peppercon AG, thomas@peppercon.de
 */

#ifndef __PP_BMC_SENSOR_SCANNER__
#define __PP_BMC_SENSOR_SCANNER__

/*
 * Note:
 * Phylosophy of scannable and startable is as follows:
 *
 * Sensor lib initialization happens in two phases:
 *
 *  1. topology instantiation
 *  2. topology activation
 * 
 * During initialization all sensor readings will be initialized
 * with -1 (not available). When activated, scanner thread will
 * start updating the sensor what triggers the sensor subscriber
 * callback.
 *
 * However, there are sensors that create its reading in another
 * way, for instance a StaticCondition or ActiveSensor.
 * In order to align this kind of sensors with the -1 initialization magic,
 * the scanner will 'start' those sensors once using the startable interface.
 */

/**
 * interface used to activate objects that do not need to be scanned
 */
typedef struct pp_sensor_startable_s {
    void (*start)(struct pp_sensor_startable_s* this);
} pp_sensor_startable_t;

/**
 * interface used by the scanner thread
 *
 * Note: the update function will be called by the scanner thread.
 *       this thread is different from the BMC main thread, so you have
 *       to be careful while accessing other topology objects.
 *       The only objects that are allowed to be accessed via their
 *       functions (don't use elements directly) are
 *         - PP_TP_I2C_COM_DEV
 *         - PP_TP_I2C_CHIP (the i2c communication functions)
 *         - PP_TP_COND
 *       Using other topology objects will suffer from race conditions,
 *       better don't do it at all!
 */
typedef struct pp_sensor_scannable_s {
    void (*update)(struct pp_sensor_scannable_s* this);
} pp_sensor_scannable_t;

/**
 * initializes sensor scanner
 */
void pp_sensor_scanner_init(void);

/**
 * cleans up sensor scanner
 */
void pp_sensor_scanner_cleanup(void);

/*
 * for starting and stopping the sensor thread,
 * look int pp/bmc/sensor.h, please
 */

/**
 * register an interface for scanning
 */
void pp_sensor_scanner_add(pp_sensor_scannable_t* s, int scan_group);

/**
 * deregisters an interface from scanning
 *
 * Note:
 * This function is not thread-safe among multiple threads
 * calling simultaneously remove on the same scannable.
 * If such an condition is detected the function will fail an
 * assertion if build with debug
 *
 * @return PP_OK if successfully deregisterd
 *         PP_ERR if s was not found
 */
int pp_sensor_scanner_remove(pp_sensor_scannable_t* s);

/**
 * register an interface for start.
 * will be called once when pp_bmc_sensor_scanner_start is called
 *
 * ATTENTION: this function is not thread save and meant to
 *            be called by the main bmc thread only
 */
void pp_sensor_scanner_add_startable(pp_sensor_startable_t* s);

/**
 * deregister an interface for start.
 * this will only have effect in case pp_bmc_sensor_scanner_start
 * hasn't been called yet.
 *
 * ATTENTION: this function is not thread save and meant to
 *            be called by the main bmc thread only
 */
int pp_sensor_scanner_remove_startable(pp_sensor_startable_t* s);

#endif /* __PP_BMC_SENSOR_SCANNER__ */
