/**
 * Tojo_parser.c
 *
 * implementation helpers for topology parser
 * 
 * (c) 2005 Peppercon AG, 3/9/2005, tbr@peppecon.de
 */

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include "topo_parser_impl.h"

/*
 * private method prototypes
 * -------------------------
 */
static arg_list_t* arg_list_merge(arg_list_t* bl, arg_list_t* sl);
static int tp_def_ctor_add(pp_tp_def_t* tp, const char* id, obj_ctor_t* ctor);
static int get_unnamed_cnt(pp_tp_def_t* tp, const char* id);

/*
 * public method implementation
 * -----------------------------
 */
pp_tp_def_t* pp_tp_def_create(pp_tp_obj_ctor_def_t* ctor, int no_ctors) {
    pp_tp_def_t* tp = malloc(sizeof(pp_tp_def_t));
    int i;
    
    tp->objctors = vector_new(NULL, 200, NULL);
    tp->ctorsmap = pp_hash_create(200);
    tp->aliasmap = pp_hash_create(100);
    tp->unnamedmap = pp_hash_create(100);

    for (i = 0; i < no_ctors; ++i) {
	obj_ctor_t* c = obj_ctor_create(strdup(ctor[i].name), ctor[i].method, 
					NULL);
	if (PP_ERR == tp_def_ctor_add(tp, ctor[i].name, c)) {
	    pp_bmc_log_error("pp_tp_def_create: dupliate c'tor symbol '%s'",
			     ctor[i].name);
	}
    }
    return tp;
}

void pp_tp_def_destroy(pp_tp_def_t* tp) {
    if (tp != NULL) {
	pp_hash_delete(tp->unnamedmap);
	pp_hash_delete(tp->aliasmap);
	pp_hash_delete(tp->ctorsmap);
	vector_delete(tp->objctors);
	free(tp);
    }
}

int pp_tp_def_incarnate(pp_tp_def_t* tp, pp_tp_incarnate_cb_t obj_cb,
			va_list cb_ap) {
    const char* fn = __FUNCTION__;
    int i, k, as, ret = PP_SUC;
    obj_ctor_t* ctor;
    pp_tp_arg_t* arg;
    pp_tp_obj_t* obj;
    int ovs = vector_size(tp->objctors);
    pp_hash_t* objs = pp_hash_create(200);
    va_list ap;

    for (i = 0; i < ovs; ++i) {
	ctor = vector_get(tp->objctors, i);
	assert(!CTOR_IS_TEMPLATE(ctor));
	
	// resolve objects in argument vector
	as = vector_size(&ctor->arglist->args);
	for (k = 0; k < as; ++k) {
	    arg = vector_get2(&ctor->arglist->args, k);
	    if (arg->type == PP_TP_ARG_SYMBOL) {
		obj = (pp_tp_obj_t*)pp_hash_get_entry(objs, arg->value.string);
		if (NULL == obj) {
		    pp_bmc_log_warn("%s: WARNING: incarnation of '%s' skipped:"
				   " can't resolve '%s'!",
				    fn, ctor->name, arg->value.string);
		    goto continue_outer;
		}
		free(arg->value.string);
		arg->type = PP_TP_ARG_OBJ;
		arg->value.obj = pp_tp_obj_duplicate(obj);
	    }
	}
	
	// call constructor
	if (NULL == (obj = ctor->method(ctor->name, &ctor->arglist->args))) {
	    pp_bmc_log_warn("%s: WARNING: incarnation of '%s' skipped",
			    fn, ctor->name);
	    continue;
	}
	
	// add to object table and call back
	pp_hash_set_entry(objs, ctor->name, obj,
			  (void(*)(void*))pp_tp_obj_release);
	va_copy(ap, cb_ap);
	obj_cb(obj, ap);
	va_end(ap);
    continue_outer: ;
    }
    pp_hash_delete(objs);
    return ret;
}

static int tp_def_ctor_add(pp_tp_def_t* tp, const char* id, obj_ctor_t* ctor) {
    if (0 > pp_hash_set_entry_gently(tp->ctorsmap, PP_HASH_PRESERVE, id, ctor,
				     (void(*)(void*))obj_ctor_destroy))
	return PP_ERR;

    if (!CTOR_IS_TEMPLATE(ctor))
	vector_add(tp->objctors, ctor);

    return PP_SUC;
}
	
/*
 * Argument handling
 * ------------------
 */

arg_list_t* arg_list_create() {
    arg_list_t* al = malloc(sizeof(arg_list_t));
    al->no_ph = 0;
    vector_new2(&al->args, 5, sizeof(pp_tp_arg_t),
		(void(*)(void*))pp_tp_arg_cleanup);
    return al;
}

void arg_list_destroy(arg_list_t* al) {
    if (al != NULL) {
	vector_delete(&al->args);
	free(al);
    }
}

/* destroys the argument list and its vector,
 * but doesn't cleanup the arguments            */
void arg_list_destroy_wo_args(arg_list_t* al) {
    if (al != NULL) {
	vector_delete_wo_elems(&al->args);
	free(al);
    }
}

/* adds arg and updates no_ph in arg_list_t */
void arg_list_add(arg_list_t* al, pp_tp_arg_t* arg) {
    vector_add2(&al->args, arg);
    if (arg->type == PP_TP_ARG_PLACEHOLDER)
	al->no_ph++;
}

int pp_tp_arg_scanf(vector_t* args, int idx, pp_strstream_t* errstr,
		    const char *format, ...) {
    char type_char, c;
    char type_arg[5], *a;
    pp_tp_arg_t* arg;
    pp_tp_arg_t  dummy_arg = { .value.intval = 0 };
    void* argp;
    pp_tp_obj_t* objp;
    va_list ap;
    const char* p;
    int i, optionals = 0, s = vector_size(args);

    va_start(ap, format);
    for (i = idx, p = format; (type_char = *p++) != '\0';) {
	if (type_char == '|') { 
	    optionals = 1;
	    continue;
	}

	// get real or dummy arg or leave if no more arguments
	if (i < s) {
	    arg = (pp_tp_arg_t*)vector_get2(args, i);
	    argp = va_arg(ap, void*);
	} else if (optionals) {
	    arg = &dummy_arg;
	    argp = va_arg(ap, void*);
	    switch (type_char) {
	      case 'b': arg->type = PP_TP_ARG_BOOL; break;
	      case 'd': arg->type = PP_TP_ARG_INT; break;
	      case 's':	arg->type = PP_TP_ARG_STRING; break;
	      case 'o': arg->type =  PP_TP_ARG_OBJ; break;
	    }
	} else {
	    break;
	}

	// determine type arg
	a = type_arg;
	if (*p == '<') { // get the type argument
	    p++;
	    while ((c = *p++) != '>' && c != '\0')
		if (a < (type_arg + sizeof(type_arg) - 1)) *a++ = c;
	}
	*a = '\0';

	// cast argument and store in given pointer
	switch (type_char) {
	  case 'b':
	      if (arg->type != PP_TP_ARG_BOOL) goto bailout_type;
	      *(int*)argp = arg->value.intval;
	      break;
	  case 'd':
	      if (arg->type != PP_TP_ARG_INT) goto bailout_type;
	      switch (type_arg[0]) {
	        case '\0': // no argument
		    *(int*)argp = arg->value.intval;
		    break;
	        case 'c': // unsigend char
		    *(unsigned char*)argp = (unsigned char)arg->value.intval;
		    break;
	        case 's': // unsigned short 
		    *(unsigned short*)argp = (unsigned short)arg->value.intval;
		    break;
	        default:
		    type_char = type_arg[0];
		    goto bailout_form;
	      }
	      break;
	  case 's':
	      if (arg->type != PP_TP_ARG_STRING) goto bailout_type;
	      *(char**)argp = arg->value.string;
	      break;
	  case 'o':
	      if (arg->type != PP_TP_ARG_OBJ) goto bailout_type;
	      if (arg == &dummy_arg) {
                  // don't check dummy_arg for obj type, set NULL result
                  *(pp_tp_obj_t**)argp = NULL;
                  break;
              }
              // regular object, check object type
              objp = arg->value.obj;
	      assert(objp != NULL);
	      switch (type_arg[0]) {
		case '\0': // noarg
		    break;
		case 'i':
		    if (!PP_TP_OBJ_IS_TYPE(PP_TP_I2C_COM_DEV, objp))
			goto bailout_obj_type;
		    break;
		case 'h':
		    if (!PP_TP_OBJ_IS_TYPE(PP_TP_I2C_CHIP, objp))
			goto bailout_obj_type;
		    break;
		case '4':
		    if (!PP_TP_OBJ_IS_TYPE(PP_TP_RS485_COM_DEV, objp))
			goto bailout_obj_type;
		    break;
		case '8':
		    if (!PP_TP_OBJ_IS_TYPE(PP_TP_RS485_CHIP, objp))
			goto bailout_obj_type;
		    break;
                case 'g':
                    if (!PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_DEV, objp))
                        goto bailout_obj_type;
                    break;
		case 's':
		    if (!PP_TP_OBJ_IS_TYPE(PP_TP_SENS_DEV, objp))
			goto bailout_obj_type;
		    switch (type_arg[1]) {
		      case '\0': // noarg
			  break;
		      case 'g': 
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_SENS, objp))
			      goto bailout_obj_type;
			  break;
		      case 'm': 
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_MULTI_SENS, objp))
			      goto bailout_obj_type;
			  break;
		      case 'c':
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_COND, objp))
			      goto bailout_obj_type;
			  break;
		      case 'a':
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_SCAN_SENS_DEV, objp))
			      goto bailout_obj_type;
			  break;
		      default:
			  type_char = type_arg[1];
			  goto bailout_form;
		    }
		    break;
		case 'p':
		    if (!PP_TP_OBJ_IS_TYPE(PP_TP_IPMI_SENS, objp))
			goto bailout_obj_type;
		    switch (type_arg[1]) {
		      case '\0': // noarg
			  break;
		      case 't': 
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_THRESH_IPMI_SENS,
						 objp))
			      goto bailout_obj_type;
			  break;
		      case 'd': 
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_DISC_IPMI_SENS, objp))
			      goto bailout_obj_type;
			  break;
		      default:
			  type_char = type_arg[1];
			  goto bailout_form;
		    }
		    break;
	        case 'r':
		    if (!PP_TP_OBJ_IS_TYPE(PP_TP_CFG_SDR, objp))
			goto bailout_obj_type;
		    switch (type_arg[1]) {
		      case '\0': // noarg
			  break;
		      case 't': 
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_THRESH_SDR, objp))
			      goto bailout_obj_type;
			  break;
		      case 'd': 
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_DISC_SDR, objp))
			      goto bailout_obj_type;
			  break;
		      default:
			  type_char = type_arg[1];
			  goto bailout_form;
		    }
		    break;
                case 'a':
                    if (!PP_TP_OBJ_IS_TYPE(PP_TP_ACTOR, objp))
                       goto bailout_obj_type;
		    switch (type_arg[1]) {
		      case '\0': // noarg
			  break;
		      case 'g': 
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_ACT, objp))
			      goto bailout_obj_type;
			  break;
		      case 'm':
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_MULTI_ACT, objp))
			      goto bailout_obj_type;
			  break;
		      case 'p': 
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_PWM_ACT, objp))
			      goto bailout_obj_type;
			  break;
		      case 'x': 
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_SMX_ACT, objp))
			      goto bailout_obj_type;
			  break;
		      default:
			  type_char = type_arg[1];
			  goto bailout_form;
		    }
		    break;
                case 'c':
                    if (!PP_TP_OBJ_IS_TYPE(PP_TP_CTRL, objp))
                        goto bailout_obj_type;
		    switch (type_arg[1]) {
		      case '\0': // noarg
			  break;
		      case 'f': 
			  if (!PP_TP_OBJ_IS_TYPE(PP_TP_FAN_CTRL, objp))
			      goto bailout_obj_type;
			  break;
		      default:
			  type_char = type_arg[1];
			  goto bailout_form;
		    }
                    break;
		case 'm':		    		
		    if (!PP_TP_OBJ_IS_TYPE(PP_TP_MUXED_SCANNER, objp))
			goto bailout_obj_type;
		    break;		    
		default:
		    type_char = type_arg[0];
		    goto bailout_form;
	      }
	      *(pp_tp_obj_t**)argp = objp;
	      break;
	  default:
	      goto bailout_form;
	}
	va_end(ap);
	++i;
    }
    if (type_char != '\0') { // we broke because of arg-vector too short
	if (errstr) pp_strappend(errstr, "too few arguments");
	errno = EINVAL;
    }
    return i - idx;
    
 bailout_obj_type:
    if (errstr)
	pp_strappendf(errstr, "type mismatch arg[%d]:'%c<%s>' != obj-type:"
		      "'0x%08x'", i, type_char, type_arg, objp->type);
    errno = EINVAL;
    return i - idx;
    
 bailout_type:
    if (errstr)
	pp_strappendf(errstr, "type mismatch arg[%d]:'%c' != '%s'",
		      i, type_char, pp_tp_arg_type_to_string(arg));
    errno = EINVAL;
    return i - idx;

 bailout_form:
    if (errstr)
	pp_strappendf(errstr, "invalid format arg[%d]: '%c'", i, type_char);
    errno = EINVAL;
    return i - idx;
}

pp_tp_arg_t pp_tp_arg_copy(pp_tp_arg_t* arg) {
    pp_tp_arg_t narg; 
    narg.type = arg->type;
    switch(arg->type) {
      case PP_TP_ARG_STRING:
      case PP_TP_ARG_SYMBOL:
	  narg.value.string = strdup(arg->value.string);
	  break;
      case PP_TP_ARG_OBJ:
	  narg.value.obj = pp_tp_obj_duplicate(arg->value.obj);
	  break;
      case PP_TP_ARG_INT:
      case PP_TP_ARG_BOOL:
      case PP_TP_ARG_PLACEHOLDER:
	  narg.value.intval = arg->value.intval;
	  break;
      default:
	  assert(0);
    }
    return narg;
}

void pp_tp_arg_cleanup(pp_tp_arg_t* arg) {
    switch(arg->type) {
      case PP_TP_ARG_STRING:
      case PP_TP_ARG_SYMBOL:
	  free(arg->value.string);
	  break;
      case PP_TP_ARG_OBJ:
	  pp_tp_obj_release(arg->value.obj);
	  break;
      case PP_TP_ARG_INT:
      case PP_TP_ARG_BOOL:
      case PP_TP_ARG_PLACEHOLDER:
	  break;  /* nothing to do */
      default:
	  assert(0);
    }
}

void pp_tp_arg_destroy(pp_tp_arg_t* arg) {
    pp_tp_arg_cleanup(arg);
    free(arg);
}

const char* pp_tp_arg_type_to_string(pp_tp_arg_t* arg) {
    const char* r; 
    switch(arg->type) {
      case PP_TP_ARG_STRING:      r = "STRING"; break;
      case PP_TP_ARG_SYMBOL:      r = "SYMBOL"; break;
      case PP_TP_ARG_OBJ:         r = "OBJ"; break;
      case PP_TP_ARG_INT:         r = "INT"; break;
      case PP_TP_ARG_BOOL:        r = "BOOL"; break;
      case PP_TP_ARG_PLACEHOLDER: r = "PLACEHOLDER"; break;
      default:
	  r = NULL;
	  assert(0);
    }
    return r;
}

/*
 * symbol and alias handling
 * -------------------------
 */

int tp_yy_alias_register(pp_tp_def_t* tp, int pos, char* id,
			 pp_tp_arg_t* arg) {
    int ret = PP_SUC;
    pp_tp_arg_t* alarg;

    /* make shallow copy of argument for hashtable, arg is pointer to tempvar*/
    alarg = malloc(sizeof(pp_tp_arg_t));
    *alarg = *arg;
    
    /* add to existing aliases, issue error if duplicate */
    if (0 > pp_hash_set_entry_gently(tp->aliasmap, PP_HASH_PRESERVE, id, alarg,
				     (void(*)(void*))pp_tp_arg_destroy)) {
	pp_tp_yyerrorf(pos, "duplicate alias symbol '%s'", id);
	pp_tp_arg_destroy(alarg);
	ret = PP_ERR;
    }
    return ret;
}

pp_tp_arg_t tp_yy_symbol_arg(pp_tp_def_t* tp, int pos, char* sym) {
    pp_tp_arg_t *arg, ret;
    if (NULL != (arg = pp_hash_get_entry(tp->aliasmap, sym))) {
	ret = pp_tp_arg_copy(arg);
    } else { /* no alias look in ctors, erros will be implicitely printed */
	tp_yy_ctor_inst_find(tp, pos, sym);
	ret.type = PP_TP_ARG_SYMBOL;
	ret.value.string = sym;
    }
    return ret;
}

/*
 * c'tor Handling
 * ---------------
 */

obj_ctor_t* obj_ctor_create(char* name, pp_tp_obj_ctor_func_t method,
			    arg_list_t* arglist) {
    obj_ctor_t *ctor;
    ctor = malloc(sizeof(obj_ctor_t));
    ctor->name = name;
    ctor->method = method;
    ctor->arglist = arglist;
    return ctor;
}

void obj_ctor_destroy(obj_ctor_t* ctor) {
    if (ctor != NULL) {
	arg_list_destroy(ctor->arglist);
	free(ctor->name);
	free(ctor);
    }
}

/* checks, whether given class name is known if not returns a 
 * 'null method ctor', this prevents error propagation during parse */
obj_ctor_t* tp_yy_ctor_create(pp_tp_def_t* tp, int pos, char* class,
			      arg_list_t* arglist) {
    obj_ctor_t *n_ctor, *b_ctor;
    if (NULL != (b_ctor = tp_yy_ctor_find(tp, pos, class)))
	n_ctor = obj_ctor_create(class, b_ctor->method,
				 arg_list_merge(b_ctor->arglist, arglist));
    else
	n_ctor = obj_ctor_create(class, NULL, arglist);
    return n_ctor;
}

obj_ctor_t* tp_yy_ctor_find(pp_tp_def_t* tp, int pos, const char* id) {
    obj_ctor_t* c;
    if (NULL == (c = pp_hash_get_entry(tp->ctorsmap, id))) {
	pp_tp_yyerrorf(pos, "undefined c'tor reference '%s'", id);
    }
    return c;
}

obj_ctor_t* tp_yy_ctor_inst_find(pp_tp_def_t* tp, int pos, const char* id) {
    obj_ctor_t* c;
    if (NULL == (c = pp_hash_get_entry(tp->ctorsmap, id))) {
	pp_tp_yyerrorf(pos, "undefined object reference '%s'", id);
    } else if (CTOR_IS_TEMPLATE(c)) {
	pp_tp_yyerrorf(pos, "reference is not an object '%s'", id);
	c = NULL;
    }
    return c;
}

/* creates an annonymous object ref name, only if ctor is complete and *
 * the parent original ctor can be found.                              */
char* tp_yy_ctor_auto_ref(pp_tp_def_t* tp, int pos, obj_ctor_t* ctor) {
    char* oid;

    if (CTOR_IS_TEMPLATE(ctor)) {
	pp_tp_yyerrorf(pos, "ambiguous object construction");
	oid = NULL;
    } else {
	pp_strstream_t ss = PP_STRSTREAM_INITIALIZER;
	pp_strappendf(&ss, "_%s_#%d", ctor->name, 
		      get_unnamed_cnt(tp, ctor->name));
	oid = pp_strstream_buf(&ss);
    }
    return oid;
}

int tp_yy_ctor_register(pp_tp_def_t* tp, int pos, char* id,
			obj_ctor_t* ctor) {
    int ret = PP_SUC;

    /* set new name in c'tor */
    free(ctor->name);
    ctor->name = id; // allocacted by parser or by auto-naming

    /* and add to tp */
    if (PP_ERR == tp_def_ctor_add(tp, id, ctor)) {
	pp_tp_yyerrorf(pos, "dupliate %s symbol '%s'",
		       (CTOR_IS_TEMPLATE(ctor) ? "c'tor" : "object"), id);
	ret = PP_ERR;
    }
    return ret;
}

/*
 * private functions
 * -------------------
 */

/*
 * Merges two arguments list by copying the base list
 * into a new list and replacing each placeholder with the
 * next argument of the sub list. In case the sublist has more
 * arguments than the base list has placeholders, theses additional
 * args will be appended at the end of the argument list
 *
 * The newly allocated list will be returned, the sub list will be deleted.
 * In case the base list is NULL, the sub list will be returned without change.
 */
static arg_list_t* arg_list_merge(arg_list_t* bl, arg_list_t* sl) {
    arg_list_t* nl;
    int bi, si, bs, ss;
    pp_tp_arg_t* barg;
    
    if (bl == NULL) {
	return sl;
    } else {
	nl = arg_list_create();
    }

    bs = vector_size(&bl->args);
    ss = vector_size(&sl->args);

    // copy base list and replace placeholders
    for (bi = si = 0; bi < bs; ++bi) {
	barg = vector_get2(&bl->args, bi);
	if (si < ss && barg->type == PP_TP_ARG_PLACEHOLDER) {
	    arg_list_add(nl, (pp_tp_arg_t*)vector_get2(&sl->args, si++));
	} else {
	    pp_tp_arg_t bc = pp_tp_arg_copy(barg);
	    arg_list_add(nl, &bc);
	}
    }

    // copy rest of sublist, if there is anything left
    for (; si < ss; ++si) {
	arg_list_add(nl, (pp_tp_arg_t*)vector_get2(&sl->args, si));
    }
    
    arg_list_destroy_wo_args(sl);
    return nl;
}

static int get_unnamed_cnt(pp_tp_def_t* tp, const char* id) {
    int cnt;
    cnt = pp_hash_get_entry_as_int(tp->unnamedmap, id);
    pp_hash_set_entry_as_int(tp->unnamedmap, id, cnt + 1);
    return cnt;
}

