/**
 * tp_delayed_cond.c
 *
 * delays the change of a condition reading by
 * the specified amount of time
 * 
 * (c) 2005 Peppercon AG, 2005/08/10, tbr@peppecon.de
 */

#include <pp/selector.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_delayed_cond.h>

struct pp_tp_delayed_cond_s {
    pp_tp_cond_t base;
    pp_tp_cond_t* cond;
    pp_tp_sensdev_subscriber_t subs;
    int true_delay; // dealay until condition becomes true
    int false_delay; // wait until condition becomes false
    int latch;
    int to_handl;
};

static void delayed_cond_dtor(pp_tp_obj_t* o);
static void delayed_cond_recv_reading(pp_tp_sensdev_subscriber_t* s, 
				      pp_tp_sensdev_t* source, int reading);
static int delayed_cond_to_handler(const int item_id, void* ctx);

pp_tp_obj_t* pp_tp_delayed_cond_ctor(const char* id, vector_t* args) {
    const char* errmsg = "[DelayedCond] '%s' failed (%s)";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_delayed_cond_t* this = NULL;
    pp_tp_cond_t* cond;
    int true_delay;
    int false_delay;

    if (pp_tp_arg_scanf(args, 0, &err, "o<sc>dd",
			&cond, &true_delay, &false_delay) != 3)
    {
	pp_bmc_log_error(errmsg, id, pp_strstream_buf(&err));
	goto bail;
    }

    if (true_delay < 0 || true_delay > 100000 ||
	false_delay < 0 || false_delay > 100000 ||
	(true_delay == 0 && false_delay == 0))
    {
	pp_bmc_log_error(errmsg, id, "delay out of range");
	errno = EINVAL;
	goto bail;
    }

    this = malloc(sizeof(pp_tp_delayed_cond_t));
    pp_tp_cond_init(&this->base, PP_TP_COND, id, delayed_cond_dtor);
    this->true_delay = true_delay;
    this->false_delay = false_delay;
    this->to_handl = 0;
    this->subs.recv_reading = delayed_cond_recv_reading;
    this->cond = pp_tp_cond_duplicate(cond);
    this->base.base.reading = pp_tp_cond_get_reading(this->cond);
    pp_tp_cond_subscribe(this->cond, &this->subs);
    
 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void delayed_cond_dtor(pp_tp_obj_t* o) {
    assert(o);
    pp_tp_delayed_cond_t* this = (pp_tp_delayed_cond_t*)o;
    pp_tp_cond_unsubscribe(this->cond, &this->subs);
    pp_tp_cond_release(this->cond);
    pp_tp_cond_cleanup(&this->base);
    free(o);
}

static int delayed_cond_to_handler(const int item_id UNUSED, void* ctx) {
    pp_tp_delayed_cond_t* this = (pp_tp_delayed_cond_t*)ctx;
    this->to_handl = 0;
    pp_tp_cond_set_reading(&this->base, this->latch);
    return PP_SUC;
}

static void delayed_cond_recv_reading(pp_tp_sensdev_subscriber_t* s, 
				      pp_tp_sensdev_t* source UNUSED, int reading){
    pp_tp_delayed_cond_t* this = PP_TP_INTF_2_OBJ_CAST(s, pp_tp_delayed_cond_t,
						       subs);
    int delay = 0;
    
    if (this->to_handl != 0) {
	pp_select_remove_to(this->to_handl);
	this->to_handl = 0;
    }

    pp_tp_cond_lock(&this->base);
    if (this->base.base.reading != reading) {
	this->latch = reading;
	if (reading > 0) {
	    if (this->true_delay > 0) delay = this->true_delay;
	} else {
	    if (this->false_delay > 0) delay = this->false_delay;
	}
	if (delay > 0) {
	    this->to_handl = pp_select_add_to(delay, 0,
					      delayed_cond_to_handler, this);
	} else {
	    this->base.base.reading = reading;
	    pp_tp_cond_notify_subscribers(&this->base, reading);
	}
    }
    pp_tp_cond_unlock(&this->base);
}

